package ru.yandex.autotests.directapi.darkside.steps;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;

import com.google.gson.Gson;
import com.jayway.awaitility.Duration;
import com.jayway.awaitility.core.ConditionFactory;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BsExportSpecialsParType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BsExportQueueRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.fakebsproxy.dao.FakeBSProxyLogBeanMongoHelper;
import ru.yandex.autotests.direct.scriptrunner.service.clientdata.ClientDataParams;
import ru.yandex.autotests.direct.scriptrunner.service.scripts.ScriptRunResult;
import ru.yandex.autotests.direct.utils.config.DirectTestRunProperties;
import ru.yandex.autotests.directapi.darkside.connection.DarksideConfig;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.FakeAdminService;
import ru.yandex.autotests.directapi.darkside.exceptions.DarkSideException;
import ru.yandex.autotests.directapi.darkside.model.RunBsTransportScriptResponse;
import ru.yandex.autotests.directapi.darkside.model.ScriptParams;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Banner;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Campaign;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.ClientDataStdLogEntry;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.ClientDataStdRequest;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Context;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Phrase;
import ru.yandex.autotests.directapi.darkside.steps.base.BaseJsonRpcSteps;
import ru.yandex.autotests.irt.testutils.allure.AllureUtils;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.autotests.irt.testutils.json.JsonUtils;
import ru.yandex.qatools.allure.annotations.Step;

import static com.jayway.awaitility.Awaitility.with;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;
import static ru.yandex.autotests.irt.testutils.allure.AllureUtils.addJsonAttachment;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

public class TransportSteps extends BaseJsonRpcSteps<FakeAdminService> {

    public static final int MAX_CAMPAIGNS_IN_CAMP_SUMS_REQUEST = 5;
    public static final String SET_ORDER_SUM = "set-order-sum";
    // отправляется быстрым транспортом (транспорт сумм) для валютных и сконвертированных копированием кампаний
    public static final String DEFAULT_CURRENCY_CONVERT_DATE = "20000101";

    public static FakeBSProxyLogBeanMongoHelper fakeBSProxyLogBeanMongoHelper;

    private LogSteps log = LogSteps.getLogger(this.getClass());

    private DirectJooqDbSteps directJooqDbSteps;
    private BsExportSteps bsExportSteps;
    private RunScriptSteps runScriptSteps;
    private ScriptParamsProfiles clientDataParamsProfiles;

    private boolean sendToBs;
    private boolean noAutoBuggy;
    private boolean fakeBs;
    private boolean noLogbroker;
    private String bsUrlData;
    private String bsUrlPrices;
    private boolean checkFullExport;
    private boolean checkCampsOnly;

    public void setCheckFullExport(boolean checkFullExport) {
        this.checkFullExport = checkFullExport;
    }

    @Override
    public void init(DarksideConfig context) {
        super.init(context);

        DirectTestRunProperties properties = DirectTestRunProperties.getInstance();
        this.sendToBs = properties.isDirectTransportBkSendToBS();
        this.fakeBs = properties.isDirectTransportFakeBS();
        this.bsUrlData = properties.getDirectTransportBSUrlData();
        this.bsUrlPrices = properties.getDirectTransportBSUrlData();
        this.noAutoBuggy = properties.isDirectTransportNoAutoBuggy();
        this.noLogbroker = properties.isDirectTransportNoLogbroker();
        this.checkFullExport = properties.isDirectTransportCheckFullLBExport();
        this.checkCampsOnly = properties.isDirectTransportCheckCampsOnly();
        fakeBSProxyLogBeanMongoHelper = new FakeBSProxyLogBeanMongoHelper();
        clientDataParamsProfiles = new ScriptParamsProfiles(properties);
    }

    TransportSteps withDBJooqSteps(DirectJooqDbSteps directJooqDbSteps) {
        this.directJooqDbSteps = directJooqDbSteps;
        return this;
    }

    TransportSteps withRunScriptSteps(RunScriptSteps runScriptSteps) {
        this.runScriptSteps = runScriptSteps;
        return this;
    }

    TransportSteps withBsExportSteps(BsExportSteps bsExportSteps) {
        this.bsExportSteps = bsExportSteps;
        return this;
    }

    private ConditionFactory LOGS_EXPORT = with().timeout(Duration.TWO_MINUTES)
            .and().with().pollDelay(Duration.TWO_SECONDS)
            .and().with().pollInterval(Duration.FIVE_SECONDS)
            .await("лог получен с ручки /bsexport/legacyLogs");

// high-level steps

    @Step("Отправка новой кампании в БК")
    public RunBsTransportScriptResponse sendNewCampaign(int shard, Long cid) {
        return sendCampaign(shard, cid, true, ScriptParamsProfiles.ProfileNames.DEFAULT.toString());
    }

    @Step("Отправка новой кампании. Профиль bsClientData: {2}")
    public RunBsTransportScriptResponse sendNewCampaignWithProfile(int shard, Long cid,
                                                                   ScriptParamsProfiles.ProfileNames profile) {
        return sendCampaign(shard, cid, true, profile.toString());
    }

    @Step("Переотправка кампании в БК")  //use it instead of deprecated sendCampaign()
    public RunBsTransportScriptResponse sendSyncedCampaign(int shard, Long cid) {
        return sendCampaign(shard, cid, false, ScriptParamsProfiles.ProfileNames.DEFAULT.toString());
    }

    @Step("Переотправка кампании в БК. Профиль bsClientData: {2}")
    public RunBsTransportScriptResponse sendSyncedCampaignWithProfile(int shard, Long cid,
                                                                      ScriptParamsProfiles.ProfileNames profile) {
        return sendCampaign(shard, cid, false, profile.toString());
    }

    /**
     * @deprecated use {@link #sendSyncedCampaign} instead
     */
    @Step("Отправка кампании в БК")
    @Deprecated
    public RunBsTransportScriptResponse sendCampaign(int shard, Long cid) {
        return sendCampaign(shard, cid, false, ScriptParamsProfiles.ProfileNames.DEFAULT.toString());
    }

    /**
     * @deprecated use {@link #sendSyncedCampaign} instead
     */
    @Deprecated
    public RunBsTransportScriptResponse sendCampaign(int shard, Integer cid) {
        return sendCampaign(shard, (long) cid, false, ScriptParamsProfiles.ProfileNames.DEFAULT.toString());
    }

    private RunBsTransportScriptResponse sendCampaign(int shard, Long cid, boolean newCamp, String profileName) {
        if (checkFullExport) {
            bsExportSteps.addCampaignToFullExportQueue(cid);
            profileName = ScriptParamsProfiles.ProfileNames.FULL_EXPORT.toString();
        } else {
            runBsExportMasterScript(shard, cid);
        }
        if (checkCampsOnly) {
            profileName = ScriptParamsProfiles.ProfileNames.CAMPS_ONLY.toString();
            directJooqDbSteps.useShard(shard).transportSteps()
                    .putCampInBsExportSpecials(cid, BsExportSpecialsParType.camps_only);
        }
        BsExportQueueRecord bsExportQueueRecord =
                directJooqDbSteps.useShard(shard).transportSteps().getBsExportQueueRecord(cid);
        assumeThat("кампания попала в очередь ppc.bs_export_queue",
                bsExportQueueRecord, notNullValue());

        ClientDataParams params = clientDataParamsProfiles.getProfileClone(profileName)
                .withCid(cid.intValue())
                .withShardId(shard);

        RunBsTransportScriptResponse resp = runBsClientDataScript(params);

        assumeThat("ответ ручки запуска скрипта содержит ссылку на запуск", resp, notNullValue());
        assumeThat("ответ ручки запуска скрипта содержит массив uuid (ссылки на логи экспорта)",
                resp.getUuid(), notNullValue());
        assumeThat("ответ ручки запуска скрипта содержит не пустой массив uuid (ссылки на логи экспорта)",
                Arrays.asList(resp.getUuid()), not(empty()));
        // уже здесь достаем логи запросов и ответов, чтобы они попали в отчет
        Map<String, Map> log = getRawClientDataLogEntriesForCampaign(resp, cid);

        // различные проверки в зависимости от того, новая кампания отправляется или нет
        if (newCamp && !checkFullExport
                // для внутренней рекламы кампании создаются выключенными по дефолту, поэтому второй итерации нет
                && !ScriptParamsProfiles.ProfileNames.INTERNAL_ADS.profileName.equals(profileName)) {
            assumeThat("ответ ручки запуска скрипта содержит ссылки на логи обеих итераций экспорта",
                    Arrays.asList(resp.getUuid()), hasSize(2));
            assumeThat("в логах присутствуют 2 итерации транспорта в БК",
                    log.keySet(), allOf(notNullValue(), hasSize(2)));
        } else {
            assumeThat("в логах присутствует отправка кампании",
                    log.keySet(), allOf(notNullValue(), not(empty())));
        }

        return resp;
    }

    // scripts
    @Step("Запуск скрипта bsExportMaster.pl (shard = {0}, cid = {1})")
    public void runBsExportMasterScript(Integer shard, Long cid) {
        ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.TestScriptRunResponse result;
        ScriptParams scriptRunParams = new ScriptParams()
                .withCids(cid)
                .withShardId(shard)
                .once()
                .noResync();
        result = runScriptSteps.runBsExportMaster(scriptRunParams);
        String resultJson = JsonUtils.toStringLow(result);
        AllureUtils.addJsonAttachment("запуск bsExportMaster.pl", scriptRunParams.toString());
        AllureUtils.addJsonAttachment("результат запуска bsExportMaster.pl: ", resultJson);
        log.info("запуск bsExportMaster.pl: " + scriptRunParams.toString() +
                "\n  результат запуска: " + resultJson);

        if (!result.getCode().equals(0)) {
            throw new DarkSideException(makeErrMsgForScriptRun("bsExportMaster.pl", result));
        }
    }

    @Step("Запуск скрипта bsClientData.pl с ключом --debug-mode (shard = {0}, cid = {1})")
    public RunBsTransportScriptResponse runBsClientDataInDebugMode(Integer shard, Long cid) {

        ClientDataParams params = clientDataParamsProfiles
                .getProfileClone(ScriptParamsProfiles.ProfileNames.DEFAULT.toString())
                .withCid(cid.intValue())
                .withShardId(shard)
                .withDebug(true);

        RunBsTransportScriptResponse resp = runBsClientDataScript(params);

        assumeThat("ответ ручки запуска скрипта содержит ссылку на запуск", resp, notNullValue());
        assumeThat("ответ ручки запуска скрипта содержит массив uuid (ссылки на логи экспорта)",
                resp.getUuid(), notNullValue());
        assumeThat("ответ ручки запуска скрипта содержит не пустой массив uuid (ссылки на логи экспорта)",
                Arrays.asList(resp.getUuid()), not(empty()));

        return resp;
    }

    @Deprecated
    public void runBsExportMasterScript(Integer shard, Integer cid) {
        runBsExportMasterScript(shard, (long) cid);
    }

    @Step("Запуск скрипта bsClientData.pl ({0})")
    public RunBsTransportScriptResponse runBsClientDataScript(ClientDataParams clientDataParams) {
        ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.TestScriptRunResponse result = null;
        result = runScriptSteps.runBsClientData(clientDataParams);
        if (!result.getCode().equals(0)) {
            throw new DarkSideException(makeErrMsgForScriptRun("bsClientData.pl", result));
        }

        if (DirectTestRunProperties.getInstance().isDirectTransportCheckStderr()
                && !result.getStderr().isEmpty()) {
            // Следующий текст ошибки означает, что на стенде не запущен Queryrec (определение языка):
            // GEARMAN_FUNCS(queryrec) error: Gearman error for do(queryrec): [47] _client_do(GEARMAN_TIMEOUT)
            // occured during gearman_client_run_tasks()
            // Запустить его можно следюущей командой: direct-mk service queryrec start
            throw new DarkSideException("Скрипт bsClientData.pl возвращает ошибки в stderr: "
                    + result.getStderr());
        }

        String stdOut = result.getStdout();
        RunBsTransportScriptResponse[] respArr = JsonUtils.getObject(stdOut, RunBsTransportScriptResponse[].class);
        return (respArr != null && respArr.length > 0) ? respArr[0] : null;
    }

    @Step("Запуск скрипта bsClientData.pl (shard = {0}, cid = {1}, profile = default)")
    public RunBsTransportScriptResponse runBsClientDataScript(int shard, Long cid) {
        return runBsClientDataScript(clientDataParamsProfiles.getDefaultProfileClone()
                .withCid(cid.intValue())  //ToDo: поменять в scriptRunner'e тип на Long
                .withShardId(shard));
    }

    @Step("Запуск скрипта bsClientData.pl (shard = {0}, cid = {1}, profile = {2})")
    public RunBsTransportScriptResponse runBsClientDataScript(
            int shard, Long cid, ScriptParamsProfiles.ProfileNames profileName) {
        return runBsClientDataScript(clientDataParamsProfiles.getProfileClone(profileName.toString())
                .withCid(cid.intValue())  //ToDo: поменять в scriptRunner'e тип на Long
                .withShardId(shard));
    }

    // получение кампании из запроса к БК в первой итерации транспорта
    public Campaign getClientDataRequestCampaign(RunBsTransportScriptResponse response, Long cid) {
        return getClientDataRequestCampaign(response, 0, cid);
    }

    @Deprecated //Todo перевести cid на Long
    public Campaign getClientDataRequestCampaign(RunBsTransportScriptResponse response, Integer cid) {
        return getClientDataRequestCampaign(response, 0, (long) cid);
    }

    // получение кампании из ответа от БК в первой итерации транспорта
    public Campaign getClientDataResponseCampaign(RunBsTransportScriptResponse response, Long cid) {
        return getClientDataResponseCampaign(response, 0, cid);
    }

    @Step("bsClientData.pl: получение кампании из запросов к БК по всем итерациям (cid = {2})")
    public List<Campaign> getClientDataRequestCampaignAllIterations(RunBsTransportScriptResponse response, Long cid) {
        List<ClientDataStdLogEntry> logEntries = getClientDataStdLogEntries(response);
        List<Campaign> campaigns = new ArrayList<>(logEntries.size());
        for (ClientDataStdLogEntry logEntry : logEntries) {
            if (logEntry == null || logEntry.getRequest() == null || logEntry.getRequest().getOrders() == null) {
                continue;
            }
            Campaign campaign = logEntry.getRequest().getOrders().get(Long.toString(cid));
            if (campaign != null) {
                campaigns.add(campaign);
            }
        }
        addJsonAttachment("все запросы к БК по данной кампании", JsonUtils.toStringLow(campaigns));
        return campaigns;
    }

    @Step("bsClientData.pl: получение кампании из запроса к БК (итерация = {1}, cid = {2})")
    public Campaign getClientDataRequestCampaign(RunBsTransportScriptResponse response, int iteration, Long cid) {
        ClientDataStdLogEntry logEntry = getClientDataStdLogEntry(response, iteration, true);
        if (isEmptyRequest(logEntry)) {
            return null;
        }
        Campaign campaign = logEntry.getRequest().getOrders().get(Long.toString(cid));
        addJsonAttachment("кампания в запросе к БК", JsonUtils.toStringLow(campaign));
        return campaign;
    }

    @Step("bsClientData.pl: получение кампании из запроса к БК (итерация = {1}, cid = {2})")
    public Campaign getClientDataRequestCampaignWithMdsMeta(RunBsTransportScriptResponse response, int iteration,
                                                            Long cid) {
        ClientDataStdLogEntry logEntry = getClientDataStdLogEntry(response, iteration, false);
        if (isEmptyRequest(logEntry)) {
            return null;
        }
        Campaign campaign = logEntry.getRequest().getOrders().get(Long.toString(cid));
        addJsonAttachment("кампания в запросе к БК", JsonUtils.toStringLow(campaign));
        return campaign;
    }

    private static boolean isEmptyRequest(ClientDataStdLogEntry logEntry) {
        return logEntry == null || logEntry.getRequest() == null || logEntry.getRequest().getOrders() == null;
    }

    @Step("bsClientData.pl: получение кампании из ответа от БК (итерация = {1}, cid = {2})")
    public Campaign getClientDataResponseCampaign(RunBsTransportScriptResponse response, int iteration, Long cid) {
        ClientDataStdLogEntry logEntry = getClientDataStdLogEntry(response, iteration, true);
        if (logEntry == null || logEntry.getResponse() == null || logEntry.getResponse().getOrders() == null) {
            return null;
        }
        Campaign campaign = logEntry.getResponse().getOrders().get(Long.toString(cid));
        addJsonAttachment("кампания в ответе от БК", JsonUtils.toStringLow(campaign));
        return campaign;
    }

    public static Context getBsClientDataContext(Campaign campaign, Long pid) {
        return (campaign != null) ? campaign.getContext(pid) : null;
    }

    public static Banner getBsClientDataBanner(Campaign campaign, Long pid, Long bid) {
        Context context = getBsClientDataContext(campaign, pid);
        return (context != null) ? context.getBanner(bid) : null;
    }

    public static Phrase getBsClientDataPhrase(Campaign campaign, Long pid, Long bidsId) {
        Context context = getBsClientDataContext(campaign, pid);
        return (context != null) ? context.getPhrase(bidsId) : null;
    }

    @Step("bsClientData.pl: получение записей в логе отправки в БК " +
            "очередью std для кампании (все итерации, cid = {1})")
    public Map<String, Map> getRawClientDataLogEntriesForCampaign(
            RunBsTransportScriptResponse bsTransportScriptResponse, Long cid) {
        return getRawClientDataLogEntriesForCampaign(bsTransportScriptResponse, cid, true);
    }

    @Step("bsClientData.pl: получение записей в логе отправки в БК " +
            "очередью std для кампании с сокращенной MdsMeta (все итерации, cid = {1})")
    public Map<String, Map> getRawClientDataLogEntriesForCampaignWithOmittedMdsMeta(
            RunBsTransportScriptResponse bsTransportScriptResponse, Long cid) {
        return getRawClientDataLogEntriesForCampaign(bsTransportScriptResponse, cid, true);
    }

    @SuppressWarnings("unchecked")
    private Map<String, Map> getRawClientDataLogEntriesForCampaign(
            RunBsTransportScriptResponse bsTransportScriptResponse, Long cid, boolean isOmitMdsMeta) {
        String cidStr = String.valueOf(cid);
        Map<String, Map> logs = getRawBsLogs(bsTransportScriptResponse, isOmitMdsMeta);

        for (String uuid : new HashSet<>(logs.keySet())) {

            Map<String, Map> logEntry = logs.get(uuid);
            // filtering request
            Map<String, Map> request = logEntry.get(ClientDataStdLogEntry.REQUEST);
            assumeThat("в логe с uuid:" + uuid + " есть request", request, not(nullValue()));
            Map<String, Map> ordersInRequest = request.get(ClientDataStdRequest.ORDERS);
            assumeThat("в логe с uuid:" + uuid + " есть ORDER", ordersInRequest, not(nullValue()));
            Map<String, Map> orderInRequest = ordersInRequest.get(cidStr);
            ordersInRequest.clear();
            if (orderInRequest != null) {
                ordersInRequest.put(cidStr, orderInRequest);
            } else {
                logs.remove(uuid);
                continue;
            }

            // filtering response
            Map<String, Map> response = logEntry.get(ClientDataStdLogEntry.RESPONSE);
            if (response != null) {
                Map<String, Map> ordersInResponse = response.get(ClientDataStdRequest.ORDERS);
                Map<String, Map> orderInResponse = ordersInResponse.get(cidStr);
                ordersInResponse.clear();
                if (orderInResponse != null) {
                    ordersInResponse.put(cidStr, orderInResponse);
                }
            }
        }
        addJsonAttachment("все запросы/ответы из лога для данной кампании", JsonUtils.toStringLow(logs));
        return logs;
    }

    /**
     * NB! В тесте нужно явно делать assume на ответ этого метода - null/nonNull
     */
    @SuppressWarnings("unchecked")
    @Step("bsClientData.pl: извлечение записи лога одной итерации отправки в БК для заданной кампании: cid = {1}, " +
            "iteration = {2}")
    public Map<String, Map> getRawClientDataLogRequestForCampaign(
            RunBsTransportScriptResponse bsTransportScriptResponse, Long cid, int iteration) {
        if (bsTransportScriptResponse == null) {
            // ничего не отправилось
            return null;
        }

        Map<String, Map> logs = getRawClientDataLogEntriesForCampaign(bsTransportScriptResponse, cid);
        if (iteration > bsTransportScriptResponse.getUuid().length) {
            // не было такой итерации
            return null;
        }
        Map<String, Map> logEntry = logs.get(bsTransportScriptResponse.getUuid()[iteration]);

        if (logEntry == null) {
            return null;
        }
        Map<String, Map> request = logEntry.get(ClientDataStdLogEntry.REQUEST);

        if (request == null) {
            return null;
        }
        Map<String, Map> ordersReq = request.get(ClientDataStdRequest.ORDERS);

        if (ordersReq == null) {
            return null;
        }
        return ordersReq.get(cid.toString());
    }

    /**
     * NB! В тесте нужно явно делать assume на ответ этого метода - null/nonNull
     */
    @Step("bsClientData.pl: извлечение записи лога первой итерации отправки в БК для заданной кампании: cid = {1}")
    public Map<String, Map> getRawClientDataLogRequestForCampaign(
            RunBsTransportScriptResponse bsTransportScriptResponse, Long cid) {
        return getRawClientDataLogRequestForCampaign(bsTransportScriptResponse, cid, 0);
    }

    /**
     * NB! В тесте нужно явно делать assume на ответ этого метода - null/nonNull
     */
    @SuppressWarnings("unchecked")
    @Step("bsClientData.pl: извлечение записи лога одной итерации отправки в БК для заданной группы баннеров: cid = " +
            "{1}, pid = {2}, iteration = {3}")
    public Map<String, Map> getRawClientDataLogRequestForContext(
            RunBsTransportScriptResponse bsTransportScriptResponse, Long cid, Long pid, int iteration) {
        Map<String, Map> orderReq = getRawClientDataLogRequestForCampaign(bsTransportScriptResponse, cid, iteration);
        Map<String, Map> contextsReq = orderReq != null ?
                orderReq.get(Campaign.CONTEXTS) : null;
        return contextsReq != null ?
                contextsReq.get(pid.toString()) : null;
    }

    /**
     * NB! В тесте нужно явно делать assume на ответ этого метода - null/nonNull
     */
    @SuppressWarnings("unchecked")
    @Step("bsClientData.pl: извлечение записи лога одной итерации отправки в БК для заданной группы фраз: cid = {1}, " +
            "pid = {2}")
    public Map<String, Map> getRawClientDataLogRequestForKeyword(
            RunBsTransportScriptResponse bsTransportScriptResponse, Long cid, Long pid) {
        Map<String, Map> pidReq = getRawClientDataLogRequestForContext(bsTransportScriptResponse, cid, pid, 0);
        return pidReq != null ? pidReq.get(Context.PHRASES) : null;
    }

    /**
     * NB! В тесте нужно явно делать assume на ответ этого метода - null/nonNull
     */
    @SuppressWarnings("unchecked")
    @Step("bsClientData.pl: извлечение записи лога одной итерации отправки в БК для заданных условий нацеливания: cid" +
            " = {1}, " +
            "pid = {2}")
    public Map<String, Map> getRawClientDataLogRequestForDynamic(
            RunBsTransportScriptResponse bsTransportScriptResponse, Long cid, Long pid) {
        Map<String, Map> pidReq = getRawClientDataLogRequestForContext(bsTransportScriptResponse, cid, pid, 0);
        return pidReq != null ? pidReq.get(Context.DYNAMICS) : null;
    }

    /**
     * NB! В тесте нужно явно делать assume на ответ этого метода - null/nonNull
     */
    @Step("bsClientData.pl: извлечение записи лога первой итерации отправки в БК для заданной группы баннеров: cid = " +
            "{1}, pid = {2}")
    public Map<String, Map> getRawClientDataLogRequestForContext(
            RunBsTransportScriptResponse bsTransportScriptResponse, Long cid, Long pid) {
        return getRawClientDataLogRequestForContext(bsTransportScriptResponse, cid, pid, 0);
    }


    /**
     * NB! В тесте нужно явно делать assume на ответ этого метода - null/nonNull
     */
    @SuppressWarnings("unchecked")
    @Step("bsClientData.pl: извлечение записи лога одной итерации отправки в БК для заданного баннера: cid = {1}, pid" +
            " = {2}, bid = {3}, iteration = {4}")
    public Map<String, Map> getRawClientDataLogRequestForBanner(
            RunBsTransportScriptResponse bsTransportScriptResponse, Long cid, Long pid, Long bid, int iteration) {
        Map<String, Map> contextReq =
                getRawClientDataLogRequestForContext(bsTransportScriptResponse, cid, pid, iteration);
        Map<String, Map> bannersReq = contextReq != null ?
                contextReq.get(Context.BANNERS) : null;
        return bannersReq != null ?
                bannersReq.get(bid.toString()) : null;
    }

    /**
     * NB! В тесте нужно явно делать assume на ответ этого метода - null/nonNull
     */
    @Step("bsClientData.pl: извлечение записи лога первой итерации отправки в БК для заданного баннера: cid = {1}, " +
            "pid = {2}, bid = {3}")
    public Map<String, Map> getRawClientDataLogRequestForBanner(
            RunBsTransportScriptResponse bsTransportScriptResponse, Long cid, Long pid, Long bid) {
        return getRawClientDataLogRequestForBanner(bsTransportScriptResponse, cid, pid, bid, 0);
    }

    // низкоуровневый метод (каждая запись в логе может содержать несколько кампаний)
    @Step("bsClientData.pl: получение записей в логе отправки в БК очередью std (все итерации, ответ ручки запуска = " +
            "{0})")
    public List<ClientDataStdLogEntry> getClientDataStdLogEntries(RunBsTransportScriptResponse response) {
        return getBsTransportLogEntries(response, ClientDataStdLogEntry.class);
    }

    // низкоуровневый метод (запись в логе может содержать несколько кампаний)
    @Step("bsClientData.pl: получение записи в логе отправки в БК очередью std " +
            "(итерация = {1}, ответ ручки запуска = {0}, хеширование MdsMeta = {2})")
    public ClientDataStdLogEntry getClientDataStdLogEntry(RunBsTransportScriptResponse response, int iteration,
                                                          boolean omitMdsMeta) {
        return getBsTransportLogEntry(response, iteration, omitMdsMeta, ClientDataStdLogEntry.class);
    }

    // низкоуровневый метод
    private <T> T getBsTransportLogEntry(RunBsTransportScriptResponse response,
                                         int iteration, boolean omitMdsMeta,
                                         Class<? extends T> classOfT) {
        if (!(response.getUuid().length > iteration)) {
            throw new DarkSideException(String.format("Попытка получить лог транспорта для %s" +
                    " итерации транспорта, uuid которой отсутствует в ответе ручки запуска скрипта", iteration));
        }
        Map map = getRawBsLogs(response, omitMdsMeta);
        return convertTo(map.get(response.getUuid()[iteration]), classOfT);
    }

    private <T> List<T> getBsTransportLogEntries(RunBsTransportScriptResponse response,
                                                 Class<? extends T> classOfT) {
        Map map = getRawBsLogs(response, true);
        List<T> logEntries = new ArrayList<>(response.getUuid().length);
        for (String uuid : response.getUuid()) {
            logEntries.add(convertTo(map.get(uuid), classOfT));
        }
        return logEntries;
    }

    public Map getRawBsLogs(RunBsTransportScriptResponse response, boolean isOmitMdsMeta) {
        // Подождём, пока в ручку /bsexport/legacyLogs придут логи
        AtomicInteger attemptCounter = new AtomicInteger();
        final AtomicReference<Map> reference = new AtomicReference<>();
        LOGS_EXPORT.until(() -> {
            log.info("Waiting logs in /bsexport/legacyLogs, attempt №" + attemptCounter.incrementAndGet());
            reference.set(bsExportSteps.getBSLogs(response.getUuid(), isOmitMdsMeta));
            // доставка логов асинхронная -> ожидаем появления логов
            // время (количество попыток * интервал) должно быть меньше таймаута, иначе будет false positive
            return reference.get().size() == response.getUuid().length || attemptCounter.intValue() > 18;
        });
        assumeThat("удалось получить логи из ручки /bsexport/legacyLogs. Убедитесь в том, что джоба " +
                        "logs.BsExportLogsLbToYtJob работает. Подробнее про поставку логов БК " +
                        "https://clubs.at.yandex-team.ru/direct-dev/519",
                reference.get().size(), equalTo(response.getUuid().length));

        return reference.get();
    }

    private static <T> T convertTo(Object original, Class<? extends T> returnClass) {
        String gsonObject = JsonUtils.toString(original);
        return new Gson().fromJson(gsonObject, returnClass);
    }


    private static String makeErrMsgForScriptRun(String scriptName, ScriptRunResult result) {
        if (!StringUtils.isEmpty(result.getError())) {
            return "Ошибка выполнения " + scriptName + ": error = " + result.getError();
        }
        if (!result.getResponse().getCode().equals(0)) {
            return "Ошибка выполнения " + scriptName + ": code =" + result.getResponse().getCode();
        }
        return null;
    }

    private static String makeErrMsgForScriptRun(String scriptName,
                                                 ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.TestScriptRunResponse result) {
        if (!StringUtils.isEmpty(result.getStderr())) {
            return "Ошибка выполнения " + scriptName + ": error = " + result.getStderr();
        }
        if (!result.getCode().equals(0)) {
            return "Ошибка выполнения " + scriptName + ": code =" + result.getCode();
        }
        return null;
    }

    @Override
    protected String getServiceName() {
        return FakeAdminService.SERVICE_NAME;
    }

    @Override
    protected Class<FakeAdminService> getClazz() {
        return FakeAdminService.class;
    }
}
