package ru.yandex.autotests.directapi.darkside.steps.base;

import java.io.InputStream;
import java.net.URI;
import java.util.HashMap;

import ru.yandex.autotests.direct.utils.clients.tvm.ServiceTicketProvider;
import ru.yandex.autotests.direct.utils.clients.tvm.ServiceTicketProviderProxy;
import ru.yandex.autotests.direct.utils.config.DirectTestRunProperties;
import ru.yandex.autotests.directapi.darkside.clients.HttpClient;
import ru.yandex.autotests.directapi.darkside.clients.HttpResponse;
import ru.yandex.autotests.directapi.darkside.connection.DarksideConfig;

import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.core.IsNot.not;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by omaz on 19.03.14.
 * Базовый класс для степов методов, работающих по http
 */
public abstract class BaseHttpSteps extends BaseIntApiSteps<HttpClient> {

    @Override
    public void init(DarksideConfig context) {
        super.init(context);

        long tvmDstClientId = DirectTestRunProperties.getInstance().getDirectIntapiTvmClientId();
        ServiceTicketProvider serviceTicketProvider = new ServiceTicketProviderProxy();

        setClient(new HttpClient(context, () -> serviceTicketProvider.getServiceTicket(tvmDstClientId)));
    }

    protected HttpResponse executeMethod(String methodName, HashMap<String, String> params) {
        URI uri = client().buildURIFromParams(methodName, params);
        return client().sendGetRequest(uri);
    }

    protected HttpResponse executeUrlencodedPostMethod(String methodName, HashMap<String, String> params) {
        URI uri = client().buildURIFromParams(methodName, params);
        return client().sendUrlencodedPostRequest(uri);
    }

    public InputStream executeMethodWithRequestAsInputStream(String methodName, HashMap<String, String> params) {
        URI uri = client().buildURIFromParams(methodName, params);
        return client().sendGetRequestWithResponseAsInputStream(uri);
    }

    public String executeMethodNoErrors(String methodName, HashMap<String, String> params) {
        HttpResponse response = executeMethod(methodName, params);
        assertThat("Код ответа не равен 200", response.getCode(), equalTo(200));
        String responseText = response.getMessage();
        assertThat("Текст ответа не должен содержать ошибку",
                responseText.toLowerCase(), not(containsString("\"error\"")));
        return responseText;
    }

    public String executeUrlencodedPostMethodNoErrors(String methodName, HashMap<String, String> params) {
        HttpResponse response = executeUrlencodedPostMethod(methodName, params);
        assertThat("Код ответа не равен 200", response.getCode(), equalTo(200));
        String responseText = response.getMessage();
        assertThat("Текст ответа не должен содержать ошибку",
                responseText.toLowerCase(), not(containsString("\"error\"")));
        return responseText;
    }

    protected HttpResponse executeMethod(String methodName, String params) {
        URI uri = client().buildURIFromParams(methodName, params);
        return client().sendGetRequest(uri);
    }

    public String executeMethodNoErrors(String methodName, String params) {
        HttpResponse response = executeMethod(methodName, params);
        assertThat("Код ответа не равен 200", response.getCode(), equalTo(200));
        String responseText = response.getMessage();
        assertThat("Текст ответа не должен содержать ошибку",
                responseText.toLowerCase(), not(containsString("\"error\"")));
        return responseText;
    }

    /**
     * Выполнение метода с ожиданием http-ответа с заданным кодом
     *
     * @param code - http-код (например 404 или 200)
     */
    public String executeMethodExpectReturnCode(String methodName, HashMap<String, String> params, Integer code) {
        HttpResponse response = executeMethod(methodName, params);
        assertThat("Ответ содержит не тот код, который ожидался", response.getCode(), equalTo(code));
        return response.getMessage();
    }

    public String executeUrlencodedPostMethodExpectReturnCode(String methodName, HashMap<String, String> params,
                                                              Integer code) {
        HttpResponse response = executeUrlencodedPostMethod(methodName, params);
        assertThat("Ответ содержит не тот код, который ожидался", response.getCode(), equalTo(code));
        return response.getMessage();
    }

    /**
     * Выполнение метода с ожиданием http-ошибки с заданным кодом и текстом в ответе
     *
     * @param errorCode    - http-код ошибки (например, 404)
     * @param errorMessage - текст ошибки
     */
    public void executeMethodExpectError(String methodName, HashMap<String, String> params, Integer errorCode,
                                         String errorMessage) {
        HttpResponse response = executeMethod(methodName, params);
        assertThat("Ответ содержит не тот код, который ожидался", response.getCode(), equalTo(errorCode));
        assertThat("Ответ содержит не то сообщение, которое ожидалось",
                response.getMessage(),
                containsString(errorMessage));
    }

    /**
     * Выполнение метода с ожиданием кода 200, но ошибки в сообщении
     *
     * @param errorMessage - текст ошибки
     */
    public void executeMethodExpectError(String methodName, HashMap<String, String> params, String errorMessage) {
        executeMethodExpectError(methodName, params, 200, errorMessage);
    }
}
