package ru.yandex.autotests.directapi.darkside.steps.fakesteps;

import org.apache.commons.lang.math.RandomUtils;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.BannerFakeInfo;
import ru.yandex.autotests.directapi.darkside.model.ModerationFlag;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.qatools.allure.annotations.Step;

import java.util.Arrays;
import java.util.concurrent.Callable;
import java.util.stream.Collectors;

import static ru.yandex.autotests.directapi.darkside.model.Status.NO;
import static ru.yandex.autotests.directapi.darkside.model.Status.YES;

public class BannersFakeSteps extends BaseFakeSteps {

    /**
     * Выставляет результат модерации текста объявления.
     * ppc.banners:statusModerate - enum('new', 'sent', 'sending', 'ready', 'yes', 'no')
     * <p>
     * На отправку в БК не влияет. Требуется выяснить, на что он влияет (!).
     *
     * @param bid    bid
     * @param status статус enum('new', 'sent', 'sending', 'ready', 'yes', 'no')
     */
    @Step("[FakeBannerParams]: установить значение statusModerate = {1} для баннера bid = {0}")
    public void setStatusModerate(Long bid, String status) {
        BannerFakeInfo banner = new BannerFakeInfo();
        banner.setBid(bid);
        banner.setStatusModerate(status);
        updateBannerParams(banner);
    }

    /**
     * Выставляет результат модерации сайтлинков.
     * ppc.banners:statusSitelinksModerate - enum('new', 'sent', 'sending', 'ready', 'yes', 'no')
     * <p>
     * Если в объявлении есть сайтлинки, то они должны пройти модерацию, для того чтобы баннер отправился в БК.
     * (важен именно факт модерации, а не результат - приняты или отклонены)
     * Если сайтлинки приняты на модерации, то баннер будет отправлен с ними, а если отклонены, то без них.
     *
     * @param bid    bid
     * @param status статус enum('new', 'sent', 'sending', 'ready', 'yes', 'no')
     */
    @Step("[FakeBannerParams]: установить значение statusSitelinksModerate = {1} для баннера bid = {0}")
    public void setStatusSitelinksModerate(Long bid, String status) {
        BannerFakeInfo banner = new BannerFakeInfo();
        banner.setBid(bid);
        banner.setStatusSitelinksModerate(status);
        updateBannerParams(banner);
    }

    /**
     * Выставляет результат модерации визитки для указанного баннера
     * (этот статус - свойство именно баннера, а не самой визитки)
     * ppc.banners:phoneflag - enum('new', 'sent', 'sending', 'ready', 'yes', 'no')
     * <p>
     * Статус и результат модерации визитки не влияют на отправку баннера в БК.
     * Если визитка для данного баннера прошла модерацию и принята, то она будет отправлена в БК,
     * а если нет, то баннер может быть отправлен без нее.
     *
     * @param bid    bid
     * @param status статус enum('new', 'sent', 'sending', 'ready', 'yes', 'no')
     */
    @Step("[FakeBannerParams]: установить значение phoneflag = {1} (статус модерации визитки) для баннера bid = {0}")
    public void setPhoneflag(Long bid, String status) {
        BannerFakeInfo banner = new BannerFakeInfo();
        banner.setBid(bid);
        banner.setPhoneFlag(status);
        updateBannerParams(banner);
    }

    /**
     * Выставляет флаг возможности отправки в БК
     * ppc.banners:statusPostModerate - enum('new', 'ready', 'sent', 'yes', 'no', 'rejected')
     * <p>
     * флаг, означающий можно ли отправлять баннер в БК
     * Смысл значений:
     * New - значение по-умолчанию
     * No - отправлять нельзя (кроме случая остановки активного баннера)
     * Yes - отправлять можно
     * Rejected - нужно отправить в БК с флагом остановки объявления
     * Ready, Sent - (вроде как статусы отправки на модерацию, т.е. промежуточные)
     *
     * @param bid    bid
     * @param status статус enum('new', 'ready', 'sent', 'yes', 'no', 'rejected')
     */
    @Step("[FakeBannerParams]: установить значение statusPostModerate = {1} для баннера bid = {0}")
    public void setStatusPostModerate(Long bid, String status) {
        BannerFakeInfo banner = new BannerFakeInfo();
        banner.setBid(bid);
        banner.setStatusPostModerate(status);
        updateBannerParams(banner);
    }

    @Step("[FakeBannerParams]: выставить всем статусам модерации баннера bid = {0} положительные значения")
    public void makeBannerFullyModerated(Long bid) {
        String status = Status.YES;

        BannerFakeInfo banner = new BannerFakeInfo();
        banner.setBid(bid);
        banner.setStatusModerate(status);
        banner.setStatusSitelinksModerate(status);
        banner.setPhoneFlag(status);
        banner.setStatusPostModerate(status);
        banner.setImageadStatusModerate(status);
        updateBannerParams(banner);
    }

    @Step("[FakeBannerParams]: установить значение statusBsSynced = {1} для баннера bid = {0}")
    public void setStatusBsSynced(Long bid, String status) {
        BannerFakeInfo banner = new BannerFakeInfo();
        banner.setBid(bid);
        banner.setStatusBsSynced(status);
        updateBannerParams(banner);
    }

    @Step("[FakeBannerParams]: установить значение statusShow = {1} для баннера bid = {0}")
    public void setStatusShow(Long bid, String status) {
        BannerFakeInfo banner = new BannerFakeInfo();
        banner.setBid(bid);
        banner.setStatusShow(status);
        updateBannerParams(banner);
    }

    @Step("[FakeBannerParams]: Принять объявления на модерации: bannerIDs = {0}")
    public void makeBannersModerated(Long... bannerIDS) {
        for (Long bannerID : bannerIDS) {
            setBannerStatusModerate(bannerID, Status.YES);
        }
    }

    @Step("[FakeBannerParams]: Отклонить объявления на модерации: bannerIDs = {0}")
    public void makeBannersDeclined(Long... bannerIDS) {
        for (Long bannerID : bannerIDS) {
            setBannerStatusModerate(bannerID, Status.NO);
        }
    }

    @Step("[FakeBannerParams]: Сделать объявления активными: bannerIDs = {0}")
    public void makeBannerActive(Long... bannerIDS) {
        for (Long bannerID : bannerIDS) {
            BannerFakeInfo banner = new BannerFakeInfo();
            banner.setBid(bannerID);
            banner.setStatusShow(YES);
            banner.setStatusActive(YES);
            banner.setStatusModerate(YES);
            banner.setPhoneFlag(YES);
            banner.setStatusBsSynced(YES);
            banner.setBannerID(String.valueOf(
                    Integer.MAX_VALUE - Math.abs(ru.yandex.autotests.irt.testutils.RandomUtils.getNextInt(2000000))));

            updateBannerParams(banner);
        }
    }

    @Step("[FakeBannerParams]: Сделать объявления активными и полностью промодерированными: bannerIDs = {0}")
    public void makeBannersActiveAndFullyModerated(Long... bannerIDS) {
        for (Long bannerID : bannerIDS) {
            BannerFakeInfo banner = new BannerFakeInfo();
            banner.setBid(bannerID);
            banner.setStatusShow(YES);
            banner.setStatusActive(YES);
            banner.setStatusModerate(YES);
            banner.setPhoneFlag(YES);
            banner.setStatusSitelinksModerate(YES);
            banner.setStatusPostModerate(YES);
            banner.setImageadStatusModerate(YES);
            banner.setStatusBsSynced(YES);
            banner.setBannerID(String.valueOf(
                    Integer.MAX_VALUE - Math.abs(ru.yandex.autotests.irt.testutils.RandomUtils.getNextInt(2000000))));

            updateBannerParams(banner);
        }
    }
    @Step("Останавливаем объявления для удаления")
    public void makeBannerReadyForDelete(Long... bannerIDs) {
        for (Long bannerID : bannerIDs) {
            BannerFakeInfo banner = new BannerFakeInfo();
            banner.setBid(bannerID);
            banner.setStatusModerate(YES);
            banner.setpStatusPostModerate(YES);
            banner.setPhoneFlag(YES);
            banner.setStatusSitelinksModerate(YES);
            banner.setpStatusModerate(YES);
            banner.setpStatusPostModerate(YES);
            banner.setStatusShow(NO);
            banner.setStatusActive(NO);
            banner.setStatusBsSynced(YES);
            banner.setpStatusBsSynced(YES);
            banner.setBannerID("");
            updateBannerParams(banner);
        }
    }

    @Step("[FakeBannerParams]: Задать флаги для баннера: bannerID = {0}")
    public void setBannerFlags(Long bid, ModerationFlag... flags) {
        String serialized = Arrays.stream(flags).map(ModerationFlag::getDbValue).collect(Collectors.joining(","));
        BannerFakeInfo banner = new BannerFakeInfo();
        banner.setBid(bid);
        banner.setFlags(serialized);
        banner.setStatusBsSynced(NO);
        updateBannerParams(banner);
    }

    @Step("[FakeGetBannerParams]: bannerID = {0}")
    public BannerFakeInfo getBannerParams(Long bannerID) {
        return service().FakeGetBannerParams(bannerID);
    }

    @Step("[FakeBannerParams]: updateBannerParams")
    public void updateBannerParams(BannerFakeInfo... bannerFakeInfos) {
        for (BannerFakeInfo bannerFakeInfo : bannerFakeInfos) {
            updateService().FakeBannerParams(bannerFakeInfo);
        }
    }

    public Long setBannerRandomFakeBannerID(Long bannerID) {
        long fakeBannerID = 500000000 + Math.abs(RandomUtils.nextInt(500000000));
        setBannerRandomFakeBannerID(bannerID, fakeBannerID);
        return fakeBannerID;
    }

    public void setBannerRandomFakeBannerID(Long bannerID, long fakeBannerID) {
        BannerFakeInfo bannerFakeInfo = getBannerParams(bannerID);
        bannerFakeInfo.setBannerID(String.valueOf(fakeBannerID));
        updateBannerParams(bannerFakeInfo);
    }

    public void setBannerFakeImageBannerIDRandom(Long bannerID) {
        Long fakeBannerID = 500000000L + Math.abs(RandomUtils.nextInt(500000000));
        BannerFakeInfo bannerFakeInfo = getBannerParams(bannerID);
        bannerFakeInfo.setImageBannerID(fakeBannerID);
        updateBannerParams(bannerFakeInfo);
    }

    @Step("[FAKE_BANNER_PARAMS]: установить значение statusModerate = {1} для bannerID = {0}")
    public void setBannerFakeStatusModerate(Long bannerID, String status) {
        BannerFakeInfo bannerFakeInfo = getBannerParams(bannerID);
        bannerFakeInfo.setStatusModerate(status);
        updateBannerParams(bannerFakeInfo);
    }

    @Step("[FAKE_BANNER_PARAMS]: установить значение statusSitelinksModerate = {1} для bannerID = {0}")
    public void setBannerFakeStatusSitelinksModerate(Long bannerID, String status) {
        BannerFakeInfo bannerFakeInfo = getBannerParams(bannerID);
        bannerFakeInfo.setStatusSitelinksModerate(status);
        updateBannerParams(bannerFakeInfo);
    }

    @Step("[FAKE_BANNER_PARAMS]: установить значение phoneFlag = {1} для bannerID = {0}")
    public void setBannerFakePhoneFlag(Long bannerID, String status) {
        BannerFakeInfo bannerFakeInfo = getBannerParams(bannerID);
        bannerFakeInfo.setPhoneFlag(status);
        updateBannerParams(bannerFakeInfo);
    }

    @Step("[FAKE_BANNER_PARAMS]: установить значение image_statusModerate = {1} для bannerID = {0}")
    public void setImageStatusModerate(Long bannerID, String status) {
        BannerFakeInfo bannerFakeInfo = getBannerParams(bannerID);
        bannerFakeInfo.setImageStatusModerate(status);
        updateBannerParams(bannerFakeInfo);
    }

    @Step("Фейковая модификация объявления {0} со статусом = {1}")
    private void setBannerStatusModerate(Long bannerID, String status) {
        BannerFakeInfo banner = new BannerFakeInfo();
        banner.setBid(bannerID);
        banner.setStatusShow(status);
        banner.setStatusModerate(status);
        banner.setpStatusPostModerate(status);
        banner.setPhoneFlag(status);
        banner.setStatusBsSynced(YES);
        banner.setStatusSitelinksModerate(status);
        banner.setpStatusModerate(status);

        updateBannerParams(banner);
    }

    @Step("[FakeBannerParams]: установить значение LastChange = {1} для баннера bid = {0}")
    public void setLastChange(Long bid, String lastChange) {
        BannerFakeInfo banner = new BannerFakeInfo();
        banner.setBid(bid);
        banner.setLastChange(lastChange);
        updateBannerParams(banner);
    }

    @Step("Останавливаем объявления")
    public void makeBannersStopped(Long... bannerIDs) {
        for (Long bannerID : bannerIDs) {
            BannerFakeInfo banner = new BannerFakeInfo();
            banner.setBid(bannerID);
            banner.setStatusModerate(YES);
            banner.setpStatusPostModerate(YES);
            banner.setPhoneFlag(YES);
            banner.setStatusSitelinksModerate(YES);
            banner.setpStatusModerate(YES);
            banner.setpStatusPostModerate(YES);
            banner.setStatusShow(NO);
            banner.setStatusActive(NO);
            banner.setStatusBsSynced(YES);
            banner.setpStatusBsSynced(YES);
            banner.setBannerID(String.valueOf(
                    Integer.MAX_VALUE - Math.abs(ru.yandex.autotests.irt.testutils.RandomUtils.getNextInt(2000000))));
            updateBannerParams(banner);
        }
    }

    @Step("[FakeBannerParams]: установить значение statusActive = {1} для баннера bid = {0}")
    public void setStatusActive(Long bid, String status) {
        BannerFakeInfo banner = new BannerFakeInfo();
        banner.setBid(bid);
        banner.setStatusActive(status);
        updateBannerParams(banner);
    }

    public Callable<Boolean> statusModerateIs(Long bid, String statusModerate) {
        return () -> getBannerParams(bid).getStatusModerate().equals(statusModerate);
    }

    public Callable<Boolean> statusBsSyncedIs(Long bid, String statusBsSynced) {
        return () -> getBannerParams(bid).getStatusBsSynced().equals(statusBsSynced);
    }
}
