package ru.yandex.autotests.directapi.darkside.steps.fakesteps;

import java.util.Arrays;
import java.util.Collection;
import java.util.concurrent.Callable;
import java.util.stream.Stream;

import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsPlatform;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.CampaignFakeInfo;
import ru.yandex.autotests.directapi.darkside.model.CampaignsType;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.darkside.model.StrategyName;
import ru.yandex.autotests.irt.testutils.RandomUtils;
import ru.yandex.qatools.allure.annotations.Step;

public class CampaignFakeSteps extends BaseFakeSteps {

    /**
     * Выставляет результат модерации кампании.
     * ppc.campaigns:statusModerate - enum('new', 'ready', 'sent', 'yes', 'no', 'mediaplan')
     * <p>
     * New - когда кампания ни разу не была отправлена на модерацию (оферту не показывали)
     * Yes - когда хотя бы один баннер в кампании принят на модерации
     * No - когда все баннеры в кампании отклонены
     * Ready, Sending, Sent - во время отправки кампании на модерацию
     * Mediaplan - ?
     * <p>
     * Кампания может отправляться на модерацию либо первый раз,
     * либо когда отправлены все баннеры в легком интерфейсе
     * (именно запись из таблицы campaigns; баннеры могут сколь угодно часто отправляться).
     * <p>
     * На отправку в БК не влияет. Требуется выяснить, на что он влияет (!).
     *
     * @param cid    cid
     * @param status статус enum('new', 'ready', 'sent', 'yes', 'no', 'mediaplan')
     */
    @Step("[FakeCampaignParams]: установить значение statusModerate = {1} для кампании cid = {0}")
    public void setStatusModerate(int cid, String status) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(cid);
        campaign.setStatusModerate(status);
        fakeCampaignParams(campaign);
    }

    public void setStatusModerate(Long cid, String status) {
        setStatusModerate(cid.intValue(), status);
    }

    /**
     * Выставляет результат пост-модерации кампании.
     * ppc.camp_options:statusPostModerate - enum('new', 'yes', 'no', 'accepted')
     * <p>
     * New - пока не модерировалась
     * Yes - кампания принята мгновенной модерацией,
     * её можно оплачивать специальными способами,
     * но пока нельзя запускать
     * No - кампанию отклонили на модерации
     * Accepted - принята на пост-модерации, можно начинать крутить. Из этого статуса кампания не выходит.
     * <p>
     * Непосредственно влияет на возможность отправки в БК.
     *
     * @param cid    cid
     * @param status статус enum('new', 'yes', 'no', 'accepted')
     */
    @Step("[FakeCampaignParams]: установить значение statusPostModerate = {1} для кампании cid = {0}")
    public void setStatusPostModerate(int cid, String status) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(cid);
        campaign.setStatusPostModerate(status);
        fakeCampaignParams(campaign);
    }

    public void setStatusPostModerate(Long cid, String status) {
        setStatusPostModerate(cid.intValue(), status);
    }

    @Step("[FakeCampaignParams]: сделать кампанию полностью промодерированной, cid = {0}")
    public void makeCampaignFullyModerated(int cid) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(cid);
        campaign.setStatusModerate(Status.YES);
        campaign.setStatusPostModerate(Status.ACCEPTED);
        fakeCampaignParams(campaign);
    }

    public void makeCampaignFullyModerated(Long cid) {
        makeCampaignFullyModerated(cid.intValue());
    }

    @Step("Задать число изменений ДБ за последние сутки: campaignID = {0}, count = {1}")
    public void setDayBudgetChangeCount(int campaignID, int count) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(campaignID);
        campaign.setDayBudgetDailyChangeCount(count);
        fakeCampaignParams(campaign);
    }
    @Step("Задать статус модерации кампании: campaignID = {0}, status = {1}")
    public void setCampaignStatusModerate(int campaignID, String status) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(campaignID);
        campaign.setStatusModerate(status);
        campaign.setStatusPostModerate(Status.ACCEPTED);
        fakeCampaignParams(campaign);
    }

    @Step("Пополняем баланс кампании {0} (фиктивно) на {1}")
    public void setCampaignSum(int campaignID, float balance) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(campaignID);
        campaign.setSum(balance);
        campaign.setSumSpent(0f);
        campaign.setSumToPay(0f);
        campaign.setSumLast(0f);
        fakeCampaignParams(campaign);
    }

    public void setCampaignSum(Long campaignID, float balance) {
        setCampaignSum(campaignID.intValue(), balance);
    }

    @Step("Установить SumSpent = {1} для кампании {0}")
    public void setSumSpent(int campaignID, float spent) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(campaignID);
        campaign.setSumSpent(spent);
        fakeCampaignParams(campaign);
    }

    public void setSumSpent(Long campaignID, float spent) {
        setSumSpent(campaignID.intValue(), spent);
    }

    @Step("Фейковая модификация кампании {0}, statusModerate = {1}")
    public void setCampaignStatusRejectedModerate(int campaignID, String status) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(campaignID);
        campaign.setStatusModerate(status);
        //после принятия статус statusPostModerate нельзя менять https://st.yandex-team.ru/DIRECT-31406#1413374566000
        if (!Status.ACCEPTED.equalsIgnoreCase(fakeGetCampaignParams(campaignID).getStatusPostModerate())) {
            campaign.setStatusPostModerate(status);
        }
        campaign.setStatusActive(status);
        fakeCampaignParams(campaign);
    }

    public void makeCampaignModerated(int campaignID) {
        setCampaignStatusModerate(campaignID, Status.YES);
    }

    public void makeCampaignModerated(Long campaignID) {
        makeCampaignModerated(campaignID.intValue());
    }

    public void makeCampaignMobileType(int campaignID) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(campaignID);
        campaign.setType(CampaignsType.MOBILE_CONTENT.toString());
        fakeCampaignParams(campaign);
    }

    public void makeCampaignMcbType(int campaignID) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(campaignID);
        campaign.setType(CampaignsType.MCB.toString());
        fakeCampaignParams(campaign);
    }

    public void makeCampaignWalletType(int campaignID) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(campaignID);
        campaign.setType(CampaignsType.WALLET.toString());
        fakeCampaignParams(campaign);
    }

    public void makeCampaignPerformanceType(int campaignID) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(campaignID);
        campaign.setType(CampaignsType.PERFORMANCE.toString());
        fakeCampaignParams(campaign);
    }

    public void makeCampaignReadyForDelete(Integer... campaignIDs) {
        makeCampaignReadyForDelete(Arrays.stream(campaignIDs).map(Integer::longValue).toArray(Long[]::new));
    }

    private void makeSingleCampaignReadyForDelete(long campaigId) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID((int) campaigId);
        campaign.setSum(0f);
        campaign.setSumSpent(0f);
        campaign.setSumToPay(0f);
        campaign.setSumLast(0f);
        campaign.setShows(0);
        campaign.setClicks(0);
        campaign.setOrderID("");
        campaign.setStatusActive(Status.NO);
        campaign.setStatusBsSynced(Status.NO);
        campaign.setStatusModerate(Status.NO);
        campaign.setArchived(Status.NO);
        campaign.setCurrencyConverted(Status.NO);
        campaign.setType(CampaignsType.TEXT.toString());
        campaign.setStrategy(StrategyName.DIFFERENT_PLACES.toString());
        fakeCampaignParams(campaign);
    }

    @Step("Сбрасываем параметры кампании, чтобы удалить")
    public void makeCampaignReadyForDelete(Long... campaignIDs) {
        for (long campaignID : campaignIDs) {
            makeSingleCampaignReadyForDelete(campaignID);
        }
    }

    @Step("Сбрасываем параметры кампании, чтобы удалить")
    public void makeCampaignReadyForDelete(Collection<Long> campaignIDs) {
        for (long campaignID : campaignIDs) {
            makeSingleCampaignReadyForDelete(campaignID);
        }
    }

    // этого не достаточно! кампания должна быть остановлена более часа назад, поле ppc.camp_options:stopTime
    @Step("Подготовить кампанию к архивации")
    public void makeCampaignReadyForArchive(Integer... campaignIDs) {
        for (int campaignID : campaignIDs) {
            CampaignFakeInfo campaign = new CampaignFakeInfo();
            campaign.setCampaignID(campaignID);
            campaign.setStatusModerate(Status.YES);
            campaign.setStatusShow(Status.NO);
            campaign.setStatusActive(Status.NO);
            campaign.setStatusBsSynced(Status.NO);
            campaign.setSum(0f);
            fakeCampaignParams(campaign);
        }
    }

    public void makeCampaignReadyForArchive(Long... campaignIDs) {
        makeCampaignReadyForArchive(Stream.of(campaignIDs).mapToInt(Long::intValue).boxed().toArray(Integer[]::new));
    }

    @Step("Запускаем кампанию {0}")
    public void makeCampaignActive(int campaignID) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(campaignID);
        campaign.setStatusModerate(Status.YES);
        campaign.setStatusPostModerate(Status.ACCEPTED);
        campaign.setLastShowTime(DateTime.now().minusDays(10).toString());
        campaign.setStatusShow(Status.YES);
        campaign.setStatusActive(Status.YES);
        campaign.setStatusBsSynced(Status.YES);
        campaign.setOrderID(String.valueOf(Integer.MAX_VALUE - Math.abs(RandomUtils.getNextInt(2000000))));
        fakeCampaignParams(campaign);
    }

    public void makeCampaignActive(Long campaignID) {
        makeCampaignActive(campaignID.intValue());
    }

    @Step("Останавливаем кампанию {0}")
    public void makeCampaignStopped(int campaignID) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(campaignID);
        campaign.setStatusModerate(Status.YES);
        campaign.setStatusShow(Status.NO);
        campaign.setStatusActive(Status.NO);
        campaign.setStatusBsSynced(Status.NO);
        fakeCampaignParams(campaign);
    }

    public void makeCampaignStopped(Long campaignID) {
        makeCampaignStopped(campaignID.intValue());
    }


    @Step("Установить orderID = {1} для кампании campaignID = {0}")
    public void setOrderID(int campaignID, int orderID) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(campaignID);
        campaign.setOrderID(String.valueOf(orderID));
        fakeCampaignParams(campaign);
    }

    @Step("Установить случайный orderID для кампании campaignID = {0}")
    public int setRandomOrderID(int campaignID) {
        int orderID = Integer.MAX_VALUE - Math.abs(RandomUtils.getNextInt(2000000));
        setOrderID(campaignID, orderID);
        return orderID;
    }

    public int setRandomOrderID(Long campaignID) {
        return setRandomOrderID(campaignID.intValue());
    }

    @Step("Установить statusActive = {1} для кампании campaignID = {0}")
    public void setStatusActive(int campaignID, String statusActive) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(campaignID);
        campaign.setStatusActive(statusActive);
        fakeCampaignParams(campaign);
    }

    public void setStatusActive(Long campaignID, String statusActive) {
        setStatusActive(campaignID.intValue(), statusActive);
    }

    @Step("Установить statusBsSynced = {1} для кампании campaignID = {0}")
    public void setStatusBsSynced(int campaignID, String statusBsSynced) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(campaignID);
        campaign.setStatusBsSynced(statusBsSynced);
        fakeCampaignParams(campaign);
    }

    public void setStatusBsSynced(Long campaignID, String statusBsSynced) {
        setStatusBsSynced(campaignID.intValue(), statusBsSynced);
    }

    @Step("Установить statusShow = {1} для кампании campaignID = {0}")
    public void setStatusShow(int campaignID, String statusShow) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(campaignID);
        campaign.setStatusShow(statusShow);
        fakeCampaignParams(campaign);
    }

    public void setStatusShow(Long campaignID, String statusShow) {
        setStatusShow(campaignID.intValue(), statusShow);
    }

    @Step("Установить bsSynced {1} для кампании campaignID = {0}")
    public void setBSSynced(int campaignID, boolean bsSynced) {
        String statusBsSynced = bsSynced ? "Yes" : "No";
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(campaignID);
        campaign.setStatusBsSynced(statusBsSynced);
        fakeCampaignParams(campaign);
    }

    public void setBSSynced(Long campaignID, boolean bsSynced) {
        setBSSynced(campaignID.intValue(), bsSynced);
    }

    @Deprecated
    @Step("Установить type {1} для кампании campaignID = {0}")
    public int setType(int campaignID, String type) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(campaignID);
        campaign.setType(type);
        fakeCampaignParams(campaign);
        return campaignID;
    }

    @Step("Установить type {1} для кампании campaignID = {0}")
    public int setType(int campaignID, CampaignsType type) {
        CampaignFakeInfo campaign = fakeGetCampaignParams(campaignID);
        campaign.setType(type.value());
        if (type == CampaignsType.CPM_BANNER) {
            // для кампаний типа CPM_BANNER показы могут быть только в сети
            // иначе получим 500-ку в интерфейсе при валидации
            campaign.setPlatform(CampaignsPlatform.context.getLiteral());
        }

        if (type == CampaignsType.PERFORMANCE) {
            campaign.setStrategy(StrategyName.DEFAULT.toString());
        }

        fakeCampaignParams(campaign);
        return campaignID;
    }

    public int setType(Long campaignID, CampaignsType type) {
        return setType(campaignID.intValue(), type);
    }

    @Step("Установить валюту {1} для кампании campaignID = {0}")
    public int setCurrency(int campaignID, Currency currency) {
        CampaignFakeInfo campaign = fakeGetCampaignParams(campaignID);
        campaign.setCurrency(currency.value());
        fakeCampaignParams(campaign);
        return campaignID;
    }

    @Step("Установить стратегию {1} для кампании campaignID = {0}")
    public int setStrategyName(int campaignID, StrategyName strategyName) {
        CampaignFakeInfo campaign = fakeGetCampaignParams(campaignID);
        campaign.setStrategyName(strategyName.toString());
        fakeCampaignParams(campaign);

        return campaignID;
    }

    @Step("Установить данные стратегии  {1} для кампании campaignID = {0}")
    public int setStrategyData(int campaignID, String strategyData) {
        CampaignFakeInfo campaign = fakeGetCampaignParams(campaignID);
        campaign.setStrategyData(strategyData);
        fakeCampaignParams(campaign);

        return campaignID;
    }

    @Step("Установить lastShowTime {1} для кампании campaignID = {0}")
    public void setLastShowTime(int campaignID, DateTime dateTime) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(campaignID);
        campaign.setLastShowTime(dateTime.toString());
        fakeCampaignParams(campaign);
    }

    /**
     * Этот метод перезатирает значения в базе, как минимум для поля camp_options.sms_flags
     * Лучше использовать jooqDbSteps.campaignsSteps().setFinishTime(cid, finishTime);
     *
     * @see ru.yandex.autotests.direct.db.steps.CampaignsSteps
     */
    @Deprecated
    @Step("Установить finishTime {1} для кампании campaignID = {0}")
    public void setFinishTime(int campaignID, String dateTime) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(campaignID);
        campaign.setFinishTime(dateTime);
        fakeCampaignParams(campaign);
    }

    @Step("Установить LastChange {1} для кампании campaignID = {0}")
    public void setLastChange(int campaignID, String dateTime) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(campaignID);
        campaign.setLastChange(dateTime);
        fakeCampaignParams(campaign);
    }

    @Step("Установить platform {1} для кампании campaignID = {0}")
    public void setPlatform(Long campaignID, CampaignsPlatform platform) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(campaignID.intValue());
        campaign.setPlatform(platform.getLiteral());
        fakeCampaignParams(campaign);
    }

    public void setLastChange(Long campaignID, String dateTime) {
        setLastChange(campaignID.intValue(), dateTime);
    }

    public void setLastShowTimeAnHourAgo(int campaignID) {
        setLastShowTime(campaignID, DateTime.now().minusHours(1));
    }

    public void setLastShowTimeAnHourAndMinuteAgo(int campaignID) {
        setLastShowTime(campaignID, DateTime.now().minusHours(1).minusMinutes(1));
    }

    public void setLastShowTimeAnHourAndMinuteAgo(Long campaignID) {
        setLastShowTimeAnHourAndMinuteAgo(campaignID.intValue());
    }


    public void setLastShowTimeNow(int campaignID) {
        setLastShowTime(campaignID, DateTime.now());
    }

    public void setLastShowTimeNow(Long campaignID) {
        setLastShowTimeNow(campaignID.intValue());
    }


    @Step("[FakeCampaignParams]: updateFakeCampaignParams")
    public void fakeCampaignParams(CampaignFakeInfo campaignFakeInfo) {
        updateService().FakeCampaignParams(campaignFakeInfo);
    }

    @Step("[FakeGetCampaignParams]: campaignID = {0}")
    public CampaignFakeInfo fakeGetCampaignParams(int campaignID) {
        return service().FakeGetCampaignParams(campaignID);
    }

    public CampaignFakeInfo fakeGetCampaignParams(Long campaignID) {
        return fakeGetCampaignParams(campaignID.intValue());
    }

    @Step("[FakeBalanceNotification]: Фейковая нотификация об изменении суммы {1} c валютой {2} на балансе кампании {0}")
    public void sendFakeNotificationFromBalance(int campaignID, float sum, Currency currency) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(campaignID);
        campaign.setMcbCampaign("0");
        campaign.setSumFromBalance(sum);
        campaign.setCurrency(currency.value());
        updateService().FakeBalanceNotification(campaign);
    }

    public void sendFakeNotificationFromBalance(Long campaignID, float sum, Currency currency) {
        sendFakeNotificationFromBalance(campaignID.intValue(), sum, currency);
    }

    public String getAgencyLogin(int campaignID) {
        CampaignFakeInfo campaign = fakeGetCampaignParams(campaignID);
        return campaign.getAgencyLogin();
    }

    public void setWalletOnOffDate(int campaignID, String walletOnOffDate) {
        CampaignFakeInfo campaign = fakeGetCampaignParams(campaignID);
        campaign.setWalletOnOffDate(walletOnOffDate);
        fakeCampaignParams(campaign);
    }

    public void resetWalletCreateDate(int campaignID) {
        DateTimeFormatter formatter = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss");
        setWalletOnOffDate(campaignID, formatter.print(DateTime.now().minusDays(2)));
    }

    public void resetWalletCreateDate(Long campaignID) {
        resetWalletCreateDate(campaignID.intValue());
    }

    public Callable<Boolean> statusBsSyncedIs(int cid, String statusBsSynced) {
        return () -> fakeGetCampaignParams(cid).getStatusBsSynced().equals(statusBsSynced);
    }

    @Step("[FakeCampaignParams]: сделать новую кампанию готовой к отправке в БК, cid = {0}")
    public void makeNewCampaignReadyForSendingToBS(int cid) {
        CampaignFakeInfo campaign = new CampaignFakeInfo();
        campaign.setCampaignID(cid);
        campaign.setStatusModerate(Status.YES);
        campaign.setStatusPostModerate(Status.ACCEPTED);
        campaign.setSum(10000f);
        fakeCampaignParams(campaign);
    }

    @Step("[FakeCampaignParams]: сделать кампанию внутренней рекламы готовой к отправке в БК, cid = {0}")
    public void makeCampaignInternalReadyForSendingToBS(Long cid) {
        new CampaignFakeInfoRequestBuilder(cid.intValue())
                .makeCampaignModerated()
                .send();
    }

    public void makeNewCampaignReadyForSendingToBS(Long cid) {
        makeNewCampaignReadyForSendingToBS(cid.intValue());
    }

    class CampaignFakeInfoRequestBuilder {
        CampaignFakeInfo campaign;

        CampaignFakeInfoRequestBuilder(int cid) {
            campaign = new CampaignFakeInfo();
            campaign.setCampaignID(cid);
        }

        @Step("[FakeCampaignParams]: кампания промодерированна")
        CampaignFakeInfoRequestBuilder makeCampaignModerated() {
            campaign.setStatusModerate(Status.YES);
            campaign.setStatusPostModerate(Status.ACCEPTED);
            return this;
        }

        @Step("[FakeCampaignParams]: кампания оплачена")
        CampaignFakeInfoRequestBuilder setSumOnCampaign() {
            campaign.setSum(10000f);
            return this;
        }

        void send() {
            fakeCampaignParams(campaign);
        }
    }
}
