package ru.yandex.autotests.direct.db.steps;

import java.math.BigDecimal;
import java.util.List;

import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BidsBaseBidType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BidsBaseRecord;
import ru.yandex.autotests.direct.db.steps.base.BasePpcSteps;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.qatools.allure.annotations.Step;

import static ru.yandex.autotests.direct.db.models.jooq.ppc.tables.BidsBase.BIDS_BASE;


public class BidsBaseSteps extends BasePpcSteps {

    public final static String OPTS_DEFAULT = "";
    public final static String OPTS_SUSPENDED = "suspended";
    public final static String OPTS_DELETED = "deleted";
    public final static String OPTS_NET_STOP = "net_stop";
    public final static String OPTS_SEARCH_STOP = "search_stop";

    @Step("DB: создание записи в таблице ppc.bids_base")
    public Long saveBidsBase(BidsBaseRecord bidsBaseRecord) {
        run(db -> db.insertInto(BIDS_BASE)
                .set(bidsBaseRecord)
                .execute());
        return bidsBaseRecord.getBidId();
    }

    @Step("DB: обновление записи в таблице ppc.bids_base")
    public void updateBidsBase(BidsBaseRecord bidsBaseRecord) {
        run(db -> db.update(BIDS_BASE)
                .set(bidsBaseRecord)
                .where(BIDS_BASE.BID_ID.eq(bidsBaseRecord.getBidId()))
                .execute()
        );
    }

    @Step("DB: получение записи из ppc.bids_base для bid_id = {0}")
    public BidsBaseRecord getBidsBaseByBidId(Long bidId) {
        return exec(db -> db.selectFrom(BIDS_BASE)
                .where(BIDS_BASE.BID_ID.eq(bidId))
                .fetchOne()
        );
    }

    @Step("DB: получение записи из ppc.bids_base для pid = {0}")
    public BidsBaseRecord getBidsBaseByPid(Long pid) {
        return exec(db -> db.selectFrom(BIDS_BASE)
                .where(BIDS_BASE.PID.eq(pid))
                .fetchOne()
        );
    }

    @Step("DB: получение записей из ppc.bids_base для pid = {0}")
    public List<BidsBaseRecord> getBidsBaseListByPid(Long pid) {
        return exec(db -> db.selectFrom(BIDS_BASE)
                .where(BIDS_BASE.PID.eq(pid))
                .fetch()
        );
    }

    public BidsBaseRecord getDefaultRelevanceMatch(Long cid, Long pid, Currency currency) {
        return new BidsBaseRecord()
                .setBidId(autoIncSteps().getNewBidId().longValue())
                .setBidType(BidsBaseBidType.relevance_match)
                .setPid(pid)
                .setCid(cid)
                .setPrice(MoneyCurrency.get(currency).getMinPrice().bigDecimalValue())
                .setPriceContext(MoneyCurrency.get(currency).getMinPrice().bigDecimalValue());

    }

    public BidsBaseRecord getDefaultOfferRetargeting(Long cid, Long pid, Currency currency) {
        return new BidsBaseRecord()
                .setBidId(autoIncSteps().getNewBidId().longValue())
                .setBidType(BidsBaseBidType.offer_retargeting)
                .setPid(pid)
                .setCid(cid)
                .setPrice(MoneyCurrency.get(currency).getMinPrice().bigDecimalValue())
                .setPriceContext(MoneyCurrency.get(currency).getMinPrice().bigDecimalValue());

    }

    public Long saveDefaultRelevanceMatch(Long cid, Long pid, Currency currency) {
        BidsBaseRecord bidsBaseRecord = getDefaultRelevanceMatch(cid, pid, currency);
        return saveBidsBase(bidsBaseRecord);
    }

    public Long saveDefaultRelevanceMatchWithOpts(Long cid, Long pid, Currency currency, String opts) {
        BidsBaseRecord bidsBaseRecord = new BidsBaseRecord()
                .setBidId(autoIncSteps().getNewBidId().longValue())
                .setBidType(BidsBaseBidType.relevance_match)
                .setPid(pid)
                .setCid(cid)
                .setPrice(MoneyCurrency.get(currency).getMinPrice().bigDecimalValue())
                .setPriceContext(MoneyCurrency.get(currency).getMinPrice().bigDecimalValue())
                .setOpts(opts);
        saveBidsBase(bidsBaseRecord);
        return bidsBaseRecord.getBidId();
    }

    public Long saveDefaultRelevanceMatchWithPrice(Long cid, Long pid, Currency currency,
            BigDecimal price, BigDecimal priceContext) {
        return saveDefaultRelevanceMatchWithPriceAndOpts(cid, pid, currency, price, priceContext, "");
    }

    public Long saveDefaultRelevanceMatchWithPriceAndOpts(Long cid, Long pid, Currency currency,
            BigDecimal price, BigDecimal priceContext, String opts) {
        BidsBaseRecord bidsBaseRecord = new BidsBaseRecord()
                .setBidId(autoIncSteps().getNewBidId().longValue())
                .setBidType(BidsBaseBidType.relevance_match)
                .setPid(pid)
                .setCid(cid)
                .setPrice(price)
                .setPriceContext(priceContext)
                .setOpts(opts);
        saveBidsBase(bidsBaseRecord);
        return bidsBaseRecord.getBidId();
    }

    public Long saveDefaultOfferRetargeting(Long cid, Long pid, Currency currency, String opts) {
        BidsBaseRecord bidsBaseRecord = getDefaultOfferRetargeting(cid, pid, currency)
                .setOpts(opts);
        return saveBidsBase(bidsBaseRecord);
    }

    public Long saveDefaultOfferRetargeting(Long cid, Long pid, Currency currency) {
        BidsBaseRecord bidsBaseRecord = getDefaultOfferRetargeting(cid, pid, currency);
        return saveBidsBase(bidsBaseRecord);
    }

    public Long saveDefaultOfferRetargeting(Long cid, Long pid, BigDecimal price, BigDecimal priceContext) {
        BidsBaseRecord bidsBaseRecord = new BidsBaseRecord()
                .setBidId(autoIncSteps().getNewBidId().longValue())
                .setBidType(BidsBaseBidType.offer_retargeting)
                .setPid(pid)
                .setCid(cid)
                .setPrice(price)
                .setPriceContext(priceContext);
        saveBidsBase(bidsBaseRecord);
        return bidsBaseRecord.getBidId();
    }

    public Long saveDefaultKeyword(Long cid, Long pid, Currency currency) {
        BidsBaseRecord bidsBaseRecord = new BidsBaseRecord()
                .setBidId(autoIncSteps().getNewBidId().longValue())
                .setBidType(BidsBaseBidType.keyword)
                .setPid(pid)
                .setCid(cid)
                .setPrice(MoneyCurrency.get(currency).getMinPrice().bigDecimalValue())
                .setPriceContext(MoneyCurrency.get(currency).getMinPrice().bigDecimalValue());
        saveBidsBase(bidsBaseRecord);
        return bidsBaseRecord.getBidId();
    }

    @Step("DB: установка opts в таблице ppc.bids_base для bid_id:{0}, opts:{1}")
    public void setBidsBaseOpts(Long bidId, String opts) {
        run(db -> db.update(BIDS_BASE)
                .set(BIDS_BASE.OPTS, opts)
                .where(BIDS_BASE.BID_ID.eq(bidId))
                .execute()
        );
    }

    @Step("DB: установка opts в таблице ppc.bids_base для bid_id:{0}, price:{1}, price_context:{1}, autobudgetPriority{2}")
    public void setBidsBasePrices(Long bidId, BigDecimal price, BigDecimal priceContext, Integer autobudgetPriority) {
        run(db -> db.update(BIDS_BASE)
                .set(BIDS_BASE.PRICE, price)
                .set(BIDS_BASE.PRICE_CONTEXT, priceContext)
                .set(BIDS_BASE.AUTOBUDGETPRIORITY, autobudgetPriority)
                .where(BIDS_BASE.BID_ID.eq(bidId))
                .execute()
        );
    }
}
