package ru.yandex.autotests.direct.db.steps;

import java.math.BigDecimal;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import javax.util.streamex.StreamEx;

import org.jooq.impl.DSL;

import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BidsDynamicRecord;
import ru.yandex.autotests.direct.db.steps.base.BasePpcSteps;
import ru.yandex.autotests.irt.testutils.json.JsonUtils;
import ru.yandex.qatools.allure.annotations.Step;

import static ru.yandex.autotests.direct.db.models.jooq.ppc.tables.BidsDynamic.BIDS_DYNAMIC;
import static ru.yandex.autotests.irt.testutils.allure.AllureUtils.addJsonAttachment;

public class BidsDynamicSteps extends BasePpcSteps {

    // TODO переключить на этот метод использование из BidsSteps
    @Step("DB: получение записей в таблице ppc.bids_dynamic по pid: {0}")
    public List<BidsDynamicRecord> getBidsDynamicByPid(Long pid) {
        List<BidsDynamicRecord> result = exec(db -> db.selectFrom(BIDS_DYNAMIC)
                .where(BIDS_DYNAMIC.PID.eq(pid))
                .fetch());

        logRecords(result);

        return result;
    }

    @Step("DB: получение записей из таблицы ppc.bids_dynamic (pid = {0}, dyn_con_id = {1})")
    public List<BidsDynamicRecord> getBidsDynamicByDynCondId(Long pid, Long dynCondId) {
        List<BidsDynamicRecord> result = exec(db -> db.selectFrom(BIDS_DYNAMIC)
                .where(BIDS_DYNAMIC.PID.eq(pid)
                        .and(BIDS_DYNAMIC.DYN_COND_ID.eq(dynCondId)))
                .fetch());

        logRecords(result);

        return result;
    }

    @Step("DB: получение записи из таблицы ppc.bids_dynamic (dyn_id = {0})")
    public BidsDynamicRecord getBidsDynamic(Long dynId) {
        BidsDynamicRecord result = exec(db -> db.selectFrom(BIDS_DYNAMIC)
                .where(BIDS_DYNAMIC.DYN_ID.eq(dynId))
                .fetchOne());
        logRecord(result);
        return result;
    }

    @Step("DB: получение списка dyn_cond_id нацеливаний для группы (pid = {0})")
    public List<Long> getBidsDynamicCondIdsByPid(Long pid) {
        List<BidsDynamicRecord> bidsDynamicList = getBidsDynamicByPid(pid);
        return StreamEx.of(bidsDynamicList).map(BidsDynamicRecord::getDynCondId).collect(Collectors.toList());
    }

    @Step("DB: получение списка dyn_id нацеливаний для группы (pid = {0})")
    public List<Long> getBidsDynamicIdsByPid(Long pid) {
        List<BidsDynamicRecord> bidsDynamicList = getBidsDynamicByPid(pid);
        return StreamEx.of(bidsDynamicList).map(BidsDynamicRecord::getDynId).toList();
    }

    @Step("DB: удаление записи из таблицы ppc.bids_dynamic (pid = {0}, dyn_cond_id = {1})")
    public void deleteBidsDynamicIdsByDynCondId(Long pid, Long dynCondId) {
        run(db -> db.deleteFrom(BIDS_DYNAMIC)
                .where(BIDS_DYNAMIC.PID.eq(pid)
                        .and(BIDS_DYNAMIC.DYN_COND_ID.eq(dynCondId)))
                .execute()
        );
    }

    @Step("DB: удаление записи из таблицы ppc.bids_dynamic (pid = {0})")
    public void deleteBidsDynamicIdsByPid(Long pid) {
        run(db -> db.deleteFrom(BIDS_DYNAMIC)
                .where(BIDS_DYNAMIC.PID.eq(pid))
                .execute()
        );
    }

    @Step("DB: установить статус suspended (pid = {0}, dyn_cond_id = {1}")
    public void setSuspended(Long pid, Long dynCondId) {
        run(db -> db.update(BIDS_DYNAMIC)
                .set(BIDS_DYNAMIC.OPTS, DSL.concat(BIDS_DYNAMIC.OPTS, "suspended"))
                .where(BIDS_DYNAMIC.PID.eq(pid)
                        .and(BIDS_DYNAMIC.DYN_COND_ID.eq(dynCondId)))
                .execute()
        );
    }

    @Step("DB: установить статус suspended (dyn_id = {0}")
    public void setSuspended(Long dynId) {
        run(db -> db.update(BIDS_DYNAMIC)
                .set(BIDS_DYNAMIC.OPTS, DSL.concat(BIDS_DYNAMIC.OPTS, "suspended"))
                .where(BIDS_DYNAMIC.DYN_ID.eq(dynId))
                .execute()
        );
    }

    @Step("DB: Oбновление поля price в таблице ppc.bids_dynamic, id: {0}, price: {1}")
    public void setPrice(Long id, BigDecimal price) {
        run(db -> db.update(BIDS_DYNAMIC)
                .set(BIDS_DYNAMIC.PRICE, price)
                .where(BIDS_DYNAMIC.DYN_ID.eq(id))
                .execute()
        );
    }

    @Step("DB: Oбновление поля price_context в таблице ppc.bids_dynamic, id: {0}, price_context: {1}")
    public void setPriceContext(Long id, BigDecimal priceContext) {
        run(db -> db.update(BIDS_DYNAMIC)
                .set(BIDS_DYNAMIC.PRICE_CONTEXT, priceContext)
                .where(BIDS_DYNAMIC.DYN_ID.eq(id))
                .execute()
        );
    }

    @Step("DB: установить приоритет нацеливания (dyn_id = {0}, AutobudgetPriority = {1})")
    public void setAutobudgetPriority(long dynId, int autobudgetPriority) {
        run(db -> db.update(BIDS_DYNAMIC)
                .set(BIDS_DYNAMIC.AUTOBUDGETPRIORITY, autobudgetPriority)
                .where(BIDS_DYNAMIC.DYN_ID.eq(dynId))
                .execute()
        );
    }

    private void logRecords(List<BidsDynamicRecord> result) {
        String logRecord =
                JsonUtils.toStringLow(StreamEx.of(result).nonNull().map(BidsDynamicRecord::intoMap).toList());
        getLogger().info("полученные записи из таблицы ppc.bids_dynamic: " + logRecord);
        addJsonAttachment("полученные записи", logRecord);
    }

    private void logRecord(BidsDynamicRecord result) {
        logRecords(Collections.singletonList(result));
    }
}
