package ru.yandex.autotests.direct.db.steps;

import java.sql.Timestamp;
import java.util.List;

import javax.util.streamex.StreamEx;

import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BsResyncQueueRecord;
import ru.yandex.autotests.direct.db.steps.base.BasePpcSteps;
import ru.yandex.autotests.irt.testutils.json.JsonUtils;
import ru.yandex.qatools.allure.annotations.Step;

import static java.util.Collections.singletonList;
import static ru.yandex.autotests.direct.db.models.jooq.ppc.Tables.BS_RESYNC_QUEUE;
import static ru.yandex.autotests.irt.testutils.allure.AllureUtils.addJsonAttachment;

/**
 * Created by buhter on 06/03/2017.
 */
public class BsResyncQueueSteps extends BasePpcSteps {

    @Step("DB: получение записей в таблице ppc.bs_resync_queue по cid: {0}")
    public List<BsResyncQueueRecord> getBsResyncQueueRecordsByCid(Long cid) {
        List<BsResyncQueueRecord> result = exec(db -> db.selectFrom(BS_RESYNC_QUEUE)
                .where(BS_RESYNC_QUEUE.CID.eq(cid))
                .fetch()
        );
        logRecords(result);
        return result;
    }

    @Step("DB: получение записи в таблице ppc.bs_resync_queue по cid: {0}")
    public BsResyncQueueRecord getBsResyncQueueRecord(Long cid) {
        BsResyncQueueRecord result = exec(db -> db.selectFrom(BS_RESYNC_QUEUE)
                .where(BS_RESYNC_QUEUE.CID.eq(cid))
                .fetchOne()
        );
        logRecord(result);
        return result;
    }

    @Step("DB: получение записи в таблице ppc.bs_resync_queue по cid: {0}, pid: {1}, bid {2}")
    public BsResyncQueueRecord getBsResyncQueueRecord(Long cid, Long pid, Long bid) {
        BsResyncQueueRecord result = exec(db -> db.selectFrom(BS_RESYNC_QUEUE)
                .where(BS_RESYNC_QUEUE.CID.eq(cid)
                        .and(BS_RESYNC_QUEUE.PID.eq(pid))
                        .and(BS_RESYNC_QUEUE.BID.eq(bid)))
                .fetchOne()
        );
        logRecord(result);
        return result;
    }

    @Step("DB: получение записи в таблице ppc.bs_resync_queue по cid: {0}, pid: {1}")
    public BsResyncQueueRecord getBsResyncQueueRecord(Long cid, Long pid) {
        BsResyncQueueRecord result = exec(db -> db.selectFrom(BS_RESYNC_QUEUE)
                .where(BS_RESYNC_QUEUE.CID.eq(cid)
                        .and(BS_RESYNC_QUEUE.PID.eq(pid)))
                .fetchOne()
        );
        logRecord(result);
        return result;
    }

    @Step("DB: удаление записей из таблицы ppc.bs_resync_queue (cid = {0})")
    public void deleteCampaignFromBsResyncQueueByCid(Long cid) {
        List<BsResyncQueueRecord> result = getBsResyncQueueRecordsByCid(cid);
        if (result != null && !result.isEmpty()) {
            String logRecord = JsonUtils.toStringLow(StreamEx.of(result).map(BsResyncQueueRecord::intoMap).toList());
            getLogger().info("удаляемые записи из таблицы ppc.bs_resync_queue: " + logRecord);
            addJsonAttachment("удаляемые записи", logRecord);
            // чтобы не удалять данные поштучно - используем автообновлемый seq-time
            Timestamp maxSequenceTime = StreamEx.of(result)
                    .map(BsResyncQueueRecord::getSequenceTime)
                    .max(Timestamp::compareTo)
                    .get();
            run(db -> db.delete(BS_RESYNC_QUEUE)
                    .where(BS_RESYNC_QUEUE.CID.eq(cid)
                            .and(BS_RESYNC_QUEUE.SEQUENCE_TIME.le(maxSequenceTime)))
                    .execute()
            );
        } else {
            getLogger().info("Записи в таблице ppc.bs_resync_queue отсутствует для cid = " + cid);
        }
    }

    public boolean isCidInResyncQueue(Long cid) {
        return getBsResyncQueueRecord(cid) != null;
    }

    public boolean isBidInResyncQueue(Long cid, Long pid, Long bid) {
        return getBsResyncQueueRecord(cid, pid, bid) != null;
    }

    public boolean isPidInResyncQueue(Long cid, Long pid) {
        return getBsResyncQueueRecord(cid, pid) != null;
    }

    private void logRecords(List<BsResyncQueueRecord> result) {
        String logRecord = JsonUtils.toStringLow(
                StreamEx.of(result).nonNull().map(BsResyncQueueRecord::intoMap).toList());
        getLogger().info("полученные записи из таблицы ppc.bs_resync_queue: " + logRecord);
        addJsonAttachment("полученные записи", logRecord);
    }

    private void logRecord(BsResyncQueueRecord record) {
        logRecords(singletonList(record));
    }
}
