package ru.yandex.autotests.direct.db.steps;

import java.sql.Timestamp;
import java.time.Instant;
import java.util.List;

import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.ClientBrandsRecord;
import ru.yandex.autotests.direct.db.steps.base.BasePpcSteps;
import ru.yandex.qatools.allure.annotations.Step;

import static ru.yandex.autotests.direct.db.models.jooq.ppc.tables.ClientBrands.CLIENT_BRANDS;

public class ClientBrandsSteps extends BasePpcSteps {

    /**
     * Получить все записи бренда по ID его главного клиента.
     * @param brandChiefClientId ID главного клиента бренда.
     * @return Список записей {@link ClientBrandsRecord}, где {@code brand_clientid} совпадает со значением аргумента.
     */
    @Step("DB: Получение записей из таблицы ppc.client_brands по brand_clientid: {0}")
    public List<ClientBrandsRecord> getBrands(Long brandChiefClientId) {
        return exec(db -> db.selectFrom(CLIENT_BRANDS)
                .where(CLIENT_BRANDS.BRAND_CLIENTID.eq(brandChiefClientId))
                .fetch());
    }

    @Step("DB: Получение записи из таблицы ppc.client_brands по clientid: {0}")
    public ClientBrandsRecord getBrand(Long clientId) {
        return exec(db -> db.selectFrom(CLIENT_BRANDS)
                .where(CLIENT_BRANDS.CLIENTID.eq(clientId))
                .fetchOne());
    }

    /**
     * Создать бренд, главным клиентом которого будет обладатель переданного ID.
     */
    @Step("DB: создание записи в таблице ppc.client_brands")
    public void createBrand(Long clientId) {
        exec(db -> db.insertInto(CLIENT_BRANDS)
                .set(new ClientBrandsRecord(clientId, clientId, now()))
                .execute());
    }

    @Step("DB: создание записи в таблице ppc.client_brands")
    public void bindToBrand(Long clientId, Long brandChiefClientId) {
        exec(db -> db.insertInto(CLIENT_BRANDS)
                .set(new ClientBrandsRecord(clientId, brandChiefClientId, now()))
                .execute());
    }

    @Step("DB: обновление записи в таблице ppc.client_brands")
    public void updateClientBrandRecord(ClientBrandsRecord record) {
        exec(db -> db.update(CLIENT_BRANDS)
                .set(record)
                .where(CLIENT_BRANDS.CLIENTID.eq(record.getClientid()))
                .execute());
    }

    /**
     * Удалить из таблицы все записи, где главным клиентом бренда является обладатель переданного ID.
     */
    @Step("DB: удаление записей из таблицы ppc.client_brands для brand_cilentid: {0}")
    public void deleteBrand(Long brandChiefClientId) {
        exec(db -> db.deleteFrom(CLIENT_BRANDS)
                .where(CLIENT_BRANDS.BRAND_CLIENTID.eq(brandChiefClientId))
                .execute());
    }

    @Step("DB: удаление записи из таблицы ppc.client_brands")
    public void unbindClientFromBrand(Long clientId, Long brandChiefClientId) {
        exec(db -> db.deleteFrom(CLIENT_BRANDS)
                .where(CLIENT_BRANDS.CLIENTID.eq(clientId)
                        .and(CLIENT_BRANDS.BRAND_CLIENTID.eq(brandChiefClientId)))
                .execute());
    }

    /**
     * Исключить клиента из всех брендов.
     */
    @Step("DB: удаление записей из таблицы ppc.client_brands для clientid: {0}")
    public void unbindClientFromAllBrands(Long clientId) {
        exec(db -> db.deleteFrom(CLIENT_BRANDS)
                .where(CLIENT_BRANDS.CLIENTID.eq(clientId))
                .execute());
    }

    private static Timestamp now() {
        return Timestamp.from(Instant.now());
    }

}
