package ru.yandex.autotests.direct.db.steps;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import org.jooq.impl.DSL;

import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.ClientsOptionsStatusbalancebanned;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.ClientsOptionsRecord;
import ru.yandex.autotests.direct.db.steps.base.BasePpcSteps;
import ru.yandex.autotests.direct.db.steps.base.DirectDbStepsException;
import ru.yandex.autotests.irt.testutils.json.JsonUtils;
import ru.yandex.qatools.allure.annotations.Step;

import static ru.yandex.autotests.direct.db.models.jooq.ppc.tables.ClientsOptions.CLIENTS_OPTIONS;
import static ru.yandex.autotests.irt.testutils.allure.AllureUtils.addJsonAttachment;

/**
 * Набор степов для работы с таблицей ppc.client_options,
 * в которой хранится показатель качества аккаунта на ClientID в разбивке по дате.
 */
public class ClientsOptionsSteps extends BasePpcSteps {
    /**
     * Enum описывающий возможные значения поля БД client_flags типа MySQL.SET
     */
    public enum ClientFlagsEnum { // MySQL Set type
        NO_TEXT_AUTOCORRECTION("no_text_autocorrection"),
        NO_DISPLAY_HREFS("no_display_hrefs"),
        NOT_CONVERT_TO_CURRENCY("not_convert_to_currency"),
        CREATE_WITHOUT_WALLET("create_without_wallet"),
        FEATURE_CONTEXT_RELEVANCE_MATCH_ALLOWED("feature_context_relevance_match_allowed"),
        FEATURE_CONTEXT_RELEVANCE_MATCH_INTERFACE_ONLY("feature_context_relevance_match_interface_only"),
        AS_SOON_AS_POSSIBLE("as_soon_as_possible");
        private final String flagName; // имя флага в БД

        ClientFlagsEnum(String flagName) {
            this.flagName = flagName;
        }

        @Override
        public String toString() {
            return flagName;
        }
    }

    final private static String FLAGS_DELIMITER = ",";

    private static String joinFlags(ClientFlagsEnum... flags) {
        return Arrays.stream(flags).map(ClientFlagsEnum::toString).collect(Collectors.joining(FLAGS_DELIMITER));
    }

    private static List<String> splitFlagsString(String flags) {
        return Arrays.stream(flags.split(FLAGS_DELIMITER)).collect(Collectors.toList());
    }

    private static String joinFlagStrings(String... flags) {
        return String.join(FLAGS_DELIMITER, flags);
    }

    private static String joinFlagStrings(List<String> flags) {
        return joinFlagStrings(flags.toArray(new String[flags.size()]));
    }

    /**
     * Метод возвращает одну запись настроек клиента записи из таблицы ppc.clients_options по ClientID
     *
     * @param clientID id клиента
     * @return ClientsOptionsRecord
     */
    public ClientsOptionsRecord getClientOptions(Long clientID) {
        return exec("DB: получение настроек клиента записи из таблицы ppc.clients_options по ClientID " + clientID,
                db -> db.selectFrom(CLIENTS_OPTIONS)
                        .where(CLIENTS_OPTIONS.CLIENTID.eq(clientID))
                        .fetchOne());
    }

    /**
     * Метод устанавливает флаги в set-поле client_flags таблицы ppc.clients_options
     *
     * @param clientId -  id клиента которому обновляем флаги
     * @param flags    - флаги которые надо проставить, если не передано то сбрасываем все
     */
    @Step("DB: Oбновление флагов в client_flags в таблице ppc.clients_options")
    public void updateClientFlags(Long clientId, ClientFlagsEnum... flags) {
        final ClientsOptionsRecord clientOptions = new ClientsOptionsRecord()
                .setClientFlags(joinFlags(flags))
                .setClientid(clientId);
        updateClientOptions(clientOptions);
    }

    /**
     * Метод устанавливает флаг в set-поля client_flags таблицы ppc.clients_options
     *
     * @param clientId  -  id клиента которому обновляем флаги
     * @param flagToSet - флаги которые надо проставить, если не передано то сбрасываем все
     */
    @Step("DB: Установка флага в client_flags в таблице ppc.clients_options")
    public void setClientFlag(Long clientId, ClientFlagsEnum flagToSet) {
        ClientsOptionsRecord clientOptions = getClientOptions(clientId);

        if (clientOptions == null) {
            createEmptyClientOptions(clientId);
            clientOptions = getClientOptions(clientId);
            if (clientOptions == null) {
                throw new DirectDbStepsException("Не найдена запись с опциями клиента ClientID: " + clientId);
            }
        }

        final String newClientFlags = clientOptions.getClientFlags() == null || clientOptions.getClientFlags().isEmpty()
                ? flagToSet.toString()
                : joinFlagStrings(clientOptions.getClientFlags(), flagToSet.toString());

        clientOptions.setClientFlags(newClientFlags);
        updateClientOptions(clientOptions);
    }

    /**
     * Метод удаляет флаг из set-поля client_flags таблицы ppc.clients_options
     *
     * @param clientId  -  id клиента которому обновляем флаги
     * @param flagToDrop - флаги которые надо проставить, если не передано то сбрасываем все
     */
    @Step("DB: Удаление флага из client_flags в таблице ppc.clients_options")
    public void dropClientFlag(Long clientId, ClientFlagsEnum flagToDrop) {
        final ClientsOptionsRecord clientOptions = getClientOptions(clientId);
        final String flagsString = clientOptions.getClientFlags();
        if (flagsString != null && !flagsString.isEmpty()) {
            List<String> flagsUpdate = splitFlagsString(flagsString).stream()
                    .filter(f -> !f.equals(flagToDrop.toString()))
                    .collect(Collectors.toList());
            clientOptions.setClientFlags( joinFlagStrings(flagsUpdate) );
            updateClientOptions(clientOptions);
        } /* else - нет флага, нет проблем. flags может NULL, может быть пустой строкой */
    }

    @Step("DB: Изменения поля hide_market_rating")
    public void setHideMarketRating(Long clientID, int hideMarketRatingFlag) {
        run(db -> db.update(CLIENTS_OPTIONS)
                .set(CLIENTS_OPTIONS.HIDE_MARKET_RATING, hideMarketRatingFlag)
                .where(CLIENTS_OPTIONS.CLIENTID.eq(clientID))
                .execute()
        );
    }

    @Step("DB: Изменение поля auto_overdraft_lim")
    public void setAutoOverdraftLimit(Long clientID, BigDecimal autoOverdraftLim) {
        run(db -> db.update(CLIENTS_OPTIONS)
                .set(CLIENTS_OPTIONS.AUTO_OVERDRAFT_LIM, autoOverdraftLim)
                .where(CLIENTS_OPTIONS.CLIENTID.eq(clientID))
                .execute()
        );
    }

    @Step("DB: Изменение поля overdraft_lim")
    public void setOverdraftLimit(Long clientID, BigDecimal overdraftLim) {
        run(db -> db.update(CLIENTS_OPTIONS)
                .set(CLIENTS_OPTIONS.OVERDRAFT_LIM, overdraftLim)
                .where(CLIENTS_OPTIONS.CLIENTID.eq(clientID))
                .execute()
        );
    }

    @Step("DB: Изменение поля statusBalanceBanned")
    public void setStatusBalanceBanned(Long clientID, ClientsOptionsStatusbalancebanned statusBalanceBanned) {
        Preconditions.checkNotNull(statusBalanceBanned);
        run(db -> db.update(CLIENTS_OPTIONS)
                .set(CLIENTS_OPTIONS.STATUSBALANCEBANNED, statusBalanceBanned)
                .where(CLIENTS_OPTIONS.CLIENTID.eq(clientID))
                .execute()
        );
    }

    @Step("DB: Изменение поля debt")
    public void setDebt(Long clientID, BigDecimal debt) {
        run(db -> db.update(CLIENTS_OPTIONS)
                .set(CLIENTS_OPTIONS.DEBT, debt)
                .where(CLIENTS_OPTIONS.CLIENTID.eq(clientID))
                .execute()
        );
    }

    @Step("DB: Удаление записи для ClientID={0} из таблицы ppc.clients_options")
    public void dropClientRecord(Long clientId) {
        run(
                db -> db.deleteFrom(CLIENTS_OPTIONS)
                .where(CLIENTS_OPTIONS.CLIENTID.eq(clientId))
                .execute()
        );
    }

    /**
     * Метод обновляет запись в таблице ppc.client_options
     *
     * @param clientOptions объект с настройками клиента
     * @return
     */
    @Step("DB: Oбновление записи в таблице ppc.clients_options")
    public void updateClientOptions(ClientsOptionsRecord clientOptions) {
        String logRecord = JsonUtils.toStringLow(clientOptions.intoMap());
        getLogger().info("Обновление: " + logRecord);
        addJsonAttachment("обновление", logRecord);
        run(db -> db.update(CLIENTS_OPTIONS)
                .set(clientOptions)
                .where(CLIENTS_OPTIONS.CLIENTID.eq(clientOptions.getClientid()))
                .execute()
        );
    }

    /**
     * Метод создает пустую запись в таблице ppc.client_options
     *
     * @param clientId
     * @return
     */
    @Step("DB: Создание пустой записи в таблице ppc.clients_options")
    public void createEmptyClientOptions(Long clientId) {
        run(db -> db.insertInto(CLIENTS_OPTIONS)
                .columns(
                        CLIENTS_OPTIONS.CLIENTID,
                        CLIENTS_OPTIONS.BALANCE_TID,
                        CLIENTS_OPTIONS.OVERDRAFT_LIM,
                        CLIENTS_OPTIONS.DEBT,
                        CLIENTS_OPTIONS.NEXTPAYDATE)
                .select(
                        db.select(
                                DSL.val(clientId),
                                DSL.val(BigInteger.ZERO),
                                DSL.val(BigDecimal.ZERO),
                                DSL.val(BigDecimal.ZERO),
                                DSL.val(0L).cast(CLIENTS_OPTIONS.NEXTPAYDATE.getDataType()))
                )
                .onDuplicateKeyIgnore()
                .execute()
        );
    }

    @Step("DB: Выставление записи в таблице ppc.clients_options для ClientID={0}, значения is_business_unit={1}")
    public void setClientOptionsIsBusinessUnit(Long clientID, int isBusinessUnit) {
        run(db -> db.update(CLIENTS_OPTIONS)
                .set(CLIENTS_OPTIONS.IS_BUSINESS_UNIT, isBusinessUnit)
                .where(CLIENTS_OPTIONS.CLIENTID.eq(clientID))
                .execute()
        );
    }

    @Step("DB: Выставление записи в таблице ppc.clients_options для ClientID={0}, значения is_brand={1}")
    public void setClientOptionsIsBrand(Long clientID, int isBrand) {
        run(db -> db.update(CLIENTS_OPTIONS)
                .set(CLIENTS_OPTIONS.IS_BRAND, isBrand)
                .where(CLIENTS_OPTIONS.CLIENTID.eq(clientID))
                .execute()
        );
    }

    @Step("DB: Выставление записи в таблице ppc.clients_options для ClientID={0}, значения is_ya_agency_client=1")
    public void makeYaAgencyClient(Long clientID) {
        run(db -> db.update(CLIENTS_OPTIONS)
                .set(CLIENTS_OPTIONS.IS_YA_AGENCY_CLIENT, 1)
                .where(CLIENTS_OPTIONS.CLIENTID.eq(clientID))
                .execute()
        );
    }
}
