package ru.yandex.autotests.direct.db.steps;

import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.FeedsBusinessType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.FeedsSource;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.FeedsUpdateStatus;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.FeedsRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.PerfFeedCategoriesRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.PerfFeedHistoryRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.PerfFeedVendorsRecord;
import ru.yandex.autotests.direct.db.steps.base.BasePpcSteps;
import ru.yandex.qatools.allure.annotations.Step;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.stream.Stream;

import static ru.yandex.autotests.direct.db.models.jooq.ppc.Tables.*;
import static ru.yandex.autotests.direct.db.models.jooq.ppc.tables.Feeds.FEEDS;


/*
* todo javadoc
*/
public class FeedsSteps extends BasePpcSteps {
    public static final String DEFAULT_FEED_NAME = "FeedForTest";

    @Step("DB: получение записей из таблицы ppc.feeds (clientId = {0})")
    public List<FeedsRecord> getFeeds(Long clientId) {
        return exec(db -> db.selectFrom(FEEDS)
                .where(FEEDS.CLIENTID.eq(clientId))
                .fetch());
    }


    @Step("DB: получение записи из таблицы ppc.feeds (feedId = {0})")
    public FeedsRecord getFeed(Long feedId) {
        return exec(db -> db.selectFrom(FEEDS)
                .where(FEEDS.FEED_ID.eq(feedId))
                .fetchOne());
    }

    @Step("DB: получение записи из таблицы ppc.feeds (feedId = {0}, clientId = {1})")
    public FeedsRecord getFeed(Long feedId, String clientId) {
        return getFeed(feedId);
    }

    @Step("DB: удаление записи из таблицы ppc.feeds (feedId = {0}, clientId = {1})")
    public void deleteFeedById(Long feedId, String clientId) {
        deleteFeedById(feedId);
    }

    @Step("DB: удаление записи из таблицы ppc.feeds (feedId = {0}")
    public void deleteFeedById(Long feedId) {
        run(db -> db.deleteFrom(FEEDS)
                .where(FEEDS.FEED_ID.eq(feedId))
                .execute());
    }

    @Step("DB: добавление записи в таблицу ppc.feeds")
    public Long createFeed(FeedsRecord feed) {
        Long feedId = autoIncSteps().getNewFeedId();
        run(db -> db.insertInto(FEEDS)
                .set(feed.setFeedId(feedId))
                .execute());
        return feedId;
    }

    public Long createFeed(FeedsRecord feed, String clientId) {
        feed.setClientid(Long.valueOf(clientId));
        return createFeed(feed);
    }

    public Long createDefaultFeed(String clientId) {
        return createFeed(getDefaultFeed(clientId));
    }

    public FeedsRecord getDefaultFeed(String clientId) {
        FeedsRecord feeds = new FeedsRecord()
                .setClientid(Long.valueOf(clientId))
                .setFeedType(null)
                .setSource(FeedsSource.url)
                .setName(DEFAULT_FEED_NAME)
                .setUrl("https://yandex.st/market-export/1.0-17/partner/help/YML.xml")
                .setFetchErrorsCount(0)
                .setRefreshInterval(10L)
                .setUpdateStatus(FeedsUpdateStatus.New);
        return feeds;
    }

    public List<Long> createFeeds(String clientId, FeedsRecord... feeds) {
        List<Long> result = new ArrayList<>();
        Stream.of(feeds).forEach(t -> result.add(createFeed(t, clientId)));
        return result;
    }

    public List<Long> createFeeds(String clientId, FeedsRecord feed, int count) {
        List<Long> result = new ArrayList<>();
        for (int i = 0; i < count; i++) {
            result.add(createFeed(feed, clientId));
        }
        return result;
    }

    @Step("DB: обновление записи в таблице ppc.feeds")
    public void updateFeed(FeedsRecord feed) {
        run(db -> db.update(FEEDS)
                .set(feed)
                .where(FEEDS.FEED_ID.eq(feed.getFeedId()))
                .execute());
    }

    @Step("DB: обновление записи в таблице ppc.feeds (clientId = {1})")
    public void updateFeed(FeedsRecord feed, String clientId) {
        updateFeed(feed);
    }

    public void updateFeedStatusAndOffer(long feedId, long offerCount, FeedsUpdateStatus status, String clientId) {
        FeedsRecord feed = getFeed(feedId, clientId);
        feed.setUpdateStatus(status);
        feed.setOffersCount(offerCount);
        updateFeed(feed, clientId);
    }

    @Step("DB: обновление статуса фида (id = {0}, статус = {1}, clientId = {2}")
    public void updateFeedStatus(long feedId, FeedsUpdateStatus status, String clientId) {
        FeedsRecord feed = getFeed(feedId, clientId);
        feed.setUpdateStatus(status);
        updateFeed(feed, clientId);
    }

    @Step("DB: обновление урла фида (id = {0}, clientId = {1}, url = {2}")
    public void updateFeedUrl(long feedId, String clientId, String url) {
        FeedsRecord feed = getFeed(feedId, clientId);
        feed.setUrl(url);
        updateFeed(feed, clientId);
    }

    @Step("DB: обновление isRemoveUtm фида (id = {0}, clientId = {1}, isRemoveUtm = {2}")
    public void updateFeedIsRemoveUtm(long feedId, String clientId, Integer isRemoveUtm) {
        FeedsRecord feed = getFeed(feedId, clientId);
        feed.setIsRemoveUtm(isRemoveUtm);
        updateFeed(feed, clientId);
    }

    @Step("DB: обновление login фида (id = {0}, clientId = {1}, login = {2}")
    public void updateFeedLogin(long feedId, String clientId, String login) {
        FeedsRecord feed = getFeed(feedId, clientId);
        feed.setLogin(login);
        updateFeed(feed, clientId);
    }

    @Step("DB: обновление login фида (id = {0}, clientId = {1}, password = {2}")
    public void updateFeedPassword(long feedId, String clientId, byte[] password) {
        FeedsRecord feed = getFeed(feedId, clientId);
        feed.setEncryptedPassword(password);
        updateFeed(feed, clientId);
    }

    @Step("DB: обновление business_type фида (id = {0}, clientId = {1}, business_type = {2}")
    public void updateFeedBusinessType(long feedId, String clientId, FeedsBusinessType feedsBusinessType) {
        FeedsRecord feed = getFeed(feedId, clientId);
        feed.setBusinessType(feedsBusinessType);
        updateFeed(feed, clientId);
    }

    @Step("DB: обновление feed_type фида (id = {0}, clientId = {1}, feed_type = {2}")
    public void updateFeedType(long feedId, String clientId, String feedType) {
        FeedsRecord feed = getFeed(feedId, clientId);
        feed.setFeedType(feedType);
        updateFeed(feed, clientId);
    }


    public void updateFeedsStatus(List<Long> feedIds, FeedsUpdateStatus status, String clientId) {
        feedIds.forEach(feedId -> updateFeedStatus(feedId, status, clientId));
    }

    public void updateFeedsStatus(FeedsUpdateStatus status, String clientId, Long... feedIds) {
        Stream.of(feedIds).forEach(t -> updateFeedStatus(t, status, clientId));
    }

    public void updateFeedsStatus(FeedsUpdateStatus status, String clientId, List<Long> feedIds) {
        feedIds.forEach(t -> updateFeedStatus(t, status, clientId));
    }

    public Callable<Boolean> updateStatusIs(FeedsUpdateStatus updateStatus, long feedId) {
        return () -> getFeed(feedId).getUpdateStatus().equals(updateStatus);
    }

    @Step("Удаление связи фида {0} с ДМО группами")
    public void deleteAdgroupsPerformanceRecords(Long feedId) {
        run(db -> db.deleteFrom(ADGROUPS_PERFORMANCE)
                .where(ADGROUPS_PERFORMANCE.FEED_ID.eq(feedId))
                .execute());
    }

    @Step("Удаление связи фида {0} с ДТО группами")
    public void deleteAdgroupsDynamicRecords(Long feedId) {
        run(db -> db.deleteFrom(ADGROUPS_DYNAMIC)
                .where(ADGROUPS_DYNAMIC.FEED_ID.eq(feedId))
                .execute());
    }

    @Step("Удаление связи фида {0} с категориями")
    public void deletePerfFeedCategoriesRecords(Long feedId) {
        run(db -> db.deleteFrom(PERF_FEED_CATEGORIES)
                .where(PERF_FEED_CATEGORIES.FEED_ID.eq(feedId))
                .execute());
    }

    @Step("Удаление связи фида {0} с вендорами")
    public void deletePerfFeedVendorsRecords(Long feedId) {
        run(db -> db.deleteFrom(PERF_FEED_VENDORS)
                .where(PERF_FEED_VENDORS.FEED_ID.eq(feedId))
                .execute());
    }

    @Step("Удаление истории фида {0} ")
    public void deletePerfFeedHistoryRecords(Long feedId) {
        run(db -> db.deleteFrom(PERF_FEED_HISTORY)
                .where(PERF_FEED_HISTORY.FEED_ID.eq(feedId))
                .execute());
    }

    @Step("DB: удаление записи из таблицы ppc.feeds (feedId = {0}) и из всех связанных таблиц")
    public void deleteFeedAndAllLinksByFeedId(Long feedId) {
        deleteAdgroupsPerformanceRecords(feedId);
        deleteAdgroupsDynamicRecords(feedId);
        deletePerfFeedVendorsRecords(feedId);
        deletePerfFeedCategoriesRecords(feedId);
        deletePerfFeedHistoryRecords(feedId);
        deleteFeedById(feedId);
    }

    public List<PerfFeedCategoriesRecord> getPerfFeedCategoriesRecords(long feedId) {
        return exec(db -> db.selectFrom(PERF_FEED_CATEGORIES)
                .where(PERF_FEED_CATEGORIES.FEED_ID.eq(feedId))
                .fetch());
    }

    public List<PerfFeedVendorsRecord> getPerfFeedVendorsRecords(long feedId) {
        return exec(db -> db.selectFrom(PERF_FEED_VENDORS)
                .where(PERF_FEED_VENDORS.FEED_ID.eq(feedId))
                .fetch());
    }

    public List<PerfFeedHistoryRecord> getPerfFeedHistoryRecords(long feedId) {
        return exec(db -> db.selectFrom(PERF_FEED_HISTORY)
                .where(PERF_FEED_HISTORY.FEED_ID.eq(feedId))
                .fetch());
    }

    public String getFeedTargetDomain(long feedId) {
        return exec(db -> db.select(FEEDS.TARGET_DOMAIN)
                .from(FEEDS)
                .where(FEEDS.FEED_ID.eq(feedId))
                .fetchOne().value1());
    }
}
