package ru.yandex.autotests.direct.db.steps;

import java.util.List;

import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.ClientsRelationsType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.MobileAppsStoreType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.ReverseClientsRelationsType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.ClientsRelationsRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.MobileAppGoalsExternalTrackerRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.MobileAppTrackersRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.MobileAppsRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.ReverseClientsRelationsRecord;
import ru.yandex.autotests.direct.db.steps.base.BasePpcSteps;
import ru.yandex.autotests.direct.utils.BaseSteps;
import ru.yandex.qatools.allure.annotations.Step;

import static ru.yandex.autotests.direct.db.models.jooq.ppc.Tables.CLIENTS_RELATIONS;
import static ru.yandex.autotests.direct.db.models.jooq.ppc.Tables.MOBILE_APPS;
import static ru.yandex.autotests.direct.db.models.jooq.ppc.Tables.MOBILE_APP_GOALS_EXTERNAL_TRACKER;
import static ru.yandex.autotests.direct.db.models.jooq.ppc.Tables.MOBILE_APP_TRACKERS;
import static ru.yandex.autotests.direct.db.models.jooq.ppc.Tables.REVERSE_CLIENTS_RELATIONS;

public class MobileAppSteps extends BasePpcSteps {
    @Step("DB: добавление записи в таблицу ppc.mobile_apps")
    public Long createMobileApp(MobileAppsRecord mobileApp) {
        Long id = autoIncSteps().getNewMobileAppId();
        run(db -> db.insertInto(MOBILE_APPS)
                .set(mobileApp.setMobileAppId(id))
                .execute());
        return id;
    }

    @Step("DB: получение приложения mobileAppId={0} из таблицы ppc.mobile_apps")
    public MobileAppsRecord getMobileAppById(Long mobileAppId) {
        return exec(db -> db.selectFrom(MOBILE_APPS)
                .where(MOBILE_APPS.MOBILE_APP_ID.eq(mobileAppId))
                .fetchOne());
    }

    @Step("DB: получение списка приложений клиента clientId={0} из таблицы ppc.mobile_apps")
    public List<MobileAppsRecord> getMobileAppsByClient(Long clientId) {
        return exec(db -> db.selectFrom(MOBILE_APPS)
                .where(MOBILE_APPS.CLIENTID.eq(clientId))
                .fetch());
    }

    @Step("DB: удаление приложения mobileAppId={0} и всех привязанных к нему трекеров")
    public void deleteMobileApp(Long mobileAppId) {
        deleteMobileAppTrackersByAppId(mobileAppId);
        exec(db -> db.deleteFrom(MOBILE_APPS)
                .where(MOBILE_APPS.MOBILE_APP_ID.eq(mobileAppId))
                .execute());
    }

    @Step("DB: удаление всех трекеров приложения mobileAppId={0} из таблицы ppc.mobile_app_trackers")
    public void deleteMobileAppTrackersByAppId(Long mobileAppId) {
        exec(db -> db.deleteFrom(MOBILE_APP_TRACKERS)
                .where(MOBILE_APP_TRACKERS.MOBILE_APP_ID.eq(mobileAppId))
                .execute());
    }

    @Step("DB: Добавление счетчика для приложения mobileAppId={0} в таблице ppc.mobile_app_trackers")
    public void addMobileAppTrackerForApp(MobileAppTrackersRecord mobileAppTrackersRecord) {
        mobileAppTrackersRecord.setMobileAppTrackerId(autoIncSteps().getNewMobileAppTrackerId());
        exec(db -> db.insertInto(MOBILE_APP_TRACKERS)
                .set(mobileAppTrackersRecord)
                .execute());
    }

    @Step("DB: Добавление цели для приложения mobileAppId={0} в таблицу ppc.mobile_app_goals_external_tracker")
    public Long addMobileAppGoal(Long mobileAppId) {
        return addMobileAppGoal(mobileAppId, "ADDED_PAYMENT_INFO");
    }

    @Step("DB: Добавление цели для приложения mobileAppId={0} в таблицу ppc.mobile_app_goals_external_tracker")
    public Long addMobileAppGoal(Long mobileAppId, String eventName) {
        MobileAppGoalsExternalTrackerRecord record = new MobileAppGoalsExternalTrackerRecord()
                .setGoalId(autoIncSteps().getNewMobileGoalId())
                .setMobileAppId(mobileAppId)
                .setEventName(eventName);
        exec(db -> db.insertInto(MOBILE_APP_GOALS_EXTERNAL_TRACKER)
                .set(record)
                .execute());
        return record.getGoalId();
    }

    @Step("DB: Удаление всех целей для приложения mobileAppId={0} из таблицы ppc.mobile_app_goals_external_tracker")
    public void deleteMobileAppGoals(Long mobileAppId) {
        exec(db -> db.deleteFrom(MOBILE_APP_GOALS_EXTERNAL_TRACKER)
                .where(MOBILE_APP_GOALS_EXTERNAL_TRACKER.MOBILE_APP_ID.eq(mobileAppId))
                .execute());
    }

    @Step("DB: Добавление доступ клиента {0} к мобильным целям клиента {1}")
    public void addMobileGoalsAccess(Long consumerOfMobileGoals, Long ownerOfMobileGoals) {
        int consumerShard = shardingSteps().getShardByClientID(consumerOfMobileGoals);
        int ownerShard = shardingSteps().getShardByClientID(ownerOfMobileGoals);

        new DirectJooqDbSteps().useShard(ownerShard).mobileAppsSteps()
                .addMobileGoalsAccessRelations(consumerOfMobileGoals, ownerOfMobileGoals);
        new DirectJooqDbSteps().useShard(consumerShard).mobileAppsSteps()
                .addMobileGoalsAccessReverseRelations(consumerOfMobileGoals, ownerOfMobileGoals);
    }

    @Step("DB: Добавление доступ клиента {0} к мобильным целям клиента {1} прямая связь")
    public void addMobileGoalsAccessRelations(Long consumerOfMobileGoals, Long ownerOfMobileGoals) {
        // shard by owner

        exec(db -> db.insertInto(CLIENTS_RELATIONS)
                .set(new ClientsRelationsRecord()
                        .setRelationId(autoIncSteps().getNextClientsRelationId())
                        .setClientIdFrom(consumerOfMobileGoals)
                        .setClientIdTo(ownerOfMobileGoals)
                        .setType(ClientsRelationsType.mobile_goals_access))
                .onDuplicateKeyIgnore()
                .execute());
    }

    @Step("DB: Добавление доступ клиента {0} к мобильным целям клиента {1} обратная связь")
    public void addMobileGoalsAccessReverseRelations(Long consumerOfMobileGoals, Long ownerOfMobileGoals) {
        // shard by consumer
        exec(db -> db.insertInto(REVERSE_CLIENTS_RELATIONS)
                .set(new ReverseClientsRelationsRecord()
                        .setReverseRelationId(autoIncSteps().getNextReverseClientsRelationId())
                        .setClientIdFrom(consumerOfMobileGoals)
                        .setClientIdTo(ownerOfMobileGoals)
                        .setType(ReverseClientsRelationsType.mobile_goals_access))
                .onDuplicateKeyIgnore()
                .execute());
    }

    public static MobileAppsRecord getDefaultMobileAppsRecord(Long clientId, Long mobileContentId) {
        return new MobileAppsRecord()
                .setClientid(clientId)
                .setStoreType(MobileAppsStoreType.GooglePlayStore)
                .setStoreHref("http://play.google.com/store/apps/details?id=ru.yandex.searchplugin")
                .setName("some app")
                .setMobileContentId(mobileContentId);
    }
}
