package ru.yandex.autotests.direct.db.steps;

import java.math.BigInteger;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.HierarchicalMultipliersType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.MobileMultiplierValuesOsType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BannerTypeMultiplierValuesRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.ExpressionMultiplierValuesRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.GeoMultiplierValuesRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.HierarchicalMultipliersRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.InventoryMultiplierValuesRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.MobileMultiplierValuesRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.WeatherMultiplierValuesRecord;
import ru.yandex.autotests.direct.db.steps.base.BasePpcSteps;
import ru.yandex.qatools.allure.annotations.Step;

import static ru.yandex.autotests.direct.db.models.jooq.ppc.Tables.EXPRESSION_MULTIPLIER_VALUES;
import static ru.yandex.autotests.direct.db.models.jooq.ppc.Tables.GEO_MULTIPLIER_VALUES;
import static ru.yandex.autotests.direct.db.models.jooq.ppc.Tables.HIERARCHICAL_MULTIPLIERS;
import static ru.yandex.autotests.direct.db.models.jooq.ppc.Tables.MOBILE_MULTIPLIER_VALUES;
import static ru.yandex.autotests.direct.db.models.jooq.ppc.Tables.WEATHER_MULTIPLIER_VALUES;
import static ru.yandex.autotests.direct.db.models.jooq.ppc.tables.BannerTypeMultiplierValues.BANNER_TYPE_MULTIPLIER_VALUES;
import static ru.yandex.autotests.direct.db.models.jooq.ppc.tables.InventoryMultiplierValues.INVENTORY_MULTIPLIER_VALUES;

public class MultipliersSteps extends BasePpcSteps {

    @Step("DB: чтение записи из таблицы ppc.hierarchical_multipliers, id = {0}")
    public HierarchicalMultipliersRecord getHierarchicalMultipliersById(Long hierarchicalMultiplierId) {
        return exec(db -> db.selectFrom(HIERARCHICAL_MULTIPLIERS)
                .where(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.eq(
                        BigInteger.valueOf(hierarchicalMultiplierId)))
                .fetchOne()
        );
    }

    @Step("DB: чтение записей из таблицы ppc.hierarchical_multipliers, cid = {0}")
    public List<HierarchicalMultipliersRecord> getHierarchicalMultipliersByCid(Long cid) {
        return exec(db -> db.selectFrom(HIERARCHICAL_MULTIPLIERS)
                .where(HIERARCHICAL_MULTIPLIERS.CID.eq(cid))
                .fetch()
        );
    }

    @Step("DB: чтение записей из таблицы ppc.hierarchical_multipliers, pid = {0}")
    public List<HierarchicalMultipliersRecord> getHierarchicalMultipliersByPid(Long pid) {
        return exec(db -> db.selectFrom(HIERARCHICAL_MULTIPLIERS)
                .where(HIERARCHICAL_MULTIPLIERS.PID.eq(pid))
                .fetch()
        );
    }

    @Step("DB: добавление записи в таблицу ppc.hierarchical_multipliers")
    public BigInteger saveHierarchicalMultipliers(HierarchicalMultipliersRecord record) {
        BigInteger hierarchicalMultiplierId = autoIncSteps().getNewHierarchicalMultipliersId();
        run(db -> db.insertInto(HIERARCHICAL_MULTIPLIERS)
                .set(record
                        .setHierarchicalMultiplierId(hierarchicalMultiplierId))
                .execute()
        );
        return hierarchicalMultiplierId;
    }


    @Step("DB: чтение записей из таблицы ppc.geo_multiplier_values, hierarchical_multiplier_id = {0}")
    public List<GeoMultiplierValuesRecord> getGeoMultiplierValuesByHierarchicalMultipliersId(Long id) {
        return exec(db -> db.selectFrom(GEO_MULTIPLIER_VALUES)
                .where(GEO_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID.eq(BigInteger.valueOf(id)))
                .fetch()
        );
    }

    @Step("DB: создание записей в ppc.weather_multiplier_values")
    public void saveWeatherMultiplierValues(WeatherMultiplierValuesRecord record) {
        run(db -> db.insertInto(WEATHER_MULTIPLIER_VALUES)
                .set(record)
                .execute());
    }

    @Step("DB: создание записей в таблицах ppc.hierarchical_multipliers и ppc.weather_multiplier_values")
    public void saveWeatherMultiplierValues(Long cid, @Nullable Long pid, List<WeatherMultiplierValuesRecord> weatherMultipliers, boolean isEnabled) {
        BigInteger hierarchicalMultipliersId = saveHierarchicalMultipliers(
                new HierarchicalMultipliersRecord()
                        .setCid(cid)
                        .setPid(pid)
                        .setIsEnabled(isEnabled ? 1 : 0)
                        .setSynteticKeyHash(BigInteger.valueOf(System.currentTimeMillis()))
                        .setType(HierarchicalMultipliersType.weather_multiplier)
        );

        weatherMultipliers.forEach(multiplier -> {
            multiplier.setHierarchicalMultiplierId(hierarchicalMultipliersId);
            multiplier.setWeatherMultiplierValueId(autoIncSteps().getNewHierarchicalMultipliersId());
            saveWeatherMultiplierValues(multiplier);
        });
    }

    @Step("DB: создание записей в ppc.expression_multiplier_values")
    public void saveExpressionMultiplierValues(ExpressionMultiplierValuesRecord record) {
        run(db -> db.insertInto(EXPRESSION_MULTIPLIER_VALUES)
                .set(record)
                .execute());
    }

    @Step("DB: создание записей в таблицах ppc.hierarchical_multipliers и ppc.expression_multiplier_values")
    public void saveExpressionMultiplierValues(Long cid, @Nullable Long pid,
                                               List<ExpressionMultiplierValuesRecord> multipliers,
                                               HierarchicalMultipliersType type,
                                               boolean isEnabled) {
        BigInteger hierarchicalMultipliersId = saveHierarchicalMultipliers(
                new HierarchicalMultipliersRecord()
                        .setCid(cid)
                        .setPid(pid)
                        .setIsEnabled(isEnabled ? 1 : 0)
                        .setSynteticKeyHash(BigInteger.valueOf(System.currentTimeMillis()))
                        .setType(type)
        );

        multipliers.forEach(multiplier -> {
            multiplier.setHierarchicalMultiplierId(hierarchicalMultipliersId);
            multiplier.setExpressionMultiplierValueId(autoIncSteps().getNewHierarchicalMultipliersId());
            saveExpressionMultiplierValues(multiplier);
        });
    }

    @Step("DB: чтение записи из таблицы ppc.geo_multiplier_values, geo_multiplier_value_id = {0}")
    public GeoMultiplierValuesRecord getGeoMultiplierValuesById(Long id) {
        return exec(db -> db.selectFrom(GEO_MULTIPLIER_VALUES)
                .where(GEO_MULTIPLIER_VALUES.GEO_MULTIPLIER_VALUE_ID.eq(BigInteger.valueOf(id)))
                .fetchOne()
        );
    }

    @Step("DB: добавление записи в таблицу ppc.geo_multiplier_values")
    public void saveGeoMultiplierValues(GeoMultiplierValuesRecord record) {
        run(db -> db.insertInto(GEO_MULTIPLIER_VALUES)
                .set(record
                        .setGeoMultiplierValueId(autoIncSteps().getNewHierarchicalMultipliersId()))
                .execute()
        );
    }

    @Step("DB: добавление записи в таблицу ppc.geo_multiplier_values")
    public void saveGeoMultiplierValues(Long cid, Map<Long, Short> multipliers) {
        saveGeoMultiplierValues(cid, true, multipliers);
    }

    @Step("DB: добавление записи в таблицу ppc.geo_multiplier_values")
    public void saveGeoMultiplierValues(Long cid, boolean isEnabled,
            Map<Long, Short> multipliers)
    {
        BigInteger hierarchicalMultipliersId = saveHierarchicalMultipliers(
                new HierarchicalMultipliersRecord()
                        .setCid(cid)
                        .setIsEnabled(isEnabled ? 1 : 0)
                        .setSynteticKeyHash(BigInteger.valueOf(System.currentTimeMillis()))
                        .setType(HierarchicalMultipliersType.geo_multiplier)
        );

        multipliers.forEach((regonId, multiplierPct) -> saveGeoMultiplierValues(
                new GeoMultiplierValuesRecord()
                        .setHierarchicalMultiplierId(hierarchicalMultipliersId)
                        .setRegionId(regonId)
                        .setMultiplierPct(multiplierPct)
                )
        );
    }

    @Step("DB: добавление записи в таблицу ppc.geo_multiplier_values")
    public void saveGeoMultiplierRecords(Long cid, List<GeoMultiplierValuesRecord> multipliers) {
        saveGeoMultiplierRecords(cid, true, multipliers);
    }

    @Step("DB: добавление записи в таблицу ppc.geo_multiplier_values")
    public void saveGeoMultiplierRecords(Long cid, boolean isEnabled,
            List<GeoMultiplierValuesRecord> multipliers)
    {
        BigInteger hierarchicalMultipliersId = saveHierarchicalMultipliers(
                new HierarchicalMultipliersRecord()
                        .setCid(cid)
                        .setIsEnabled(isEnabled ? 1 : 0)
                        .setSynteticKeyHash(BigInteger.valueOf(System.currentTimeMillis()))
                        .setType(HierarchicalMultipliersType.geo_multiplier)
        );

        multipliers.forEach((multiplier) -> saveGeoMultiplierValues(
                multiplier.setHierarchicalMultiplierId(hierarchicalMultipliersId)));
    }

    @Step("DB: чтение записей из таблицы ppc.mobile_multiplier_values, hierarchical_multiplier_id={0}")
    public List<MobileMultiplierValuesRecord> getMobileMultiplierValuesByHierarchicalMultipliersId(BigInteger id) {
        return exec(db -> db.selectFrom(MOBILE_MULTIPLIER_VALUES)
                .where(MOBILE_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID.eq(id))
                .fetch()
        );
    }

    @Step("DB: чтение записей из таблицы ppc.mobile_multiplier_values, adGroupId={0}")
    public List<MobileMultiplierValuesRecord> getMobileMultiplierValuesByAdGroupId(Long adGroupId) {
        return exec(db -> db.select(MOBILE_MULTIPLIER_VALUES.fields())
                .from(MOBILE_MULTIPLIER_VALUES)
                .join(HIERARCHICAL_MULTIPLIERS)
                .on(MOBILE_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID
                        .eq(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID))
                .where(HIERARCHICAL_MULTIPLIERS.PID.eq(adGroupId))
                .fetchInto(MobileMultiplierValuesRecord.class)
        );
    }

    public void saveCampaignPerformanceTgoMultiplier(long cid, Short multiplierPct) {
        savePerformanceTgoMultiplier(cid, null, multiplierPct);
    }

    public void saveGroupPerformanceTgoMultiplier(long cid, long pid, Short multiplierPct) {
        savePerformanceTgoMultiplier(cid, pid, multiplierPct);
    }

    private void savePerformanceTgoMultiplier(long cid, Long pid, Short multiplierPct) {
        saveHierarchicalMultipliers(
                new HierarchicalMultipliersRecord()
                        .setType(HierarchicalMultipliersType.performance_tgo_multiplier)
                        .setCid(cid)
                        .setPid(pid)
                        .setIsEnabled(1)
                        .setSynteticKeyHash(BigInteger.valueOf(System.currentTimeMillis()))
                        .setMultiplierPct(multiplierPct)
        );
    }

    public void saveCampaignDesktopMultiplier(long cid, Short multiplierPct) {
        saveDesktopMultiplier(cid, null, multiplierPct);
    }

    public void saveGroupDesktopMultiplier(long cid, long pid, Short multiplierPct) {
        saveDesktopMultiplier(cid, pid, multiplierPct);
    }

    private void saveDesktopMultiplier(long cid, Long pid, Short multiplierPct) {
        saveHierarchicalMultipliers(
                new HierarchicalMultipliersRecord()
                        .setType(HierarchicalMultipliersType.desktop_multiplier)
                        .setCid(cid)
                        .setPid(pid)
                        .setIsEnabled(1)
                        .setSynteticKeyHash(BigInteger.valueOf(System.currentTimeMillis()))
                        .setMultiplierPct(multiplierPct)
        );
    }

    @Step("DB: добавление записи в таблицу ppc.mobile_multiplier_values")
    public void saveMobileMultiplierValues(MobileMultiplierValuesRecord record) {
        run(db -> db.insertInto(MOBILE_MULTIPLIER_VALUES)
                .set(record
                        .setMobileMultiplierValueId(autoIncSteps().getNewHierarchicalMultipliersId()))
                .execute()
        );
    }

    @Step("DB: добавление записи в таблицу ppc.mobile_multiplier_values")
    public void saveMobileMultiplierValues(Long cid, Short multiplier, MobileMultiplierValuesOsType osType)
    {
        BigInteger hierarchicalMultipliersId = saveHierarchicalMultipliers(
                new HierarchicalMultipliersRecord()
                        .setCid(cid)
                        .setIsEnabled(1)
                        .setSynteticKeyHash(BigInteger.valueOf(System.currentTimeMillis()))
                        .setType(HierarchicalMultipliersType.mobile_multiplier)
        );
        saveMobileMultiplierValues(
                new MobileMultiplierValuesRecord()
                        .setHierarchicalMultiplierId(hierarchicalMultipliersId)
                        .setOsType(osType)
                        .setMultiplierPct(multiplier)
        );
    }

    @Step("DB: создание записи в таблице ppc.inventory_multiplier_values")
    public InventoryMultiplierValuesRecord saveInventoryModifierValues(
            InventoryMultiplierValuesRecord bidModifierRecord)
    {
        return exec(db -> db.insertInto(INVENTORY_MULTIPLIER_VALUES)
                .set(bidModifierRecord)
                .returning()
                .fetchOne());
    }

    @Step("DB: создание записи в таблице ppc.inventory_multiplier_values")
    public void saveInventoryModifierValues(Long cid, boolean isEnabled,
            List<InventoryMultiplierValuesRecord> multipliers)
    {
        BigInteger hierarchicalMultipliersId = saveHierarchicalMultipliers(
                new HierarchicalMultipliersRecord()
                        .setCid(cid)
                        .setIsEnabled(isEnabled ? 1 : 0)
                        .setSynteticKeyHash(BigInteger.valueOf(System.currentTimeMillis()))
                        .setType(HierarchicalMultipliersType.inventory_multiplier)
        );

        multipliers.forEach(multiplier -> {
            multiplier.setHierarchicalMultiplierId(hierarchicalMultipliersId);
            multiplier.setInventoryMultiplierValueId(autoIncSteps().getNewHierarchicalMultipliersId());
            saveInventoryModifierValues(multiplier);
        });
    }

    @Step("DB: создание записи в таблице ppc.banner_type_multiplier_values")
    public void saveBannerTypeModifierValues(BannerTypeMultiplierValuesRecord bidModifierRecord) {
        exec(db -> db.insertInto(BANNER_TYPE_MULTIPLIER_VALUES)
                .set(bidModifierRecord)
                .returning()
                .fetchOne());
    }

    @Step("DB: создание записи в таблице ppc.inventory_multiplier_values")
    public void saveBannerTypeModifierValues(Long cid, boolean isEnabled,
            List<BannerTypeMultiplierValuesRecord> multipliers)
    {
        BigInteger hierarchicalMultipliersId = saveHierarchicalMultipliers(
                new HierarchicalMultipliersRecord()
                        .setCid(cid)
                        .setIsEnabled(isEnabled ? 1 : 0)
                        .setSynteticKeyHash(BigInteger.valueOf(System.currentTimeMillis()))
                        .setType(HierarchicalMultipliersType.banner_type_multiplier)
        );

        multipliers.forEach(multiplier -> {
            multiplier.setHierarchicalMultiplierId(hierarchicalMultipliersId);
            multiplier.setBannerTypeMultiplierValueId(autoIncSteps().getNewHierarchicalMultipliersId());
            saveBannerTypeModifierValues(multiplier);
        });
    }

    public void deleteAllMultipliers(Long cid) {
        exec(db -> db.deleteFrom(HIERARCHICAL_MULTIPLIERS)
                .where(HIERARCHICAL_MULTIPLIERS.CID.eq(cid))
                .execute()
        );
    }
}
