package ru.yandex.autotests.direct.db.steps;

import java.sql.Timestamp;
import java.util.Arrays;
import java.util.List;

import org.joda.time.DateTime;

import ru.yandex.autotests.direct.db.beans.creatives.CreativeTemplate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.PerfCreativesCreativeType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.PerfCreativesStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.PerfCreativesRecord;
import ru.yandex.autotests.direct.db.steps.base.BasePpcSteps;
import ru.yandex.qatools.allure.annotations.Step;

import static ru.yandex.autotests.direct.db.models.jooq.ppc.tables.PerfCreatives.PERF_CREATIVES;


/**
 * Степы для работы с таблицей ppc.perf_creatives
 * Реализованы получение, удаление. обновление(включая обновление некоторых полей),
 * добавление записей, включая добавление креативов фиксированных типов
 */
public class PerfCreativesSteps extends BasePpcSteps {
    public static final String DEFAULT_PREVIEW_URL =
            "https://cdn-austrian.economicblogs.org/wp-content/uploads/2016/09/AdobeStock_58349892-300x300.jpeg";
    public static final String DEFAULT_LIVE_PREVIEW_URL = "https://ya.ru/preview_12343";
    private static final Long DEFAULT_CPC_VIDEO_LAYOUT_ID = 51L;
    private static final Long DEFAULT_PERF_CREATIVE_LAYOUT_ID = 45L;
    private static final Long TGO_LAYOUT_ID = 44L;

    @Step("DB: получение записи из таблицы ppc.perf_creatives (creative_id = {0}")
    public PerfCreativesRecord getPerfCreatives(long creativeId) {
        return exec(db ->
                db.selectFrom(PERF_CREATIVES)
                        .where(PERF_CREATIVES.CREATIVE_ID.eq(creativeId))
                        .fetchOne());
    }

    @Step("DB: получение списка записей ppc.perf_creatives для пользователя (clientID = {0}")
    public List<PerfCreativesRecord> getPerfCreativesByClientId(long clientId) {
        return exec(db ->
                db.selectFrom(PERF_CREATIVES)
                        .where(PERF_CREATIVES.CLIENTID.eq(clientId))
                        .fetch());
    }

    @Step("DB: получение списка записей определенного типа ppc.perf_creatives для пользователя (clientID = {0}")
    public List<PerfCreativesRecord> getPerfCreativesByClientId(long clientId, PerfCreativesCreativeType creativeType) {
        return exec(db ->
                db.selectFrom(PERF_CREATIVES)
                        .where(PERF_CREATIVES.CLIENTID.eq(clientId))
                        .and(PERF_CREATIVES.CREATIVE_TYPE.eq(creativeType))
                        .fetch());
    }

    @Step("DB: удаление записей в ppc.perf_creatives для пользователя (clientID = {0}")
    public void deletePerfCreativesByClientId(long clientId) {
        run(db ->
                db.deleteFrom(PERF_CREATIVES)
                        .where(PERF_CREATIVES.CLIENTID.eq(clientId))
                        .execute());
    }

    @Step("DB: изменение записи в таблице ppc.perf_creatives")
    public void updatePerfCreatives(PerfCreativesRecord perfCreatives) {
        run(db ->
                db.update(PERF_CREATIVES)
                        .set(perfCreatives)
                        .where(PERF_CREATIVES.CREATIVE_ID.eq(perfCreatives.getCreativeId()))
                        .execute());
    }

    @Step("DB: создание записи в таблице ppc.perf_creatives")
    public void savePerfCreatives(PerfCreativesRecord perfCreatives) {
        run(db ->
                db.insertInto(PERF_CREATIVES)
                        .set(perfCreatives)
                        .returning(PERF_CREATIVES.fields())
                        .fetchOne());
    }

    @Step("DB: обновляем поле moderate_info в таблице ppc.perf_creatives для creative_id = {0}")
    public void setModerateInfo(Long creativeId, String moderateInfo) {
        run(db -> db.update(PERF_CREATIVES)
                .set(PERF_CREATIVES.MODERATE_INFO, moderateInfo)
                .where(PERF_CREATIVES.CREATIVE_ID.eq(creativeId))
                .execute());
    }

    @Step("DB: обновляем поля width, height в таблице ppc.perf_creatives для creative_id = {0}")
    public void setWidthHeight(Long creativeId, Integer width, Integer height) {
        run(db -> db.update(PERF_CREATIVES)
                .set(PERF_CREATIVES.WIDTH, width.shortValue())
                .set(PERF_CREATIVES.HEIGHT, height.shortValue())
                .where(PERF_CREATIVES.CREATIVE_ID.eq(creativeId))
                .execute());
    }

    @Step("DB: обновляем поле creative_group_id в таблице ppc.perf_creatives для creative_id = {0}")
    public void setCreativeGroupId(Long creativeId, Long creativeGroupId) {
        run(db -> db.update(PERF_CREATIVES)
                .set(PERF_CREATIVES.CREATIVE_GROUP_ID, creativeGroupId)
                .where(PERF_CREATIVES.CREATIVE_ID.eq(creativeId))
                .execute());
    }

    @Step("DB: обновляем поле status_moderate в таблице ppc.perf_creatives для creative_id = {0}, status_moderate = {1}")
    public void setStatusModerate(Long creativeId, PerfCreativesStatusmoderate statusModerate) {
        run(db -> db.update(PERF_CREATIVES)
                .set(PERF_CREATIVES.STATUSMODERATE, statusModerate)
                .where(PERF_CREATIVES.CREATIVE_ID.eq(creativeId))
                .execute());
    }

    public void setDefaultCpmPriceCampCreativeSize(Long creativeId) {
        setWidthHeight(creativeId, 1456, 180);
    }

    @Step("DB: создание записи в таблице ppc.perf_creatives (+ в ppcdict.shard_creative_id) (clientId = {0})")
    public Long saveDefaultCanvasCreativesForClient(long clientId) {
        Long creativeId = super.autoIncSteps().getNewBannerCreativeId();
        PerfCreativesRecord perfCreatives = getDefaultCanvasCreative(creativeId, clientId);
        savePerfCreatives(perfCreatives);
        getInstance(CreativesShardingSteps.class, getContext())
                .addCreativeClientLink(clientId, perfCreatives.getCreativeId());
        return perfCreatives.getCreativeId();
    }

    @Step("DB: создание записи в таблице ppc.perf_creatives (+ в ppcdict.shard_creative_id) (clientId = {0})")
    public Long saveDefaultHtml5CreativesForClient(long clientId) {
        Long creativeId = super.autoIncSteps().getNewBannerCreativeId();
        PerfCreativesRecord perfCreatives = getDefaultHtml5CreativeRecord(creativeId, clientId);
        savePerfCreatives(perfCreatives);
        getInstance(CreativesShardingSteps.class, getContext())
                .addCreativeClientLink(clientId, perfCreatives.getCreativeId());
        return perfCreatives.getCreativeId();
    }

    @Step("DB: создание записи в таблице ppc.perf_creatives (+ в ppcdict.shard_creative_id) "
            + "(clientId = {0}, width = {1}, height = {2})")
    public Long saveDefaultHtml5CreativesForClientWithSize(long clientId, short width, short height) {
        Long creativeId = super.autoIncSteps().getNewBannerCreativeId();
        PerfCreativesRecord perfCreatives = getDefaultHtml5CreativeRecord(creativeId, clientId)
                .setWidth(width)
                .setHeight(height);
        savePerfCreatives(perfCreatives);
        getInstance(CreativesShardingSteps.class, getContext())
                .addCreativeClientLink(clientId, perfCreatives.getCreativeId());
        return perfCreatives.getCreativeId();
    }

    @Step("DB: создание записи в таблице ppc.perf_creatives (+ в ppcdict.shard_creative_id) (clientId = {1})")
    public PerfCreativesRecord savePerfCreativesForClient(PerfCreativesRecord perfCreatives, long clientId) {
        savePerfCreatives(perfCreatives);
        getInstance(CreativesShardingSteps.class, getContext())
                .addCreativeClientLink(clientId, perfCreatives.getCreativeId());
        return getPerfCreatives(perfCreatives.getCreativeId());
    }

    public Long saveDefaultPerfCreativesForClient(String name, CreativeTemplate template,
            long clientId)
    {
        Long creativeId = super.autoIncSteps().getNewBannerCreativeId();
        PerfCreativesRecord perfCreatives = getDefaultPerfomanceCreativeRecord(name, template, creativeId, clientId,
                DEFAULT_PERF_CREATIVE_LAYOUT_ID);
        savePerfCreativesForClient(perfCreatives, clientId);
        return creativeId;
    }

    @Step("DB: удаление записи из таблицы ppc.perf_creatives")
    public void deletePerfCreatives(List<Long> ids) {
        run(db ->
                db.deleteFrom(PERF_CREATIVES)
                        .where(PERF_CREATIVES.CREATIVE_ID.in(ids))
                        .execute());
    }

    public void deletePerfCreatives(Long... ids) {
        deletePerfCreatives(Arrays.asList(ids));
    }

    public Long saveDefaultPerfCreative(String clientId) {
        return saveDefaultPerfCreative(Long.valueOf(clientId));
    }

    public Long saveDefaultPerfCreative(Long clientId) {
        Long creativeId = super.autoIncSteps().getNewBannerCreativeId();
        PerfCreativesRecord perfCreatives = getDefaultPerfomanceCreativeRecord(creativeId, clientId,
                DEFAULT_PERF_CREATIVE_LAYOUT_ID);
        savePerfCreativesForClient(perfCreatives, clientId);
        return perfCreatives.getCreativeId();
    }

    private PerfCreativesRecord getDefaultPerfomanceCreativeRecord(long creativeId, long clientId, long layoutId) {
        PerfCreativesRecord perfCreatives = new PerfCreativesRecord();
        perfCreatives.setCreativeType(PerfCreativesCreativeType.performance);
        perfCreatives.setCreativeId(creativeId);
        perfCreatives.setStockCreativeId(creativeId);
        perfCreatives.setClientid(clientId);
        perfCreatives.setWidth((short) 240);
        perfCreatives.setHeight((short) 400);
        perfCreatives.setName("testName");
        perfCreatives.setPreviewUrl("ya.ru");
        perfCreatives.setStatusmoderate(PerfCreativesStatusmoderate.New);
        perfCreatives.setModerateSendTime(new Timestamp(DateTime.now().toDate().getTime()));
        perfCreatives.setModerateTryCount((short) 0);
        perfCreatives.setTemplateId(CreativeTemplate.MANY_OFFERS_MANY_DESCRIPTIONS_AND_CAROUSEL_240x400.getId());
        perfCreatives.setLayoutId(layoutId);
        return perfCreatives;
    }

    private PerfCreativesRecord getDefaultCanvasCreative(long creativeId, long clientId) {
        PerfCreativesRecord perfCreatives = new PerfCreativesRecord();
        perfCreatives.setCreativeType(PerfCreativesCreativeType.canvas);
        perfCreatives.setCreativeId(creativeId);
        perfCreatives.setStockCreativeId(creativeId);
        perfCreatives.setClientid(clientId);
        perfCreatives.setWidth((short) 240);
        perfCreatives.setHeight((short) 400);
        perfCreatives.setName("testName");
        perfCreatives.setPreviewUrl("ya.ru");
        perfCreatives.setLivePreviewUrl("ya.ru");
        perfCreatives.setStatusmoderate(PerfCreativesStatusmoderate.New);
        perfCreatives.setModerateSendTime(new Timestamp(DateTime.now().toDate().getTime()));
        perfCreatives.setModerateTryCount((short) 0);
        perfCreatives.setTemplateId(null);
        return perfCreatives;
    }

    private PerfCreativesRecord getDefaultPerfomanceCreativeRecord(String name, CreativeTemplate template,
            Long creativeId, long clientId, long layoutId)
    {
        PerfCreativesRecord perfCreatives = new PerfCreativesRecord();
        perfCreatives.setCreativeId(creativeId);
        perfCreatives.setStockCreativeId(creativeId);
        perfCreatives.setClientid(clientId);
        perfCreatives.setWidth(template.getWidth().shortValue());
        perfCreatives.setHeight(template.getHeight().shortValue());
        perfCreatives.setName(name);
        perfCreatives.setPreviewUrl("ya.ru");
        perfCreatives.setStatusmoderate(PerfCreativesStatusmoderate.New);
        perfCreatives.setModerateSendTime(new Timestamp(DateTime.now().toDate().getTime()));
        perfCreatives.setModerateTryCount((short) 0);
        perfCreatives.setTemplateId(template.getId());
        perfCreatives.setLayoutId(layoutId);
        return perfCreatives;
    }

    @Step("DB: создание видеоподложки в таблице ppc.perf_creatives (+ в ppcdict.shard_creative_id) (clientId = {0})")
    public Long saveDefaultVideoCreative(Long clientId, long layoutId) {
        Long creativeId = super.autoIncSteps().getNewBannerCreativeId();
        PerfCreativesRecord videoCreative = getDefaultVideoCreative(creativeId, clientId, layoutId);
        savePerfCreativesForClient(videoCreative, clientId);
        return videoCreative.getCreativeId();
    }

    @Step("DB: создание аудиокреатива в таблице ppc.perf_creatives (+ в ppcdict.shard_creative_id) (clientId = {0})")
    public Long saveDefaultCpmAudioCreative(Long clientId) {
        // 301 -- layout из интервала, принадлежащего аудиокреативам
        return saveDefaultVideoCreative(clientId, 301L);
    }

    @Step("DB: создание кликового видео в таблице ppc.perf_creatives (+ в ppcdict.shard_creative_id) (clientId = {0})")
    public Long saveDefaultCpcVideoCreative(Long clientId) {
        Long creativeId = super.autoIncSteps().getNewBannerCreativeId();
        PerfCreativesRecord videoCreative = getDefaultVideoCreative(creativeId, clientId, DEFAULT_CPC_VIDEO_LAYOUT_ID);
        savePerfCreativesForClient(videoCreative, clientId);
        return videoCreative.getCreativeId();
    }

    private PerfCreativesRecord getDefaultVideoCreative(long creativeId, long clientId, long layoutId) {
        PerfCreativesRecord perfCreatives = new PerfCreativesRecord();
        perfCreatives.setCreativeType(PerfCreativesCreativeType.video_addition);
        perfCreatives.setCreativeId(creativeId);
        perfCreatives.setStockCreativeId(creativeId);
        perfCreatives.setClientid(clientId);
        perfCreatives.setName("testVideoAdditionName");
        perfCreatives.setPreviewUrl(DEFAULT_PREVIEW_URL);
        perfCreatives.setStatusmoderate(PerfCreativesStatusmoderate.New);
        perfCreatives.setModerateSendTime(new Timestamp(DateTime.now().toDate().getTime()));
        perfCreatives.setModerateTryCount((short) 0);
        perfCreatives.setLivePreviewUrl(DEFAULT_LIVE_PREVIEW_URL);
        perfCreatives.setDuration(15L);
        perfCreatives.setLayoutId(layoutId);
        return perfCreatives;
    }

    private PerfCreativesRecord getDefaultHtml5CreativeRecord(long creativeId, long clientId) {
        PerfCreativesRecord perfCreatives = new PerfCreativesRecord();
        perfCreatives.setCreativeType(PerfCreativesCreativeType.html5_creative);
        perfCreatives.setCreativeId(creativeId);
        perfCreatives.setStockCreativeId(creativeId);
        perfCreatives.setClientid(clientId);
        perfCreatives.setWidth((short) 240);
        perfCreatives.setHeight((short) 400);
        perfCreatives.setName("testName");
        perfCreatives.setPreviewUrl("ya.ru");
        perfCreatives.setLivePreviewUrl("ya.ru");
        perfCreatives.setStatusmoderate(PerfCreativesStatusmoderate.New);
        perfCreatives.setModerateSendTime(new Timestamp(DateTime.now().toDate().getTime()));
        perfCreatives.setModerateTryCount((short) 0);
        perfCreatives.setTemplateId(CreativeTemplate.MANY_OFFERS_MANY_DESCRIPTIONS_AND_CAROUSEL_240x400.getId());
        return perfCreatives;
    }

    public Long saveDefaultPerfTGOCreative(Long clientId) {
        Long creativeId = super.autoIncSteps().getNewBannerCreativeId();
        PerfCreativesRecord creative = getDefaultPerfomanceCreativeRecord(creativeId, clientId, TGO_LAYOUT_ID);
        savePerfCreativesForClient(creative, clientId);
        return creative.getCreativeId();
    }
}
