package ru.yandex.autotests.direct.db.steps;

import ru.yandex.autotests.direct.db.models.jooq.ppcdict.tables.ShardCreativeId;
import ru.yandex.autotests.direct.db.models.jooq.ppcdict.tables.ShardLogin;
import ru.yandex.autotests.direct.db.models.jooq.ppcdict.tables.ShardUid;
import ru.yandex.autotests.direct.db.models.jooq.ppcdict.tables.records.ShardClientIdRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppcdict.tables.records.ShardIncStrategyIdRecord;
import ru.yandex.autotests.direct.db.steps.base.BasePpcDictSteps;
import ru.yandex.autotests.direct.db.steps.base.DirectDbStepsException;
import ru.yandex.qatools.allure.annotations.Step;

import static ru.yandex.autotests.direct.db.models.jooq.ppcdict.tables.ShardClientId.SHARD_CLIENT_ID;
import static ru.yandex.autotests.direct.db.models.jooq.ppcdict.tables.ShardIncBid.SHARD_INC_BID;
import static ru.yandex.autotests.direct.db.models.jooq.ppcdict.tables.ShardIncCid.SHARD_INC_CID;
import static ru.yandex.autotests.direct.db.models.jooq.ppcdict.tables.ShardIncPid.SHARD_INC_PID;
import static ru.yandex.autotests.direct.db.models.jooq.ppcdict.tables.ShardIncStrategyId.SHARD_INC_STRATEGY_ID;
import static ru.yandex.autotests.direct.db.models.jooq.ppcdict.tables.ShardLogin.SHARD_LOGIN;
import static ru.yandex.autotests.direct.db.models.jooq.ppcdict.tables.ShardOrderId.SHARD_ORDER_ID;
import static ru.yandex.autotests.direct.db.models.jooq.ppcdict.tables.ShardUid.SHARD_UID;

/*
 * todo javadoc
 */
public class ShardingSteps extends BasePpcDictSteps {

    @Step("DB: получение шарда для логина {0}")
    public int getShardByLogin(String login) throws DirectDbStepsException {
        String normalizedLogin = login.replace(".", "-");
        return exec(db -> db.select(SHARD_CLIENT_ID.SHARD, SHARD_LOGIN.LOGIN).from(SHARD_CLIENT_ID)
                .join(SHARD_UID).on(SHARD_UID.CLIENTID.eq(SHARD_CLIENT_ID.CLIENTID))
                .join(SHARD_LOGIN).on(SHARD_LOGIN.UID.eq(SHARD_UID.UID))
                .where(SHARD_LOGIN.LOGIN.eq(normalizedLogin))
                .fetchOptional()
                .orElseThrow(() -> new DirectDbStepsException("Клиент '" + normalizedLogin + "' не найден в базе"))
                .value1());
    }

    @Step("Получение шарда для clientId = {0}")
    public Integer getShardByClientID(long clientID) throws DirectDbStepsException {
        return exec(db ->
                db.selectFrom(SHARD_CLIENT_ID)
                        .where(SHARD_CLIENT_ID.CLIENTID.eq(clientID))
                        .fetchOptional()
                        .orElseThrow(() -> new DirectDbStepsException("Не найден клиент с id " + clientID))
                        .getShard());
    }

    public Integer getShardByClientIDRaw(long clientID) {
        return silentExecution(db ->
                db.selectFrom(SHARD_CLIENT_ID)
                        .where(SHARD_CLIENT_ID.CLIENTID.eq(clientID))
                        .fetchOptional()
                        .orElse(new ShardClientIdRecord())
                        .getShard());
    }

    @Step("Получение шарда для bid = {0}")
    public Integer getShardByBid(long bid) throws DirectDbStepsException {
        return exec(db ->
                db.select().from(SHARD_INC_BID)
                        .join(SHARD_CLIENT_ID)
                        .on(SHARD_CLIENT_ID.CLIENTID.eq(SHARD_INC_BID.CLIENTID))
                        .where(SHARD_INC_BID.BID.eq(bid))
                        .fetchOptional()
                        .orElseThrow(() -> new DirectDbStepsException("Не найден баннер " + bid))
                        .getValue(SHARD_CLIENT_ID.SHARD));
    }

    @Step("Получение шарда для bid = {0}, 0 если шард не найден")
    public Integer getShardByBidOrZero(long bid) {
        return exec(db ->
                db.select().from(SHARD_INC_BID)
                        .join(SHARD_CLIENT_ID)
                        .on(SHARD_CLIENT_ID.CLIENTID.eq(SHARD_INC_BID.CLIENTID))
                        .where(SHARD_INC_BID.BID.eq(bid))
                        .fetchOptional()
                        .map(v -> v.get(SHARD_CLIENT_ID.SHARD))
                        .orElse(0));
    }

    @Step("Получение шарда для cid = {0}")
    public Integer getShardByCid(long cid) throws DirectDbStepsException {
        return exec(db ->
                db.select().from(SHARD_INC_CID)
                        .join(SHARD_CLIENT_ID)
                        .on(SHARD_CLIENT_ID.CLIENTID.eq(SHARD_INC_CID.CLIENTID))
                        .where(SHARD_INC_CID.CID.eq(cid))
                        .fetchOptional()
                        .orElseThrow(() -> new DirectDbStepsException("Не найдена кампания " + cid))
                        .getValue(SHARD_CLIENT_ID.SHARD));
    }

    @Step("Получение ClientID по метабазе для cid = {0}")
    public Long getClientIdByCid(Long cid) throws DirectDbStepsException {
        return exec(db ->
                db.select().from(SHARD_INC_CID)
                        .where(SHARD_INC_CID.CID.eq(cid))
                        .fetchOptional()
                        .orElseThrow(() -> new DirectDbStepsException("Не найдена кампания " + cid))
                        .getValue(SHARD_CLIENT_ID.CLIENTID)
        );
    }

    @Step("Получение собственника для cid = {0}")
    public String getOwnerForCid(long cid) throws DirectDbStepsException {
        return exec(db ->
                db.select(SHARD_LOGIN.LOGIN)
                        .from(SHARD_INC_CID)
                        .join(SHARD_UID).on(SHARD_UID.CLIENTID.eq(SHARD_INC_CID.CLIENTID))
                        .join(SHARD_LOGIN).on(SHARD_LOGIN.UID.eq(SHARD_UID.UID))
                        .where(SHARD_INC_CID.CID.eq(cid))
                        .fetchOptional()
                        .orElseThrow(() -> new DirectDbStepsException("Не найдена кампания " + cid))
                        .getValue(SHARD_LOGIN.LOGIN));
    }

    public Integer getShardByCidRaw(long cid) {
        return silentExecution(db ->
                db.select().from(SHARD_INC_CID)
                        .join(SHARD_CLIENT_ID)
                        .on(SHARD_CLIENT_ID.CLIENTID.eq(SHARD_INC_CID.CLIENTID))
                        .where(SHARD_INC_CID.CID.eq(cid))
                        .fetchOptional()
                        .orElse(new ShardClientIdRecord())
                        .getValue(SHARD_CLIENT_ID.SHARD));
    }

    @Step("Получение шарда для pid = {0}")
    public Integer getShardByPid(long pid) throws DirectDbStepsException {
        return exec(db ->
                db.select().from(SHARD_INC_PID)
                        .join(SHARD_CLIENT_ID)
                        .on(SHARD_CLIENT_ID.CLIENTID.eq(SHARD_INC_PID.CLIENTID))
                        .where(SHARD_INC_PID.PID.eq(pid))
                        .fetchOptional()
                        .orElseThrow(() -> new DirectDbStepsException("Не найдено информации о фразе " + pid))
                        .getValue(SHARD_CLIENT_ID.SHARD));
    }

    @Step("Получение шарда для orderID = {0}")
    public Integer getShardByOrderId(long orderID) throws DirectDbStepsException {
        return exec(db ->
                db.select().from(SHARD_ORDER_ID)
                        .join(SHARD_CLIENT_ID)
                        .on(SHARD_CLIENT_ID.CLIENTID.eq(SHARD_ORDER_ID.CLIENTID))
                        .where(SHARD_ORDER_ID.ORDERID.eq(orderID))
                        .fetchOptional()
                        .orElseThrow(() -> new DirectDbStepsException("Не найдено информации о заказе " + orderID))
                        .getValue(SHARD_CLIENT_ID.SHARD));
    }

    @Step("Получаем id последнего креатива") //работает только для performance креативов
    public Long getLastCreativeId() {
        return exec(db ->
                db.selectFrom(ShardCreativeId.SHARD_CREATIVE_ID)
                        .where(ShardCreativeId.SHARD_CREATIVE_ID.CREATIVE_ID.lessThan((long) Integer.MAX_VALUE / 2))
                        .orderBy(ShardCreativeId.SHARD_CREATIVE_ID.CREATIVE_ID.desc())
                        .limit(1)
                        .fetchOne()).getCreativeId();
    }

    @Step("Получаем clientId по логину {0}")
    public Long getClientIdByLogin(String login) {
        return exec(db -> db.select(ShardUid.SHARD_UID.CLIENTID).from(ShardLogin.SHARD_LOGIN)
                .join(ShardUid.SHARD_UID).on(ShardUid.SHARD_UID.UID.eq(ShardLogin.SHARD_LOGIN.UID))
                .where(ShardLogin.SHARD_LOGIN.LOGIN.eq(login))
                .fetchOne()
                .value1()
        );
    }

    @Step("Получаем uid по логину {0}")
    public Long getUidByLogin(String login) {
        return exec(db -> db.selectFrom(ShardLogin.SHARD_LOGIN)
                .where(ShardLogin.SHARD_LOGIN.LOGIN.eq(login))
                .fetchOne()
                .getUid()
        );
    }

    @Step("Проверяем, есть ли креатив в базе")
    public boolean isCreativeExist(Long creativeId) {
        return exec(db ->
                db.select(ShardCreativeId.SHARD_CREATIVE_ID.CREATIVE_ID)
                        .from(ShardCreativeId.SHARD_CREATIVE_ID)
                        .where(ShardCreativeId.SHARD_CREATIVE_ID.CREATIVE_ID.eq(creativeId))
                        .limit(1)
                        .fetch())
                .size() != 0;
    }

    @Step("DB: сохранение маппинга OrderID {0} в метабазу")
    public void createOrderIdMapping(Long orderId, Long clientId) {
        run(db -> db.insertInto(SHARD_ORDER_ID)
                .set(SHARD_ORDER_ID.ORDERID, orderId)
                .set(SHARD_ORDER_ID.CLIENTID, clientId)
                .onDuplicateKeyUpdate()
                .set(SHARD_ORDER_ID.CLIENTID, clientId)
                .execute()
        );
    }


    @Step("DB: получение следующиего значения pid")
    public Long getNextPid(Long clientId) {
        return exec(db -> db.insertInto(SHARD_INC_PID)
                .set(SHARD_INC_PID.CLIENTID, clientId)
                .returning()
                .fetchOne()
        ).getPid();
    }

    @Step("DB: получение следующиего значения bid")
    public Long getNextBid(Long clientId) {
        return exec(db -> db.insertInto(SHARD_INC_BID)
                .set(SHARD_INC_PID.CLIENTID, clientId)
                .returning()
                .fetchOne()
        ).getBid();
    }

    @Step("DB: удаление bid из метабазы")
    public void deleteBid(long bid) {
        run(db -> db.deleteFrom(SHARD_INC_BID)
                .where(SHARD_INC_BID.BID.eq(bid))
                .execute()
        );
    }

    @Step("DB: получение записи из таблицы ppcdict.shard_inc_strategy_id (strategy_id:{0}, ClientID:{1}")
    public ShardIncStrategyIdRecord getShardIncStrategyIdRecord(Long strategyId, Long clientId) {
        return exec(db -> db.selectFrom(SHARD_INC_STRATEGY_ID)
                .where(SHARD_INC_STRATEGY_ID.STRATEGY_ID.eq(strategyId))
                .and(SHARD_INC_STRATEGY_ID.CLIENTID.eq(clientId))
                .fetchOne());
    }
}
