package ru.yandex.autotests.direct.db.steps;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.gson.Gson;
import com.google.gson.JsonElement;

import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.StrategiesType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.StrategiesRecord;
import ru.yandex.autotests.direct.db.steps.base.BasePpcSteps;
import ru.yandex.qatools.allure.annotations.Step;

import static ru.yandex.autotests.direct.db.models.jooq.ppc.Tables.STRATEGIES;

@ParametersAreNonnullByDefault
public class StrategiesSteps extends BasePpcSteps {

    @Step("DB: получение записи из таблицы ppc.strategies (strategy_id:{0}")
    public StrategiesRecord getStrategiesRecord(Long strategyId) {
        return exec(db -> db.selectFrom(STRATEGIES)
                .where(STRATEGIES.STRATEGY_ID.eq(strategyId))
                .fetchOne());
    }

    @Step("DB: удаление стратегии strategy_id={0}")
    public void deleteStrategy(Long strategyId) {
        exec(db -> db.deleteFrom(STRATEGIES)
                .where(STRATEGIES.STRATEGY_ID.eq(strategyId))
                .execute());
    }

    @Step("DB: изменение записи в таблице ppc.strategies")
    public void updateStrategies(StrategiesRecord strategies) {
        run(db ->
                db.update(STRATEGIES)
                        .set(strategies)
                        .where(STRATEGIES.STRATEGY_ID.eq(strategies.getStrategyId()))
                        .execute());
    }

    @Step("DB: установка полей type и strategy_data")
    public void setStrategyData(Long strategyId, StrategiesType type, String strategyData) {
        StrategiesRecord strategiesRecord = getStrategiesRecord(strategyId);
        strategiesRecord.setType(type);
        strategiesRecord.setStrategyData(strategyData);
        updateStrategies(strategiesRecord);
    }

    @Step("DB: сохранение кц для стратегии strategy_id={0}")
    public void setStrategyMeaningfulGoals(Long strategyId, String meaningfulGoals) {
        StrategiesRecord strategiesRecord = getStrategiesRecord(strategyId);
        strategiesRecord.setMeaningfulGoals(meaningfulGoals);
        updateStrategies(strategiesRecord);
    }

    @Step("DB: установить ограничение ставки для автобюджета для strategy_id = {0}")
    public void setAutobudgetBid(long strategyId, double autobudgetBid) {
        StrategiesRecord strategiesRecord = getStrategiesRecord(strategyId);
        JsonElement strategyData = new Gson().fromJson(strategiesRecord.getStrategyData(), JsonElement.class);
        strategyData.getAsJsonObject().addProperty("bid", autobudgetBid);
        strategiesRecord.setStrategyData(strategyData.toString());
        updateStrategies(strategiesRecord);
    }

    @Step("DB: выставление поля pay_for_conversion в параметрах стратегии")
    public void setStrategyDataPayForConversion(Long strategyId, Long payForConversion) {
        StrategiesRecord strategiesRecord = getStrategiesRecord(strategyId);
        JsonElement strategyData = new Gson().fromJson(strategiesRecord.getStrategyData(), JsonElement.class);
        strategyData.getAsJsonObject().addProperty("pay_for_conversion", payForConversion);
        strategiesRecord.setStrategyData(strategyData.toString());
        updateStrategies(strategiesRecord);
    }
}
