package ru.yandex.autotests.direct.db.steps;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.TurbolandingMetrikaCountersRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.TurbolandingsRecord;
import ru.yandex.autotests.direct.db.steps.base.BasePpcSteps;
import ru.yandex.qatools.allure.annotations.Step;

import static org.jooq.impl.DSL.max;
import static ru.yandex.autotests.direct.db.models.jooq.ppc.Tables.TURBOLANDINGS;
import static ru.yandex.autotests.direct.db.models.jooq.ppc.Tables.TURBOLANDING_METRIKA_COUNTERS;

/**
 * Created by adubinkin on 09/10/17.
 */
public class TurboLandingsSteps extends BasePpcSteps {
    public static final String DEFAULT_TURBOLANDING_HREF = "https://yandex.ru/turbo?text=qlean&screen=welcome";
    public static final String DEFAULT_TURBOLANDING_TURBO_SITE_HREF = "defaultsite.turbo.site/page123";
    public static final String DEFAULT_TURBOLANDING_NAME = "default turbolanding";
    public static final Long TURBOLANDING_ID_CREATE_OFFSET = 10L;


    @Step("DB: получение записи в таблице ppc.turbolandings по turboLandingId: {0}")
    public TurbolandingsRecord getTurboLanding(Long turboLandingId) {
        return exec(db -> db.selectFrom(TURBOLANDINGS)
                .where(TURBOLANDINGS.TL_ID.eq(turboLandingId))
                .fetchOne()
        );
    }

    @Step("DB: получение записей в таблице ppc.turbolandings по clientId: {0}")
    public List<TurbolandingsRecord> getTurboLandingsByClientId(Long clientId) {
        return exec(db -> db.selectFrom(TURBOLANDINGS)
                .where(TURBOLANDINGS.CLIENTID.eq(clientId))
                .fetch()
        );
    }

    @Step("DB: получение записей в таблице ppc.turbolandings по clientId: {0}")
    public Long getNextTurbolandingIdForCreate() {
        return exec(db -> db.select(max(TURBOLANDINGS.TL_ID))
                .from(TURBOLANDINGS)
                .fetchOne(0, Long.class)
        ) + TURBOLANDING_ID_CREATE_OFFSET;
    }

    @Step("DB: Oбновление записи в таблице ppc.turbolandings")
    public void updateTurboLandings(TurbolandingsRecord turboLanding) {
        run(db -> db.update(TURBOLANDINGS)
                .set(turboLanding)
                .where(TURBOLANDINGS.TL_ID.eq(turboLanding.getTlId()))
                .execute()
        );
    }

    @Step("DB: добавление записи в таблицу ppc.turbolandings")
    public void addTurboLanding(TurbolandingsRecord turboLanding) {
        run(db -> db.insertInto(TURBOLANDINGS)
                .set(TURBOLANDINGS.CLIENTID, turboLanding.getClientid())
                .set(TURBOLANDINGS.TL_ID, turboLanding.getTlId())
                .set(TURBOLANDINGS.HREF, turboLanding.getHref())
                .set(TURBOLANDINGS.TURBO_SITE_HREF, turboLanding.getTurboSiteHref())
                .set(TURBOLANDINGS.METRIKA_COUNTERS_JSON, turboLanding.getMetrikaCountersJson())
                .execute());
    }

    @Step("DB: Создание записи в таблице ppc.turbolandings, если запись с таким tl_id существует - она будет " +
            "перезаписана")
    public void addOrUpdate(TurbolandingsRecord turboLanding) {
        run(db -> db.insertInto(TURBOLANDINGS)
                .set(TURBOLANDINGS.CLIENTID, turboLanding.getClientid())
                .set(TURBOLANDINGS.TL_ID, turboLanding.getTlId())
                .set(TURBOLANDINGS.HREF, turboLanding.getHref())
                .set(TURBOLANDINGS.TURBO_SITE_HREF, turboLanding.getTurboSiteHref())
                .set(TURBOLANDINGS.NAME, turboLanding.getName())
                .set(TURBOLANDINGS.METRIKA_COUNTERS_JSON, turboLanding.getMetrikaCountersJson())
                .onDuplicateKeyUpdate()
                .set(TURBOLANDINGS.CLIENTID, turboLanding.getClientid())
                .set(TURBOLANDINGS.HREF, turboLanding.getHref())
                .set(TURBOLANDINGS.NAME, turboLanding.getName())
                .set(TURBOLANDINGS.METRIKA_COUNTERS_JSON, turboLanding.getMetrikaCountersJson())
                .execute());
    }

    @Step("DB: Удаление записи в таблице ppc.turbolandings по переданному turboLandingId: {0}")
    public void deleteTurboLanding(long turboLandingId) {
        deleteTurboLandings(Collections.singletonList(turboLandingId));
    }

    @Step("DB: Удаление записи в таблице ppc.turbolandings по переданному списку turboLandingId: {0}")
    public void deleteTurboLandings(Collection<Long> turboLandingIds) {
        run(db -> db.delete(TURBOLANDINGS)
                .where(TURBOLANDINGS.TL_ID.in(turboLandingIds))
                .execute()
        );
    }

    @Step("DB: создание в ppc.turbolandings записи с дефолтными значениями полей")
    public long createDefaultTurbolanding(long clientId) {
        long turbolandingId = 1 + exec(db -> db.select(TURBOLANDINGS.TL_ID.max())
                .from(TURBOLANDINGS).fetchOne().value1());

        TurbolandingsRecord turbolanding = getDefaultTurbolanding(clientId);
        turbolanding.setTlId(turbolandingId);
        addOrUpdate(turbolanding);

        return turbolandingId;
    }

    public TurbolandingsRecord getDefaultTurbolanding(long clientId) {
        return new TurbolandingsRecord()
                .setHref(DEFAULT_TURBOLANDING_HREF)
                .setTurboSiteHref(DEFAULT_TURBOLANDING_TURBO_SITE_HREF)
                .setClientid(clientId)
                .setName(DEFAULT_TURBOLANDING_NAME)
                .setMetrikaCountersJson("[]");
    }

    @Step("DB: получение из ppc.turbolanding_metrika_counters записи c внутренним счетчиком")
    public Optional<TurbolandingMetrikaCountersRecord> getTurboladindingInternalMetrikaCounters(Long clientId) {

        Set<Long> turbolanidingIds = exec(db -> db.selectFrom(TURBOLANDINGS)
                .where(TURBOLANDINGS.CLIENTID.eq(clientId))
                .fetchSet(TURBOLANDINGS.TL_ID));

        return exec(db -> db.selectFrom(TURBOLANDING_METRIKA_COUNTERS)
                .where(TURBOLANDING_METRIKA_COUNTERS.TL_ID.in(turbolanidingIds)
                        .and(TURBOLANDING_METRIKA_COUNTERS.IS_USER_COUNTER.eq(0)))
                .fetchOptional()
        );
    }
}

