package ru.yandex.autotests.direct.db.steps;

import java.math.BigDecimal;
import java.util.Arrays;

import org.jooq.DSLContext;
import org.jooq.impl.DSL;

import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.WalletCampaignsAutopayMode;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.WalletCampaignsIsSumAggregated;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.WalletCampaignsRecord;
import ru.yandex.autotests.direct.db.steps.base.BasePpcSteps;
import ru.yandex.qatools.allure.annotations.Step;

import static ru.yandex.autotests.direct.db.models.jooq.ppc.Tables.CAMPAIGNS_MULTICURRENCY_SUMS;
import static ru.yandex.autotests.direct.db.models.jooq.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.autotests.direct.db.models.jooq.ppc.tables.WalletCampaigns.WALLET_CAMPAIGNS;

/**
 * Created by buhter on 06/04/16.
 */
public class WalletCampaignsSteps extends BasePpcSteps {

    @Step
    public WalletCampaignsRecord getWalletCampaigns(Long walletCid) {
        return exec(db -> db.selectFrom(WALLET_CAMPAIGNS)
                .where(WALLET_CAMPAIGNS.WALLET_CID.eq(walletCid))
                .fetchOne()
        );
    }

    @Step("DB: изменение записи в таблице ppc.wallet_campaigns")
    public void updateWalletCampaigns(WalletCampaignsRecord walletCampaigns) {
        run(db -> db.update(WALLET_CAMPAIGNS)
                .set(walletCampaigns)
                .where(WALLET_CAMPAIGNS.WALLET_CID.eq(walletCampaigns.getWalletCid()))
                .execute());
    }

    @Step("DB: обновление записи в ppc.wallet_campaigns walletCid = {0}, autopayMode = {1}")
    public void updateWalletCampaigns(Long walletCid, WalletCampaignsAutopayMode walletCampaignsAutopayMode) {
        WalletCampaignsRecord walletCampaigns = getWalletCampaigns(walletCid);
        walletCampaigns.setAutopayMode(walletCampaignsAutopayMode);
        updateWalletCampaigns(walletCampaigns);
    }

    @Step("DB: перевод общего счета {0} в новую схему зачислений, перенос зачислений с дочерних кампаний")
    public void migrateWalletToNewSums(Long walletCid) {
        run(db -> db.transaction(conf ->
                {
                    DSLContext ctx = DSL.using(conf);
                    ctx.update(CAMPAIGNS)
                            .set(CAMPAIGNS.SUM_BALANCE, CAMPAIGNS.SUM)
                            .where(CAMPAIGNS.WALLET_CID.eq(walletCid).or(CAMPAIGNS.CID.eq(walletCid)))
                            .execute();

                    BigDecimal[] childSums = ctx.select(CAMPAIGNS.SUM)
                            .from(CAMPAIGNS)
                            .where(CAMPAIGNS.WALLET_CID.eq(walletCid).or(CAMPAIGNS.CID.eq(walletCid)))
                            .fetchArray(CAMPAIGNS.SUM);

                    BigDecimal sumSum = Arrays.stream(childSums).reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
                    ctx.update(CAMPAIGNS)
                            .set(CAMPAIGNS.SUM, sumSum)
                            .where(CAMPAIGNS.CID.eq(walletCid))
                            .execute();

                    ctx.update(CAMPAIGNS)
                            .set(CAMPAIGNS.SUM, BigDecimal.ZERO)
                            .where(CAMPAIGNS.WALLET_CID.eq(walletCid))
                            .execute();

                    BigDecimal[] childChipsCosts = ctx.select(CAMPAIGNS_MULTICURRENCY_SUMS.CHIPS_COST)
                            .from(CAMPAIGNS_MULTICURRENCY_SUMS
                                    .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(CAMPAIGNS_MULTICURRENCY_SUMS.CID))
                            )
                            .where(CAMPAIGNS.WALLET_CID.eq(walletCid).or(CAMPAIGNS.CID.eq(walletCid)))
                            .fetchArray(CAMPAIGNS_MULTICURRENCY_SUMS.CHIPS_COST);
                    BigDecimal chipCostSum = Arrays.stream(childChipsCosts).reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
                    ctx.update(WALLET_CAMPAIGNS)
                            .set(WALLET_CAMPAIGNS.TOTAL_CHIPS_COST, chipCostSum)
                            .set(WALLET_CAMPAIGNS.IS_SUM_AGGREGATED, WalletCampaignsIsSumAggregated.Yes)
                            .where(WALLET_CAMPAIGNS.WALLET_CID.eq(walletCid))
                            .execute();
                }
        ));
    }
}
