package ru.yandex.autotests.direct.db.utils;

import java.util.Map;

import org.hamcrest.Description;
import org.hamcrest.Factory;
import org.hamcrest.TypeSafeMatcher;
import org.jooq.Record;

import ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher;
import ru.yandex.autotests.irt.testutils.beandiffer2.comparestrategy.CompareStrategy;
import ru.yandex.autotests.irt.testutils.beandiffer2.comparestrategy.defaultcomparestrategy.DefaultCompareStrategies;

import static ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher.beanDiffer;

/**
 * Матчер, который сравнивает Jooq-ие Record объекты.
 * Вызывает у Record intoMap() и передает beanDiffer-у
 */
public class JooqRecordDifferMatcher<T extends Record> extends TypeSafeMatcher<T> {

    private T expected;
    private BeanDifferMatcher<Map<String, Object>> beanDiffer;
    private CompareStrategy compareStrategy;

    /**
     * Ожидается, что будет передан не null
     * Для проверки на null есть специальный матчер isNull()
     *
     * @param expected - not null
     */
    private JooqRecordDifferMatcher(T expected) {
        this.expected = expected;
        this.beanDiffer = beanDiffer(expected.intoMap());
    }

    public JooqRecordDifferMatcher<T> useCompareStrategy(CompareStrategy compareStrategy) {
        this.compareStrategy = compareStrategy;
        return this;
    }

    @Override
    protected boolean matchesSafely(T actual) {
        Map<String, Object> actualMap = actual != null ? actual.intoMap() : null;
        return beanDiffer.useCompareStrategy(compareStrategy).matches(actualMap);
    }

    @Override
    public void describeTo(Description description) {
        String expectedType = expected.getClass().getSimpleName();
        description.appendText("the same objects of type ").appendText(expectedType);
    }

    @Override
    protected void describeMismatchSafely(T item, Description mismatchDescription) {
        beanDiffer.describeMismatch(item, mismatchDescription);
    }

    @Factory
    public static <T extends Record> JooqRecordDifferMatcher<T> recordDiffer(T expected) {
        if (expected == null) {
            throw new IllegalArgumentException("expected bean must not null");
        }
        return new JooqRecordDifferMatcher<>(expected)
                .useCompareStrategy(DefaultCompareStrategies.allFields());
    }
}