package ru.yandex.autotests.direct.db.utils;

import java.util.Map;
import java.util.List;
import java.util.stream.Collectors;

import org.hamcrest.Description;
import org.hamcrest.Factory;
import org.hamcrest.TypeSafeMatcher;
import org.jooq.Record;

import ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher;
import ru.yandex.autotests.irt.testutils.beandiffer2.comparestrategy.CompareStrategy;
import ru.yandex.autotests.irt.testutils.beandiffer2.comparestrategy.defaultcomparestrategy.DefaultCompareStrategies;

import static ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher.beanDiffer;

/**
 * Матчер, который сравнивает списки Jooq Record объектов.
 * Вызывает у каждого Record intoMap() и передает beanDiffer-у
 */
public class JooqRecordListDifferMatcher<T extends List<? extends Record>> extends TypeSafeMatcher<T>{

    private List<? extends Record> expected;
    private BeanDifferMatcher<List<Map<String, Object>>> beanDiffer;
    private CompareStrategy compareStrategy;

    /**
     * Ожидается, что будет передан не null
     * Для проверки на null есть специальный матчер isNull()
     *
     * @param expected - not null
     */

    private JooqRecordListDifferMatcher(T expected) {
        this.expected = expected;
        List<Map<String, Object>> listOfMaps = recordsIntoMaps(this.expected);
        this.beanDiffer = beanDiffer(listOfMaps);
    }

    public JooqRecordListDifferMatcher<T> useCompareStrategy(CompareStrategy compareStrategy) {
        this.compareStrategy = compareStrategy;
        return this;
    }

    @Override
    protected boolean matchesSafely(T actual) {
        List<Map<String, Object>> actualMaps = null;
        if (actual != null) {
            actualMaps = recordsIntoMaps(actual);
        }
        return beanDiffer.useCompareStrategy(compareStrategy).matches(actualMaps);
    }

    @Override
    public void describeTo(Description description) {
        String expectedType = expected.getClass().getSimpleName();
        description.appendText("the same objects of type ").appendText(expectedType);
    }

    @Override
    protected void describeMismatchSafely(T item, Description mismatchDescription) {
        beanDiffer.describeMismatch(item, mismatchDescription);
    }

    @Factory
    public static <T extends List<? extends Record>> JooqRecordListDifferMatcher<T> recordsDiffer(T expected) {
        if (expected == null) {
            throw new IllegalArgumentException("expected bean must not null");
        }
        return new JooqRecordListDifferMatcher<>(expected)
                .useCompareStrategy(DefaultCompareStrategies.allFields());
    }

    private List<Map<String, Object>> recordsIntoMaps (List<? extends Record> records) {
        return records.stream()
                .map(r -> r.intoMap())
                .collect(Collectors.toList());
    }
}