package ru.yandex.autotests.direct.handles;

import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.cfg.Configuration;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Example;
import org.hibernate.criterion.Projection;
import org.hibernate.internal.util.SerializationHelper;
import org.hibernate.service.ServiceRegistry;
import org.hibernate.service.ServiceRegistryBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import ru.yandex.autotests.direct.handles.db.beans.DBElementBean;
import ru.yandex.autotests.direct.handles.db.beans.DbSearchBean;
import ru.yandex.autotests.direct.handles.db.beans.DbSearchByPrimaryKeyBean;
import ru.yandex.autotests.direct.handles.db.beans.DbSearchResultBean;
import ru.yandex.autotests.direct.handles.db.beans.DbUpdateBean;
import ru.yandex.autotests.direct.utils.ReflectionUtils;
import ru.yandex.autotests.irt.testutils.beans.BeanHelper;
import ru.yandex.autotests.irt.testutils.json.JsonUtils;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.util.List;

import static org.apache.commons.lang.StringUtils.capitalize;


/**
 * Created by omaz on 17.01.14.
 * Сервис для доступа в БД Директа
 */
@Service
public class DBConnectionService {

    private static final String ZERO_DATE_INNER_1 = "0001-01-01 00:00:00.0";
    private static final String ZERO_DATE_INNER_2 = "0001-01-01";
    private static final String ZERO_DATE_MYSQL = "0000-00-00 00:00:00.0";
    private static final Integer DEFAULT_SELECT_LIMIT = 10;

    private static final Logger log = LoggerFactory.getLogger(DBConnectionService.class);

    private SessionFactory getSessionFactory(Class logicClass, String host, String port) {
        SessionFactory sessionFactory;
        try {
            Class.forName("com.mysql.jdbc.Driver");
            Configuration configuration = new Configuration()
                    .setProperty("hibernate.connection.url", "jdbc:mysql://" + host + ":" + port)
                    .addAnnotatedClass(logicClass)
                    .configure();
            ServiceRegistry serviceRegistry = new ServiceRegistryBuilder().applySettings(configuration.getProperties()).buildServiceRegistry();
            sessionFactory = configuration.buildSessionFactory(serviceRegistry);
        } catch (Exception e) {
            throw new RuntimeException("Не удалось подключиться к БД", e);
        }
        return sessionFactory;
    }


    private Criteria createCriteria(Object filter, Session session) {
        Criteria criteria = session.createCriteria(filter.getClass());
        criteria.add(Example.create(filter));
        return criteria;
    }

    /**
     * Метод делает SELECT-запрос к таблице по primary key
     *
     * @param logicClass класс, описывающий таблицу
     * @param id         передаваемое значение primary key
     */
    private <T> T findByPrimaryKey(Object id, Class logicClass, String host, String port) {
        SessionFactory sessionFactory = getSessionFactory(logicClass, host, port);
        Session session = sessionFactory.openSession();
        T result;
        try {
            result = (T) session.get(logicClass, (Serializable) id);
        } catch (Exception e) {
            throw new RuntimeException("Не удалось открыть сессию для " + logicClass, e);
        } finally {
            session.close();
            sessionFactory.close();
        }
        return result;
    }

    public DbSearchResultBean findByPrimaryKey(DbSearchByPrimaryKeyBean dbSearchBean) throws ClassNotFoundException {
        Object searchResult = findByPrimaryKey(
                JsonUtils.getObject(
                        dbSearchBean.getId(),
                        Class.forName(dbSearchBean.getIdClassName())
                ),
                Class.forName(dbSearchBean.getClassName()),
                dbSearchBean.getHost(),
                dbSearchBean.getPort());
        DbSearchResultBean resultBean = new DbSearchResultBean();
        DBElementBean elementBean = new DBElementBean();
        if (searchResult != null)
            elementBean.setClassName(searchResult.getClass().getName());
        elementBean.setData(JsonUtils.toString(searchResult));
        resultBean.setResult(elementBean);
        return resultBean;
    }

    /**
     * SELECT по полям, не являющимся primary key.
     * Возвращает null или ОДНО найденное значение.
     * Если значений > 1 - выкинется эксепшн.
     *
     * @param filter - объект класса таблицы с заданными для поиска полями
     */
    private <T> T findOne(T filter, String host, String port) {
        SessionFactory sessionFactory = getSessionFactory(filter.getClass(), host, port);
        Session session = sessionFactory.openSession();
        T result;
        Criteria criteria;
        try {
            criteria = createCriteria(filter, session);
            result = (T) criteria.uniqueResult();
        } catch (Exception e) {
            throw new RuntimeException("Не удалось открыть сессию для " + filter.getClass(), e);
        } finally {
            session.close();
            sessionFactory.close();
        }

        return result;
    }

    public DbSearchResultBean findOne(DbSearchBean dbSearchBean) {
        Object searchResult = null;
        try {
            searchResult = findOne(
                    JsonUtils.getObject(
                            dbSearchBean.getFilter(),
                            Class.forName(dbSearchBean.getClassName())
                    ),
                    dbSearchBean.getHost(),
                    dbSearchBean.getPort());
        } catch (ClassNotFoundException e) {
            throw new RuntimeException("Не удалось найти найти класс для бина " + dbSearchBean.getClassName(), e);
        }
        DbSearchResultBean resultBean = new DbSearchResultBean();
        DBElementBean elementBean = new DBElementBean();
        if (searchResult != null)
            elementBean.setClassName(searchResult.getClass().getName());
        elementBean.setData(JsonUtils.toString(searchResult));
        resultBean.setResult(elementBean);
        return resultBean;
    }

    /**
     * SELECT по полям, не являющимся primary key.
     * Возвращает все значения.
     *
     * @param filter      - объект класса таблицы с заданными для поиска полями
     * @param restriction - ограничение (опционально)
     */
    private <T> List<T> findAll(T filter, Criterion restriction, Integer limit, String host, String port) {
        SessionFactory sessionFactory = getSessionFactory(filter.getClass(), host, port);
        Session session = sessionFactory.openSession();
        List<T> result;
        Criteria criteria;
        limit = limit == null ? DEFAULT_SELECT_LIMIT : limit;
        try {
            criteria = createCriteria(filter, session);
            if (restriction != null)
                criteria.add(restriction);
            criteria.setMaxResults(limit);
            result = criteria.list();
        } catch (Exception e) {
            throw new RuntimeException("Не удалось открыть сессию для " + filter.getClass(), e);
        } finally {
            session.close();
            sessionFactory.close();
        }
        return result;
    }

    public DbSearchResultBean findAll(DbSearchBean dbSearchBean) {
        List<Object> searchResult = null;
        try {
            searchResult = findAll(
                    JsonUtils.getObject(
                            dbSearchBean.getFilter(),
                            Class.forName(dbSearchBean.getClassName())
                    ),
                    dbSearchBean.getRestriction() == null ? null :
                            (Criterion) SerializationHelper.deserialize(dbSearchBean.getRestriction()),
                    dbSearchBean.getLimit(),
                    dbSearchBean.getHost(),
                    dbSearchBean.getPort());
        } catch (ClassNotFoundException e) {
            throw new RuntimeException("Не удалось найти найти класс для бина " + dbSearchBean.getClassName(), e);
        }
        DbSearchResultBean resultBean = new DbSearchResultBean();
        DBElementBean dbElementBean = new DBElementBean();
        if (searchResult != null && searchResult.size() > 0)
            dbElementBean.setClassName(dbSearchBean.getClassName());
        dbElementBean.setData(
                JsonUtils.toString(searchResult)
        );
        resultBean.setResult(dbElementBean);
        return resultBean;
    }

    /**
     * SELECT с проекцией (max, min, count и т.п.).
     * Возвращает все значения.
     *
     * @param filter - объект класса таблицы с заданными для поиска полями
     * @param restriction - ограничение (опционально)
     */
    private <T> List<T> findWithProjection(T filter, Criterion restriction, Projection projection, Integer limit,
                                           String host, String port) {
        SessionFactory sessionFactory = getSessionFactory(filter.getClass(), host, port);
        Session session = sessionFactory.openSession();
        List<T> result;
        Criteria criteria;
        limit = limit == null ? DEFAULT_SELECT_LIMIT : limit;
        try {
            criteria = createCriteria(filter, session);
            if (restriction != null)
                criteria.add(restriction);
            criteria.setProjection(projection);
            criteria.setMaxResults(limit);
            result = criteria.list();
        } catch (Exception e) {
            throw new RuntimeException("Не удалось открыть сессию для " + filter.getClass(), e);
        } finally {
            session.close();
            sessionFactory.close();
        }
        return result;
    }

    public DbSearchResultBean findWithProjection(DbSearchBean dbSearchBean) {
        List<Object> searchResult = null;
        try {
            searchResult = findWithProjection(
                    JsonUtils.getObject(
                            dbSearchBean.getFilter(),
                            Class.forName(dbSearchBean.getClassName())
                    ),
                    dbSearchBean.getRestriction() == null ? null :
                            (Criterion) SerializationHelper.deserialize(dbSearchBean.getRestriction()),
                    (Projection) JsonUtils.getObject(
                            dbSearchBean.getProjection(),
                            Class.forName(dbSearchBean.getProjectionClassName())
                    ),
                    dbSearchBean.getLimit(),
                    dbSearchBean.getHost(),
                    dbSearchBean.getPort());
        } catch (ClassNotFoundException e) {
            throw new RuntimeException("Не удалось найти найти класс для бина " + dbSearchBean.getClassName(), e);
        }
        DbSearchResultBean resultBean = new DbSearchResultBean();
        DBElementBean dbElementBean = new DBElementBean();
        if (searchResult != null && searchResult.size() > 0)
            dbElementBean.setClassName(dbSearchBean.getClassName());
        dbElementBean.setData(
                JsonUtils.toString(searchResult)
        );
        resultBean.setResult(dbElementBean);
        return resultBean;
    }

    /**
     * UPDATE записи в БД
     *
     * @param object - объект класса таблицы со всеми полями (заданы должны быть и измененные, и нет)
     */
    private <T> boolean update(T object, String configurationName, String port) {
        SessionFactory sessionFactory = getSessionFactory(object.getClass(), configurationName, port);
        Session session = sessionFactory.openSession();
        // костыль для обхода несовместимости дат в hibernate и mysql
        convertZeroDatesToMysqlFormat(object);
        try {
            session.beginTransaction();
            session.update(object);
            session.getTransaction().commit();
            log.info("Updated table " + object.getClass().getName() + "\n" + JsonUtils.toString(object));
            return true;
        } catch (Exception e) {
            throw new RuntimeException("Не удалось выполнить UPDATE для " + object, e);
        } finally {
            session.close();
            sessionFactory.close();
        }
    }

    public void update(DbUpdateBean dbUpdateBean) {
        try {
            update(
                    JsonUtils.getObject(
                            dbUpdateBean.getObject(),
                            Class.forName(dbUpdateBean.getClassName())
                    ),
                    dbUpdateBean.getConfiguration(),
                    dbUpdateBean.getPort());
        } catch (ClassNotFoundException e) {
            throw new RuntimeException("Не удалось выполнить UPDATE для " + dbUpdateBean.getObject(), e);
        }
    }

    /**
     * INSERT object в БД
     *
     * @param object - объект класса таблицы со всеми полями
     */
    private <T> Serializable save(T object, String configurationName, String port) {
        SessionFactory sessionFactory = getSessionFactory(object.getClass(), configurationName, port);
        Session session = sessionFactory.openSession();
        // костыль для обхода несовместимости дат в hibernate и mysql
        convertZeroDatesToMysqlFormat(object);
        try {
            session.beginTransaction();
            Serializable result = session.save(object);
            session.getTransaction().commit();
            log.info("INSERT INTO table " + object.getClass().getName() + "\n" + JsonUtils.toString(object));
            return result;
        } catch (Exception e) {
            throw new RuntimeException("Не удалось выполнить SAVE для " + object, e);
        } finally {
            session.close();
            sessionFactory.close();
        }
    }

    public Serializable save(DbUpdateBean dbUpdateBean) {
        try {
            return save(
                    JsonUtils.getObject(
                            dbUpdateBean.getObject(),
                            Class.forName(dbUpdateBean.getClassName())
                    ),
                    dbUpdateBean.getConfiguration(),
                    dbUpdateBean.getPort());
        } catch (ClassNotFoundException e) {
            throw new RuntimeException("Не удалось выполнить SAVE для " + dbUpdateBean.getObject(), e);
        }
    }

    /**
     * Удаление из БД
     *
     * @param object - объект класса таблицы со всеми полями
     */
    private <T> void delete(T object, String configurationName, String port) {
        SessionFactory sessionFactory = getSessionFactory(object.getClass(), configurationName, port);
        Session session = sessionFactory.openSession();
        try {
            session.beginTransaction();
            session.delete(object);
            session.getTransaction().commit();
            log.info("DELETE table " + object.getClass().getName() + "\n" + JsonUtils.toString(object));
        } catch (Exception e) {
            throw new RuntimeException("Не удалось выполнить DELETE для " + object, e);
        } finally {
            session.close();
            sessionFactory.close();
        }
    }

    public void delete(DbUpdateBean dbUpdateBean) {
        try {
            delete(JsonUtils.getObject(
                            dbUpdateBean.getObject(),
                            Class.forName(dbUpdateBean.getClassName())
                    ),
                    dbUpdateBean.getConfiguration(),
                    dbUpdateBean.getPort());
        } catch (ClassNotFoundException e) {
            throw new RuntimeException("Не удалось выполнить DELETE для " + dbUpdateBean.getObject(), e);
        }
    }

    /**
     * костыль для обхода проблем с совместимостью дат MySQL, java.util.Date и java.sql.Date:
     * когда читаем - меняем нулевые даты на 0001-01-01 00:00:00.0, когда пишем - меняем обратно
     * (иначе проблемы с обработкой null'ов)
     * @param object объект
     */
    // TODO: http://stackoverflow.com/questions/782823/handling-datetime-values-0000-00-00-000000-in-jdbc
    private void convertZeroDatesToMysqlFormat(Object object) {
        for (Field field : object.getClass().getDeclaredFields()) {
            Object value = BeanHelper.getProperty(object, field.getName());
            if (ZERO_DATE_INNER_1.equals(value) || ZERO_DATE_INNER_2.equals(value)) {
                ReflectionUtils.invokeSetter(object, capitalize(field.getName()), ZERO_DATE_MYSQL);
            }
        }
    }
}
