package ru.yandex.autotests.direct.handles.service.utils.aqua;

import ch.lambdaj.Lambda;
import org.slf4j.LoggerFactory;
import ru.yandex.aqua.AquaApiUtils;
import ru.yandex.aqua.beans.Launch;
import ru.yandex.aqua.beans.LaunchPack;
import ru.yandex.aqua.beans.LaunchStatus;
import ru.yandex.aqua.beans.Pack;
import ru.yandex.aqua.beans.Project;
import ru.yandex.aqua.beans.Property;
import ru.yandex.autotests.direct.handles.service.utils.constants.AquaStage;

import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;

import static ch.lambdaj.Lambda.having;
import static ch.lambdaj.Lambda.on;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.anyOf;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.equalToIgnoringCase;
import static org.hamcrest.Matchers.greaterThanOrEqualTo;

/**
 * User: buhter
 * Date: 16.09.13
 * Time: 10:31
 */
public class Aqua {
    public static final String AQUA_API_BASE_URL_MODIFIER = "/aqua-api/services";
    private static final String MAX_EXECUTION_TIME_PACK_PROPERTY = "max.execution.time";
    private static final String DIRECT_GROUP_ID = "ru.yandex.autotests.direct";
    private static final String PASSPORT_MONITOR_GROUP_ID = "ru.yandex.autotests.passport-monitor";
    private static final long ONE_HOUR = 3600000;
    public static final int UNFINISHED_LAUNCHES_MAX_COUNT = 100;


    private String aquaStage;

    public Aqua(String aquaStage) {
        this.aquaStage = aquaStage + AQUA_API_BASE_URL_MODIFIER;
    }

    public Pack getPack(String packId) {
        return AquaApiUtils.getPackById(packId, aquaStage);
    }

    public void revokeLaunch(String launchID) {
        AquaApiUtils.revokeLaunchById(launchID, aquaStage);
    }

    public Launch updateLaunch(Launch launch) {
        return updateLaunch(launch.getId());
    }

    public Launch updateLaunch(String launchId) {
        return AquaApiUtils.getLaunchById(launchId, aquaStage);
    }


    public Launch launchDirectHandlesTestClass(String testClassName, String testPackage
            , HashMap<String, String> properties) {
        return launch(AquaApiUtils.getSingleSuiteFromProject(
                getDirectHandlesProjectIdForLaunch(), testClassName, testPackage, aquaStage, properties));
    }

    public Launch launchDirectTestClass(String testClassName, String testPackage
            , String projectId, HashMap<String, String> properties) {
        return launch(AquaApiUtils.getSingleSuiteFromProject(
                projectId, testClassName, testPackage, aquaStage, properties));
    }
    public Launch launchDirectTestClass(String packId, String testClassName, String testPackage) {
        return launch(AquaApiUtils.getSingleSuiteFromPack(packId, testClassName, testPackage, aquaStage));
    }

    public Launch launchPackByID(String packID) throws Exception {
        return launch(getPackForLaunchByID(packID, new HashMap<String, String>()));
    }

    public Launch launchPackByIDWithProperties(String packID, HashMap<String, String> properties) throws Exception {
        return launch(getPackForLaunchByID(packID, properties));
    }

    public Launch launchPackByIDWithProperties(String packID, HashMap<String, String> properties, int tip)
            throws Exception {
        return launch(getPackForLaunchByID(packID, properties, tip));
    }

    private Launch launch(Pack pack) {
        return updateLaunch(AquaApiUtils.launchPack(pack, aquaStage));
    }

    private static String getDirectHandlesProjectIdForLaunch() {
        return "ru.yandex.autotests.direct:direct-handles-aqua-helper:1.0-SNAPSHOT";
    }

    private Pack getPackForLaunchByID(String packID, HashMap<String, String> properties) throws Exception {
        Pack packForLaunch = AquaApiUtils.getPackById(packID, aquaStage);
        if (packForLaunch == null) {
            throw new Exception("Pack with ID=" + packID + " not found.");
        }
        AquaApiUtils.setPropertiesToPack(packForLaunch, properties);
        packForLaunch.setBuildReport(true);
        return packForLaunch;
    }

    private Pack getPackForLaunchByID(String packID, HashMap<String, String> properties, int tip) throws Exception {
        Pack packForLaunch = AquaApiUtils.getPackById(packID, aquaStage);
        if (packForLaunch == null) {
            throw new Exception("Pack with ID=" + packID + " not found.");
        }
        AquaApiUtils.setPropertiesToPack(packForLaunch, properties);
        packForLaunch.setBuildReport(true);
        packForLaunch.setTip(tip);
        return packForLaunch;
    }

    public String getAquaStage() {
        return aquaStage.replace(AQUA_API_BASE_URL_MODIFIER, "");
    }

    public List<Project> getProjects(String pattern) {
        return Lambda.filter(having(on(Project.class).getTitle(), containsString(pattern)),
                AquaApiUtils.getProjects(aquaStage));
    }

    public List<Project> getProjectsByGroupID(String groupID) {
        return Lambda.filter(having(on(Project.class).getMaven().getGroupId(), equalTo(groupID)),
                AquaApiUtils.getProjects(aquaStage));
    }

    public List<Pack> getProjectPacks(String projectId) {
        return AquaApiUtils.getProjectPacks(projectId, aquaStage);
    }

    public List<Pack> getPacks(String pattern) {
        return Lambda.filter(having(on(Pack.class).getName(), containsString(pattern)), AquaApiUtils.getPacks(aquaStage));
    }

    public String getPropertyValue(LaunchPack pack, String key) {
        List<Property> properties =
                Lambda.filter(having(on(Property.class).getKey(), equalToIgnoringCase(key)), pack.getProperties());
        if (properties.size() == 0) {
            return null;
        } else {
            return properties.get(0).getValue();
        }
    }

    //получить верхнюю границу времени выполнения пака. Если установлено свойтсво пака 'max.execution.time'
    // (типа float - в часах, например 3.5 - 3 часа 30 минут), то возвращается значение свойства в виде числа long,
    //иначе 1 час
    public long getLaunchMaxTime(Launch launch) {
        Aqua aqua = new Aqua(AquaStage.AQUA_BASE_URL);
        String maxExecutionTime = aqua.getPropertyValue(launch.getPack(), MAX_EXECUTION_TIME_PACK_PROPERTY);
        long borderValue = ONE_HOUR;
        if (maxExecutionTime != null) {
            try {
                borderValue = Float.valueOf(Float.valueOf(maxExecutionTime) * ONE_HOUR).longValue();
            } catch (NumberFormatException e) {
                LoggerFactory.getLogger(Aqua.class).error("Can't read property 'max.execution.time' " +
                        "from pack " + launch.getPack().getName() + ". Expected float value, but was: "+maxExecutionTime);
            }
        }
        return borderValue;
    }

    //возвращает самый поздний запуск из долгих
    //если долгих запусков нет, то возвращает null
    public Launch getLongestLaunch(List<Launch> launches) {
        long maxLaunchStartTime = 0;
        Launch launchWithMaxStartTime = null;
        for (Launch launch : launches) {
            if (new Date().getTime() - launch.getStartTime() >= getLaunchMaxTime(launch)) {
                if (maxLaunchStartTime == 0) {
                    maxLaunchStartTime = launch.getStartTime();
                    launchWithMaxStartTime = launch;
                }
                if (launch.getStartTime() > maxLaunchStartTime) {
                    maxLaunchStartTime = launch.getStartTime();
                    launchWithMaxStartTime = launch;
                }
            }
        }
        return launchWithMaxStartTime;
    }

    public List<Launch> getUnfinishedLaunchIdByPackId(String packId) {
        long hour = 3600000;
        //берем всле ланчи за последние 48 часов
        long pastBorder = new Date().getTime() - 48 * hour;
        return Lambda.filter(allOf(
                        having(on(Launch.class).getLaunchStatus(),
                                anyOf(equalTo(LaunchStatus.RUNNING), equalTo(LaunchStatus.RUNNABLE))),
                        having(on(Launch.class).getStartTime(), greaterThanOrEqualTo(pastBorder))
                ),
                AquaApiUtils.getUserLaunches("", UNFINISHED_LAUNCHES_MAX_COUNT, 0, packId, aquaStage));
    }


    public List<Launch> getUnfinishedProjectLaunchesForLastTwoDays() {
        List<Launch> result = new LinkedList<Launch>();
        result.addAll(AquaApiUtils.getProjectLaunches(DIRECT_GROUP_ID, null, null, true, aquaStage));
        result.addAll(AquaApiUtils.getProjectLaunches(PASSPORT_MONITOR_GROUP_ID, null, null, true, aquaStage));
        return result;
    }

    public List<List<Launch>> groupByPackID(List<Launch> launches) {
        List<List<Launch>> result = new LinkedList<List<Launch>>();
        if (launches.size() > 0) {
            String packId = launches.get(0).getPack().getId();
            List<Launch> group = new LinkedList<Launch>();
            for (Launch launch : launches) {
                if (!launch.getPack().getId().equals(packId)) {
                    packId = launch.getPack().getId();
                    result.add(group);
                    group = new LinkedList<Launch>();
                }
                group.add(launch);
            }
            result.add(group);
        }
        return result;
    }
}