package ru.yandex.autotests.direct.intapi.client;

import java.io.IOException;
import java.lang.annotation.Annotation;
import java.lang.reflect.Type;
import java.time.LocalDateTime;
import java.time.ZonedDateTime;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import okhttp3.Interceptor;
import okhttp3.Response;
import retrofit2.Converter;
import retrofit2.Retrofit;

import retrofit2.converter.gson.GsonConverterFactory;
import retrofit2.converter.scalars.ScalarsConverterFactory;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonParseException;
import okhttp3.OkHttpClient;
import okhttp3.RequestBody;
import okhttp3.ResponseBody;
import okhttp3.Request;
import okhttp3.logging.HttpLoggingInterceptor;

import ru.yandex.autotests.direct.intapi.config.ClientConfig;
import ru.yandex.autotests.direct.intapi.config.Slf4jLogger;

public class ApiClient {
    private static final String TVM_SERVICE_TICKET_HEADER = "X-Ya-Service-Ticket";

    private OkHttpClient okClient;
    private Retrofit.Builder adapterBuilder;

    private boolean verbose;
    private String baseUrl;
    private Supplier<String> tvmTicketProvider;

    private ApiClient(Supplier<String> tvmTicketProvider) {
        this.tvmTicketProvider = tvmTicketProvider;
    }

    public void createDefaultAdapter() {
        Gson gson = new GsonBuilder().registerTypeAdapter(LocalDateTime.class, new JsonDeserializer<LocalDateTime>() {
            public LocalDateTime deserialize(JsonElement json, Type type, JsonDeserializationContext jsonDeserializationContext) throws JsonParseException {
                return ZonedDateTime.parse(json.getAsJsonPrimitive().getAsString()).toLocalDateTime();
            }
        })
        .create();

        OkHttpClient.Builder okHttpClientbuilder = new OkHttpClient.Builder()
                .connectTimeout(ClientConfig.get().getConnectTimeout(), TimeUnit.SECONDS)
                .readTimeout(ClientConfig.get().getReadTimeout(), TimeUnit.SECONDS)
                .writeTimeout(ClientConfig.get().getWriteTimeout(), TimeUnit.SECONDS);

        if(verbose) {
            HttpLoggingInterceptor loggingInterceptor = new HttpLoggingInterceptor(new Slf4jLogger());
            loggingInterceptor.setLevel(HttpLoggingInterceptor.Level.BODY);
            okHttpClientbuilder.addInterceptor(loggingInterceptor);
        }
        okHttpClientbuilder.addInterceptor(new Interceptor() {
            @Override
            public Response intercept(Chain chain) throws IOException {
                Request.Builder builder = chain.request().newBuilder();
                String tvmServiceTicket = tvmTicketProvider.get();
                if (tvmServiceTicket != null) {
                    builder.addHeader(TVM_SERVICE_TICKET_HEADER, tvmServiceTicket);
                }
                return chain.proceed(builder.build());
            }
        });
        okClient = okHttpClientbuilder.build();

        adapterBuilder = new Retrofit
            .Builder()
            .baseUrl(baseUrl)
            .client(okClient)
            
            .addConverterFactory(ScalarsConverterFactory.create())
            .addConverterFactory(GsonCustomConverterFactory.create(gson));
    }

    public <S> S createService(Class<S> serviceClass) {
        return adapterBuilder.build().create(serviceClass);
    }

    public Retrofit.Builder getAdapterBuilder() {
        return adapterBuilder;
    }

    public void setAdapterBuilder(Retrofit.Builder adapterBuilder) {
        this.adapterBuilder = adapterBuilder;
    }

    public OkHttpClient getOkClient() {
        return okClient;
    }

    public static class Builder {
        private String baseUrl = "https://intapi.test.direct.yandex.ru:9443/";
        private boolean verbose = false;
        private Supplier<String> tvmTicketProvider = () -> null;

        public static Builder create() {
            return new Builder();
        }

        public Builder baseUrl(String baseUrl) {
            this.baseUrl = baseUrl;
            return this;
        }

        public Builder verbose(boolean verbose) {
            this.verbose = verbose;
            return this;
        }

        public Builder tvmTicketProvider(Supplier<String> tvmTicketProvider) {
            this.tvmTicketProvider = tvmTicketProvider;
            return this;
        }

        public ApiClient build() {
            ApiClient client = new ApiClient(tvmTicketProvider);
            client.verbose = verbose;
            client.baseUrl = baseUrl;
            client.createDefaultAdapter();
            return client;
        }
    }
}

/**
 * This wrapper is to take care of this case:
 * when the deserialization fails due to JsonParseException and the
 * expected type is String, then just return the body string.
 */
class GsonResponseBodyConverterToString<T> implements Converter<ResponseBody, T> {
	  private final Gson gson;
	  private final Type type;

	  GsonResponseBodyConverterToString(Gson gson, Type type) {
	    this.gson = gson;
	    this.type = type;
	  }

	  public T convert(ResponseBody value) throws IOException {
	    String returned = value.string();
	    try {
	      return gson.fromJson(returned, type);
	    }
	    catch (JsonParseException e) {
                return (T) returned;
        }
	 }
}

class GsonCustomConverterFactory extends Converter.Factory {
	public static GsonCustomConverterFactory create(Gson gson) {
	    return new GsonCustomConverterFactory(gson);
	  }

	  private final Gson gson;
	  private final GsonConverterFactory gsonConverterFactory;

	  private GsonCustomConverterFactory(Gson gson) {
	    if (gson == null) throw new NullPointerException("gson == null");
	    this.gson = gson;
	    this.gsonConverterFactory = GsonConverterFactory.create(gson);
	  }

    @Override
    public Converter<ResponseBody, ?> responseBodyConverter(Type type, Annotation[] annotations, Retrofit retrofit) {
        if(type.equals(String.class))
            return new GsonResponseBodyConverterToString<Object>(gson, type);
        else
            return gsonConverterFactory.responseBodyConverter(type, annotations, retrofit);
    }

    @Override
    public Converter<?, RequestBody> requestBodyConverter(Type type, Annotation[] parameterAnnotations, Annotation[] methodAnnotations, Retrofit retrofit) {
            return gsonConverterFactory.requestBodyConverter(type, parameterAnnotations, methodAnnotations, retrofit);
    }
}
