package ru.yandex.autotests.direct.analyzer.tree;

import ru.yandex.autotests.direct.analyzer.perl.PerlFunction;
import ru.yandex.autotests.direct.analyzer.perl.PerlModule;
import org.hamcrest.Matcher;

import java.util.HashMap;
import java.util.Map;

/**
 * Created by alexey-n on 22.01.15.
 */
public class NodeBuilder {

    private String name;
    private Map<String, PerlModule> modules;
    private Matcher<String> moduleNameMatcher;
    private Matcher<String> packageNameMatcher;
    private Matcher<String> functionNameMatcher;
    private FunctionNameFormat format;
    private CombineType combineType;


    public NodeBuilder(String name, Map<String, PerlModule> modules) {
        this.name = name;
        this.modules = modules;
        combineType = CombineType.OR;
        format = FunctionNameFormat.ONLY_FNAME;
        moduleNameMatcher = null;
        packageNameMatcher = null;
        functionNameMatcher = null;
    }

    public NodeBuilder withModuleName(Matcher<String> matcher) {
        moduleNameMatcher = matcher;
        return this;
    }

    public NodeBuilder withPackageName(Matcher<String> matcher) {
        packageNameMatcher = matcher;
        return this;
    }

    public NodeBuilder withCombineType(CombineType type) {
        combineType = type;
        return this;
    }

    public NodeBuilder withFunctionName(Matcher<String> matcher) {
        functionNameMatcher = matcher;
        return this;
    }

    public NodeBuilder withFunctionNameFormat(FunctionNameFormat functionNameFormat) {
        format = functionNameFormat;
        return this;
    }

    public String getName() {
        return name;
    }

    public FunctionTree build() {
        FunctionTree tree = new FunctionTree();
        tree.setName(name);

        for (PerlModule module : modules.values()) {
            if (!validateModuleName(module.getPerlPackage())) {
                continue;
            }
            for (PerlFunction function : module.getDeclaredFunctions().values()) {
                boolean functionMatchResult = functionNameMatcher != null ? functionNameMatcher.
                        matches(function.getFunctionName()) : true;
                if(!functionMatchResult) {
                    continue;
                }
                FunctionTree child = new FunctionTree();
                child.setName(getFunctionName(function));
                System.out.println(function.getPerlModule().getPerlPackage() + "->" + function.getFunctionName());
                HashMap<String, String> callMap = new HashMap<String, String>();
                callMap.put(function.getPerlModule().getPerlPackage() + "->" + function.getFunctionName(), "");
                boolean result = TreeHelper.recoursiveChildAdd(function, child, 1, callMap);
                child.setChanged(result);
                tree.addChildren(child);
            }
        }

        return tree;
    }

    public boolean validateModuleName(String perlPackage) {
        String[] path = perlPackage.split("::");
        String moduleName = path[path.length - 1];
        String packageName = perlPackage.replace("::" + moduleName, "").replace(moduleName, "");
        boolean module = moduleNameMatcher != null ? moduleNameMatcher.matches(moduleName) : true;
        boolean pack = packageNameMatcher != null ? packageNameMatcher.matches(packageName) : true;
        boolean result = false;
        switch (combineType) {
            case OR:
                result = module || pack;
                break;
            case AND:
                result = module && pack;
                break;
            case XOR:
                result = (module || pack) && (!(module && pack));
                break;
            case NOT:
                result = !module && !pack;
                break;
        }
        return result;
    }

    private String getFunctionName(PerlFunction function) {
        String result = "";
        switch (format) {
            case ONLY_FNAME:
                result = function.getFunctionName();
                break;
            case PACKAGE_AND_NAME:
                result = function.getPerlModule().getPerlPackage() + "->" + function.getFunctionName();
                break;
        }

        return result;
    }
}
