package ru.yandex.direct.proxy.model;

import java.io.IOException;
import java.util.List;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.mchange.v1.lang.ClassUtils;
import one.util.streamex.StreamEx;
import org.jooq.Field;
import org.jooq.Identity;
import org.jooq.Record;
import org.jooq.StoreQuery;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.proxy.model.web.StepItem;
import ru.yandex.direct.proxy.model.web.callstep.ExecuteStepRequest;
import ru.yandex.direct.proxy.model.web.callstep.Parameter;

import static com.fasterxml.jackson.annotation.PropertyAccessor.FIELD;
import static com.google.common.base.Preconditions.checkState;

public class Converter {
    private static Logger logger = LoggerFactory.getLogger(Converter.class);
    private static ObjectMapper objectMapperForParse = new ObjectMapper().addMixIn(StoreQuery.class, StoreQueryMixIn.class);
    private static ObjectMapper objectMapperForWrite = new ObjectMapper()
            .configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false)
            .setVisibility(FIELD, JsonAutoDetect.Visibility.ANY)
            .setSerializationInclusion(JsonInclude.Include.NON_NULL);


    public static ExecuteStepData toExecuteStepData(ExecuteStepRequest executeStepRequest) {
        return new ExecuteStepData()
                .withArgs(executeStepRequest.getArgs())
                .withStepPath(stringToStepPath(executeStepRequest.getStepPath()))
                .withToken(executeStepRequest.getToken());
    }

    public static StepPath stringToStepPath(String rawStepPath) {
        String[] stepPathParts = rawStepPath.split("\\.");
        checkState(stepPathParts.length == 3, "invalid step path");

        return new StepPath(StepTypeEnum.fromString(stepPathParts[0]), stepPathParts[1], stepPathParts[2]);
    }

    public static ParameterData toParameterData(Parameter parameter) {
        ParameterData result = new ParameterData();
        result.withParameterName(parameter.getName());
        try {
            result.withClazz(ClassUtils.forName(parameter.getClazz()));
        } catch (ClassNotFoundException e) {
            logger.warn("Cannot deserialize parameter class");
        }
        result.withValue(parseClass(parameter.getValue(), result.getClazz()));
        return result;

    }

    public static List<StepItem> toStepItems(List<StepItemData> stepItemDatas) {
        return StreamEx.of(stepItemDatas).map(Converter::toStepItem).toList();
    }

    private static StepItem toStepItem(StepItemData stepItemData) {
        List<ParameterData<?>> parameterDataList = stepItemData.getParameterDataList();
        List<Parameter> parameters = StreamEx.of(parameterDataList).map(Converter::toParameter).toList();
        return new StepItem()
                .withParameterList(parameters)
                .withReturnType(stepItemData.getReturnType())
                .withStepPath(stepItemData.getStepPath());
    }

    private static Parameter toParameter(ParameterData parameterData) {
        Parameter result = new Parameter();
        result.withName(parameterData.getName());
        result.withClazz(parameterData.getClazz().getName());

        result.withValue(valueToString(parameterData.getValue()));
        return result;

    }

    private static String valueToString(Object value) {
        try {
            return objectMapperForWrite.writer().writeValueAsString(value);
        } catch (JsonProcessingException e) {
            logger.warn("Cannot serialize parameter value");
        }
        return null;
    }

    /**
     * Из за двух методов со схожей сигнатурой падает Jakson при попытке десериализации: https://st.yandex-team.ru/DIRECT-88768
     * Поэтому явно игнорим обработку этих методов: https://github.com/FasterXML/jackson-docs/wiki/JacksonMixInAnnotations
     */
    private interface StoreQueryMixIn<R extends Record> {

        @JsonIgnore
        void setReturning(Identity<R, ?> identity);

        @JsonIgnore
        void setReturning(Field<?>... fields);
    }

    public static <T> T parseClass(String argument, Class<T> classOfT) {
        try {
            return objectMapperForParse.readValue(argument, classOfT);
        } catch (IOException e) {
            throw new IllegalArgumentException("invalid args", e);
        }
    }

    public static StepItemData fromStepsStorageData(StepPath stepPath, List<ParameterData<?>> parameters,
            Class returnType)
    {
        return new StepItemData()
                .withParameterDataList(parameters)
                .withReturnType(returnType.getTypeName())
                .withStepPath(stepPath.toString());
    }
}
