package ru.yandex.direct.proxy.service;

import java.time.LocalDateTime;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.proxy.model.Converter;
import ru.yandex.direct.proxy.model.ExecuteStepData;
import ru.yandex.direct.proxy.model.web.callstep.CallStepErrorResponse;
import ru.yandex.direct.proxy.model.web.callstep.CallStepRequest;
import ru.yandex.direct.proxy.model.web.callstep.CallStepResponse;
import ru.yandex.direct.proxy.model.web.callstep.CallStepSuccessResponse;
import ru.yandex.direct.proxy.model.web.callstep.ExecuteStepRequest;

@Service
public class CallStepService {
    private final CallStepsHelper callStepsHelper;
    private final Logger log;
    private final TokenValidationService tokenValidationService;
    private final TokenStorageService tokenStorageService;

    public CallStepService(CallStepsHelper callStepsHelper,
            TokenValidationService tokenValidationService,
            TokenStorageService tokenStorageService)
    {
        this.tokenStorageService = tokenStorageService;
        this.log = LoggerFactory.getLogger(this.getClass());
        this.callStepsHelper = callStepsHelper;
        this.tokenValidationService = tokenValidationService;
    }

    public Object callApiStep(CallStepRequest callStepRequest) {
        if (!tokenValidationService.validate(callStepRequest.getToken())) {
            return "invalid token: " + callStepRequest.getToken();
        }
        tokenStorageService.get(callStepRequest.getToken()).updateLastChange(LocalDateTime.now());

        return callStepsHelper.callApiStep(callStepRequest);
    }

    public CallStepResponse executeStep(ExecuteStepRequest executeStepRequest) {
        log.info("request: ", executeStepRequest);

        ExecuteStepData executeStepData = Converter.toExecuteStepData(executeStepRequest);

        if (!tokenValidationService.validate(executeStepRequest.getToken())) {
            return new CallStepErrorResponse().withError("invalid token: " + executeStepRequest.getToken());
        }
        tokenStorageService.get(executeStepRequest.getToken()).updateLastChange(LocalDateTime.now());

        try {
            Object callStepResult = callStepsHelper.executeStep(executeStepData);
            return new CallStepSuccessResponse().withResult(callStepResult);
        } catch (IllegalStateException e) {
            log.warn("error result", e.getMessage());
            return new CallStepErrorResponse().withError(e.getMessage());
        }
    }

    public CallStepResponse callStepNew(CallStepRequest callStepRequest) {
        log.info("request: ", callStepRequest);
        if (!tokenValidationService.validate(callStepRequest.getToken())) {
            return new CallStepErrorResponse().withError("invalid token: " + callStepRequest.getToken());
        }
        tokenStorageService.get(callStepRequest.getToken()).updateLastChange(LocalDateTime.now());

        try {
            Object callStepResult = callStepsHelper.callStep(callStepRequest);
            return new CallStepSuccessResponse().withResult(callStepResult);
        } catch (IllegalStateException e) {
            log.warn("error result", e.getMessage());
            return new CallStepErrorResponse().withError(e.getMessage());
        }
    }

}
