package ru.yandex.direct.proxy.service;

import java.util.List;
import java.util.Map;
import java.util.function.Function;

import com.google.common.collect.BiMap;
import com.google.common.collect.ImmutableMap;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Component;

import ru.yandex.autotests.direct.cmd.DirectCmdSteps;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.directapi.darkside.steps.DarkSideSteps;
import ru.yandex.autotests.directapi.steps.UserSteps;
import ru.yandex.direct.proxy.model.Converter;
import ru.yandex.direct.proxy.model.ExecuteStepData;
import ru.yandex.direct.proxy.model.ParameterData;
import ru.yandex.direct.proxy.model.StepPath;
import ru.yandex.direct.proxy.model.StepTypeEnum;
import ru.yandex.direct.proxy.model.web.callstep.CallStepRequest;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.proxy.service.Utils.createStepClassToTypeMap;

@Component
public class CallStepsHelper {
    //todo убрать после полного перехода на execute
    private final Map<StepTypeEnum, Function<CallStepRequest, Object>> stepTypeToCallMethod;
    //работает в обе стороны, чтобы вызывать по string и превращать в string после reflection
    private final BiMap<StepTypeEnum, Class<?>> stepClassByType;
    private final TokenStorageService tokenStorageService;
    private final StepsStorage stepsStorage;

    public CallStepsHelper(TokenStorageService tokenStorageService,
            StepsStorage stepsStorage)
    {
        this.stepsStorage = stepsStorage;
        this.stepTypeToCallMethod = ImmutableMap.<StepTypeEnum, Function<CallStepRequest, Object>>builder()
                .put(StepTypeEnum.API, this::callApiStep)
                .put(StepTypeEnum.CMD, this::callCmdStep)
                .put(StepTypeEnum.DARK_SIDE, this::callDarkSideStep)
                .put(StepTypeEnum.DB, this::callDbSteps)
                .build();
        this.tokenStorageService = tokenStorageService;
        stepClassByType = createStepClassToTypeMap().inverse();

    }

    public <T> Object executeStep(ExecuteStepData executeStepRequest) {
        StepPath stepPath = executeStepRequest.getStepPath();
        Class<T> stepType = (Class<T>) stepClassByType.get(stepPath.getStepsTypeEnum());
        T stepsObject = tokenStorageService.get(executeStepRequest.getToken()).getSteps(stepType);
        return stepsStorage.callStepByPathWithArgs(stepsObject, stepPath, executeStepRequest.getArgs());
    }

    public Object callStep(CallStepRequest callStepRequest) {
        StepTypeEnum stepsTypeEnum = tokenStorageService.get(callStepRequest.getToken()).getStepsTypeEnum();
        if (stepsTypeEnum == null) {
            return callApiStep(callStepRequest);
        }
        checkState(callStepRequest.getMethods().size() == 2, "invalid path of step");
        return stepTypeToCallMethod.get(stepsTypeEnum).apply(callStepRequest);
    }

    public Object callApiStep(CallStepRequest callStepRequest) {
        UserSteps step = tokenStorageService.get(callStepRequest.getToken()).getSteps(UserSteps.class);
        List<Object> args = StreamEx.of(callStepRequest.getParameterList())
                .map(Converter::toParameterData)
                .map(ParameterData::getValue)
                .toList();

        StepPath stepPath = new StepPath(StepTypeEnum.API, callStepRequest.getMethods().get(0),
                callStepRequest.getMethods().get(1));
        return stepsStorage.callStepByPathWithArgsWithResolvedClasses(step, stepPath, args);
    }

    private Object callCmdStep(CallStepRequest callStepRequest) {
        DirectCmdSteps step = tokenStorageService.get(callStepRequest.getToken()).getSteps(DirectCmdSteps.class);
        List<Object> args = StreamEx.of(callStepRequest.getParameterList())
                .map(Converter::toParameterData)
                .map(ParameterData::getValue)
                .toList();

        StepPath stepPath = new StepPath(StepTypeEnum.CMD, callStepRequest.getMethods().get(0),
                callStepRequest.getMethods().get(1));
        return stepsStorage.callStepByPathWithArgsWithResolvedClasses(step, stepPath, args);
    }

    private Object callDarkSideStep(CallStepRequest callStepRequest) {
        DarkSideSteps step = tokenStorageService.get(callStepRequest.getToken()).getSteps(DarkSideSteps.class);
        List<Object> args = StreamEx.of(callStepRequest.getParameterList())
                .map(Converter::toParameterData)
                .map(ParameterData::getValue)
                .toList();

        StepPath stepPath = new StepPath(StepTypeEnum.DARK_SIDE, callStepRequest.getMethods().get(0),
                callStepRequest.getMethods().get(1));
        return stepsStorage.callStepByPathWithArgsWithResolvedClasses(step, stepPath, args);
    }

    private Object callDbSteps(CallStepRequest callStepRequest) {
        DirectJooqDbSteps step = tokenStorageService.get(callStepRequest.getToken()).getSteps(DirectJooqDbSteps.class);
        List<Object> args = StreamEx.of(callStepRequest.getParameterList())
                .map(Converter::toParameterData)
                .map(ParameterData::getValue)
                .toList();

        StepPath stepPath = new StepPath(StepTypeEnum.DB, callStepRequest.getMethods().get(0),
                callStepRequest.getMethods().get(1));
        return stepsStorage.callStepByPathWithArgsWithResolvedClasses(step, stepPath, args);
    }
}
