package ru.yandex.direct.proxy.service;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.MultiMap;
import org.apache.commons.collections4.map.MultiValueMap;
import org.reflections.Reflections;
import org.reflections.scanners.MethodAnnotationsScanner;
import org.reflections.scanners.MethodParameterScanner;
import org.reflections.scanners.SubTypesScanner;
import org.reflections.util.ClasspathHelper;

import ru.yandex.autotests.direct.utils.BaseSteps;
import ru.yandex.autotests.directapi.steps.BaseApiSteps;
import ru.yandex.autotests.directapi.steps.UserSteps;
import ru.yandex.direct.proxy.model.ParameterData;
import ru.yandex.direct.proxy.model.StepInfo;
import ru.yandex.direct.proxy.model.StepPath;
import ru.yandex.direct.proxy.model.StepTypeEnum;
import ru.yandex.qatools.allure.annotations.Step;

import static ru.yandex.direct.proxy.model.StepTypeEnum.fromStepClass;
import static ru.yandex.direct.proxy.service.StepsStorage.STEP_AGGREGATOR_CLASSES;

public class StepStorageCreator {
    private MultiMap<StepPath, List<ParameterData>> paramsByFullPathOfStep;
    private Map<StepTypeEnum, Map<String, MultiMap<String, Method>>> stepMethodsByPathsParts;
    private Map<String, Method> subStepGetterByFullPath;

    StepStorageCreator() {
        this.paramsByFullPathOfStep = new MultiValueMap<>();
        this.stepMethodsByPathsParts = new HashMap<>();
        this.subStepGetterByFullPath = new HashMap<>();
    }

    private static BiMap<Class, List<Method>> feelStepClassGetters(Reflections r) {
        BiMap<Class, List<Method>> stepClassGetters = StreamEx.of((r.getSubTypesOf(BaseSteps.class)))
                .flatMap(t -> StreamEx.of(r.getMethodsReturn(t)))
                .filter(t -> STEP_AGGREGATOR_CLASSES.contains(t.getDeclaringClass()))
                .groupingBy(Method::getDeclaringClass, HashBiMap::create, Collectors.toList());
        BiMap<Class, List<Method>> apiStepClassGetters = StreamEx.of((r.getSubTypesOf(BaseApiSteps.class)))
                .flatMap(t -> StreamEx.of(r.getMethodsReturn(t)))
                .filter(t -> t.getDeclaringClass().equals(UserSteps.class))
                .groupingBy(Method::getDeclaringClass, HashBiMap::create, Collectors.toList());

        stepClassGetters.putAll(apiStepClassGetters);
        return stepClassGetters;
    }

    private static List<ParameterData> convertParameters(java.lang.reflect.Parameter[] parameters) {
        return StreamEx.of(parameters)
                .map(t -> new ParameterData<>().withParameterName(t.getName()).withClazz(t.getType()))
                .map(ParameterData::generateValue)
                .toList();
    }

    public StepsStorage create() {
        fillStepsStorage();
        return new StepsStorage(paramsByFullPathOfStep, stepMethodsByPathsParts, subStepGetterByFullPath);
    }

    private void fillStepsStorage() {
        Reflections r = new Reflections(ClasspathHelper.forPackage("ru.yandex.autotests.direct"),
                ClasspathHelper.forPackage("ru.yandex.autotests.directapi"),
                new SubTypesScanner(), new MethodAnnotationsScanner(), new MethodParameterScanner());

        Map<Class, List<Method>> steps = StreamEx.of(r.getMethodsAnnotatedWith(Step.class))
                .filter(m -> Modifier.isPublic(m.getModifiers()))
                .groupingBy(Method::getDeclaringClass);

        BiMap<Class, List<Method>> stepClassGetters = feelStepClassGetters(r);

        List<StepInfo> stepsList = EntryStream.of(stepClassGetters)
                .flatMapValues(StreamEx::of)
                .mapKeyValue((k, v) -> new StepInfo().withStepClass(v.getReturnType()).withStepClassGetter(v)
                        .withAggregatorClass(k))
                .toList();
        EntryStream.of(steps)
                .forKeyValue((key, val) -> {
                    for (Method stepMethod : val) {
                        StepInfo si = StreamEx.of(stepsList)
                                .filter(t -> t.getStepClass().equals(key))
                                .findFirst().orElse(null);
                        if (si == null) {
                            return;
                        }

                        StepTypeEnum stepsTypeEnum = fromStepClass(si.getAggregatorClass());
                        StepPath stepPath =
                                new StepPath(stepsTypeEnum, si.getStepClassGetter().getName(), stepMethod.getName());

                        paramsByFullPathOfStep.put(stepPath, convertParameters(stepMethod.getParameters()));

                        Map<String, MultiMap<String, Method>> methodsByStepGetterName =
                                stepMethodsByPathsParts.computeIfAbsent(stepsTypeEnum, x -> new HashMap<>());
                        MultiMap<String, Method> methodByName = methodsByStepGetterName
                                .computeIfAbsent(si.getStepClassGetter().getName(), x -> new MultiValueMap<>());
                        subStepGetterByFullPath.put(stepPath.toString(), si.getStepClassGetter());
                        methodByName.put(stepMethod.getName(), stepMethod);
                    }
                });

    }
}
