package ru.yandex.direct.proxy.service;

import java.lang.reflect.Executable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;

import com.google.common.primitives.Primitives;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections4.MultiMap;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.autotests.direct.cmd.DirectCmdSteps;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.web.api.steps.DirectWebApiSteps;
import ru.yandex.autotests.directapi.darkside.steps.DarkSideSteps;
import ru.yandex.direct.proxy.model.Converter;
import ru.yandex.direct.proxy.model.ParameterData;
import ru.yandex.direct.proxy.model.StepArgument;
import ru.yandex.direct.proxy.model.StepItemData;
import ru.yandex.direct.proxy.model.StepPath;
import ru.yandex.direct.proxy.model.StepTypeEnum;

import static java.util.Arrays.asList;
import static ru.yandex.direct.proxy.model.Converter.fromStepsStorageData;

public class StepsStorage {
    static final List<Class> STEP_AGGREGATOR_CLASSES =
            asList(DirectJooqDbSteps.class, DirectCmdSteps.class, DarkSideSteps.class, DirectWebApiSteps.class);

    private MultiMap<StepPath, List<ParameterData>> paramsByFullPathOfStep;
    private Map<StepTypeEnum, Map<String, MultiMap<String, Method>>> stepMethodsByPathsParts;
    private Map<String, Method> subStepGetterByFullPath;

    StepsStorage(MultiMap<StepPath, List<ParameterData>> paramsByFullPathOfStep,
                 Map<StepTypeEnum, Map<String, MultiMap<String, Method>>> stepMethodsByPathsParts,
                 Map<String, Method> subStepGetterByFullPath) {
        this.paramsByFullPathOfStep = paramsByFullPathOfStep;
        this.stepMethodsByPathsParts = stepMethodsByPathsParts;
        this.subStepGetterByFullPath = subStepGetterByFullPath;
    }

    public static StepsStorage create() {
        return new StepStorageCreator().create();
    }


    public List<StepItemData> searchSteps(String stepSubstring) {
        Predicate<StepPath> stepPathPredicate = path -> StringUtils.containsIgnoreCase(path.toString(), stepSubstring);
        if (StreamEx.of(paramsByFullPathOfStep.keySet()).nonNull().noneMatch(stepPathPredicate)) {
            return Collections.emptyList();
        }
        List<StepPath> stepPathList = StreamEx.of(paramsByFullPathOfStep.keySet())
                .nonNull()
                .filter(stepPathPredicate)
                .toList();

        Comparator<StepItemData> comparator = Comparator.comparing(StepItemData::getStepPath);
        return StreamEx.of(stepPathList)
                .mapToEntry(fullStepPath -> paramsByFullPathOfStep.get(fullStepPath))
                .mapValues(parameters -> (List) parameters)
                .removeValues(List::isEmpty)
                .mapValues(parameters -> (List<ParameterData<?>>) parameters.get(0))
                .mapKeyValue((stepPath, parameters) -> fromStepsStorageData(stepPath, parameters,
                        getStepByPathAndParameterDatas(stepPath, parameters).getReturnType()))
                .sorted(comparator)
                .toList();
    }

    public Object callStepByPathWithArgs(Object stepObj, StepPath stepPath, List<StepArgument> args) {
        Method stepMethod = getStepByPathAndArgNames(stepPath, args);

        List<? extends Class<?>> argumentsClasses = StreamEx.of(stepMethod.getParameters())
                .map(Parameter::getType)
                .toList();
        try {
            Object subStep = subStepGetterByFullPath.get(stepPath.toString()).invoke(stepObj);
            Object[] argsWithResolvedClasses = EntryStream.zip(args, argumentsClasses)
                    .mapKeys(StepArgument::getValue)
                    .mapKeyValue(Converter::parseClass)
                    .toArray();

            return stepMethod.invoke(subStep, argsWithResolvedClasses);
        } catch (InvocationTargetException e) {
            throw new IllegalStateException("Can not call method with name: " + stepPath.getStepMethodName()
                    + " and args: " + StringUtils.join(args) + " because of steps error " + e
                    .getTargetException(), e);
        } catch (IllegalAccessException e) {
            throw new IllegalArgumentException("Can not call method with name: " + stepPath.getStepMethodName()
                    + " and args: " + StringUtils.join(args));
        }
    }

    public Object callStepByPathWithArgsWithResolvedClasses(Object stepObj, StepPath stepPath,
                                                            List<Object> argsWithResolvedTypes) {
        List<? extends Class<?>> argTypes = StreamEx.of(argsWithResolvedTypes)
                .map(Object::getClass)
                .toList();
        Method stepMethod = getStepByPathAndArgTypes(stepPath, argTypes);
        try {
            Object subStep = subStepGetterByFullPath.get(stepPath.toString()).invoke(stepObj);
            return stepMethod.invoke(subStep, argsWithResolvedTypes.toArray());
        } catch (InvocationTargetException e) {
            throw new IllegalStateException("Can not call method with name: " + stepPath.getStepMethodName()
                    + " and args: " + StringUtils.join(argsWithResolvedTypes) + " because of steps error " + e
                    .getTargetException(), e);
        } catch (IllegalAccessException e) {
            throw new IllegalArgumentException("Can not call method with name: " + stepPath.getStepMethodName()
                    + " and args: " + StringUtils.join(argsWithResolvedTypes));
        }
    }

    private Method getStepByPathAndArgNames(StepPath stepPath, List<StepArgument> args) {
        List<Method> stepsWithSamePath = (List<Method>) stepMethodsByPathsParts.get(stepPath.getStepsTypeEnum())
                .get(stepPath.getSubStepGetterName())
                .get(stepPath.getStepMethodName());

        List<String> argNames = StreamEx.of(args)
                .map(StepArgument::getName)
                .toList();
        return getMethodByArgNames(stepsWithSamePath, argNames);
    }

    private Method getStepByPathAndParameterDatas(StepPath stepPath, List<ParameterData<?>> args) {
        List<? extends Class<?>> types = StreamEx.of(args)
                .map(ParameterData::getClazz)
                .toList();
        return getStepByPathAndArgTypes(stepPath, types);
    }

    private Method getStepByPathAndArgTypes(StepPath stepPath, List<? extends Class<?>> argTypes) {
        List<Method> stepsWithSamePath = (List<Method>) stepMethodsByPathsParts.get(stepPath.getStepsTypeEnum())
                .get(stepPath.getSubStepGetterName())
                .get(stepPath.getStepMethodName());
        return getMethodByArgTypes(stepsWithSamePath, argTypes);
    }

    private Method getMethodByArgNames(List<Method> methods, List<String> argumentNames) {
        return StreamEx.of(methods)
                .mapToEntry(Executable::getParameters)
                .mapValues(StepsStorage::parameterNames)
                .findFirst(entry -> CollectionUtils.isEqualCollection(entry.getValue(), argumentNames))
                .orElseThrow(IllegalArgumentException::new)
                .getKey();

    }

    private static List<String> parameterNames(java.lang.reflect.Parameter[] parameters) {
        return StreamEx.of(parameters)
                .map(java.lang.reflect.Parameter::getName)
                .toList();
    }

    private Method getMethodByArgTypes(List<Method> methods, List<? extends Class<?>> argumentTypes) {
        List<? extends Class<?>> wrappedTypes = StreamEx.of(argumentTypes).map(Primitives::wrap).toList();
        return StreamEx.of(methods)
                .mapToEntry(Executable::getParameters)
                .mapValues(Utils::parameterClasses)
                .findFirst(entry -> CollectionUtils
                        .isEqualCollection(entry.getValue(), wrappedTypes))
                .orElseThrow(IllegalArgumentException::new)
                .getKey();

    }

}
