package ru.yandex.autotests.direct.utils.clients.blackbox;

import java.net.URI;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import com.google.gson.annotations.SerializedName;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import org.apache.http.client.utils.URIBuilder;

import ru.yandex.autotests.direct.utils.clients.tvm.ServiceTicketProvider;
import ru.yandex.autotests.direct.utils.clients.tvm.ServiceTicketProviderProxy;
import ru.yandex.autotests.direct.utils.config.DirectTestRunProperties;
import ru.yandex.autotests.httpclientlite.HttpClientLite;
import ru.yandex.autotests.httpclientlite.core.request.utils.keyvalue.annotations.SerializeKey;
import ru.yandex.autotests.httpclientlite.utils.client.HttpClientBuilderUtils;
import ru.yandex.autotests.httpclientlite.utils.client.HttpClientConfig;

import static java.util.Collections.emptyList;
import static ru.yandex.autotests.direct.utils.clients.tvm.ServiceTicketProvider.SERVICE_TICKET_HEADER;

/**
 * Клиент для паспортного blackbox-а
 *
 * @author egorovmv
 */
public final class BlackboxClient {
    private final HttpClientLite client;
    private final URI serviceUrl;
    private ServiceTicketProvider serviceTicketProvider;

    public BlackboxClient(URI serviceUrl) {
        Objects.requireNonNull(serviceUrl, "serviceUrl");

        this.client = new HttpClientLite.Builder()
                .withClient(
                        HttpClientBuilderUtils.defaultTestConfiguration(
                                new HttpClientConfig())
                                .build())
                .formRequest()
                .jsonResponse()
                .consoleLogger()
                .build();
        this.serviceUrl = serviceUrl;
        this.serviceTicketProvider = new ServiceTicketProviderProxy();
    }

    public PassportUserInfo getUserInfoByLogin(String login) {
        Objects.requireNonNull(login, "login");

        long tvmDstClientId = DirectTestRunProperties.getInstance().getDirectPassportBlackboxTvmClientId();
        try {
            Map<String, String> headers = Collections.singletonMap(SERVICE_TICKET_HEADER,
                    serviceTicketProvider.getServiceTicket(tvmDstClientId));

            UserInfoResponse response = client.get(
                    new URIBuilder(serviceUrl)
                            .addParameter("method", "userinfo")
                            .addParameter("login", login)
                            // Пока так, так как не очень понятно как из NetworkInterface-а выбрать нужный и чтобы
                            // определить ip-адрес
                            .addParameter("userip", "127.0.0.1")
                            .addParameter("emails", "getall")
                            .addParameter("regname", "yes")
                            .addParameter("format", "json")
                            .build(),
                    UserInfoResponse.class,
                    headers);

            return Optional.ofNullable(response.getUsers())
                    .flatMap(l -> l.stream().findFirst())
                    .map(u -> new PassportUserInfo(
                            login,
                            Optional.ofNullable(u.getUid())
                                    .map(UserUid::getValue)
                                    .get(),
                            Optional.ofNullable(u.getDisplayName())
                                    .map(UserDisplayName::getName)
                                    .orElse(null),
                            u.getEmails().stream()
                                    .sorted(Comparator.comparing(UserEmail::getDefaultStatus).reversed())
                                    .findFirst()
                                    .map(UserEmail::getEmail)
                                    .orElse(null)))
                    .orElse(null);
        } catch (Exception e) {
            throw new BlackboxClientException(e);
        }
    }

    public static final class BlackboxClientException extends RuntimeException {
        BlackboxClientException(Throwable cause) {
            super(cause);
        }
    }

    private static final class UserInfoResponse {
        @SerializeKey("users")
        private List<UserInfo> users;

        @Override
        public String toString() {
            return ReflectionToStringBuilder.toString(this);
        }

        public List<UserInfo> getUsers() {
            return users;
        }

        public void setUsers(List<UserInfo> users) {
            this.users = users;
        }
    }

    private static final class UserInfo {
        @SerializedName("uid")
        private UserUid uid;
        @SerializedName("display_name")
        private UserDisplayName displayName;
        @SerializedName("address-list")
        private List<UserEmail> emails;

        @Override
        public String toString() {
            return ReflectionToStringBuilder.toString(this);
        }

        public UserUid getUid() {
            return uid;
        }

        public void setUid(UserUid uid) {
            this.uid = uid;
        }

        public UserDisplayName getDisplayName() {
            return displayName;
        }

        public void setDisplayName(UserDisplayName displayName) {
            this.displayName = displayName;
        }

        public List<UserEmail> getEmails() {
            return emails != null ? emails : emptyList();
        }

        public void setEmails(List<UserEmail> emails) {
            this.emails = emails;
        }
    }

    public static final class UserUid {
        @SerializedName("value")
        private String value;

        @Override
        public String toString() {
            return ReflectionToStringBuilder.toString(this);
        }

        public String getValue() {
            return value;
        }

        public void setValue(String value) {
            this.value = value;
        }
    }

    public static final class UserDisplayName {
        @SerializedName("name")
        private String name;

        @Override
        public String toString() {
            return ReflectionToStringBuilder.toString(this);
        }

        public String getName() {
            return name;
        }

        public void setName(String name) {
            this.name = name;
        }
    }

    public static final class UserEmail {
        @SerializedName("address")
        private String email;
        @SerializedName("default")
        private boolean defaultStatus;

        @Override
        public String toString() {
            return ReflectionToStringBuilder.toString(this);
        }

        public String getEmail() {
            return email;
        }

        public void setEmail(String email) {
            this.email = email;
        }

        public boolean getDefaultStatus() {
            return defaultStatus;
        }

        public void setDefaultStatus(boolean defaultStatus) {
            this.defaultStatus = defaultStatus;
        }
    }
}
