package ru.yandex.autotests.direct.utils.clients.tus;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import com.google.common.collect.ImmutableMap;
import org.apache.http.client.utils.URIBuilder;

import ru.yandex.autotests.httpclientlite.HttpClientLite;
import ru.yandex.autotests.httpclientlite.utils.client.HttpClientBuilderUtils;
import ru.yandex.autotests.httpclientlite.utils.client.HttpClientConfig;

public class TusClient {

    private static final String CONSUMER_NAME = "direct-tests";
    private static final String CREATING_ACCOUNT_PATH = "/1/create_account/portal/";
    private static final String URL_SCHEME = "https";

    private static final Map<String, String> REQUEST_PARAMS = ImmutableMap.of(
            "env", "prod",
            "tus_consumer", CONSUMER_NAME
    );
    private static final String LOGIN_PARAM = "login";
    private static final String PASSWORD_PARAM = "password";

    private final TusConfig config;
    private final HttpClientLite client;

    public TusClient() {
        this.config = new TusConfig();
        this.client = new HttpClientLite.Builder()
                .withClient(HttpClientBuilderUtils.defaultTestConfiguration(new HttpClientConfig()).build())
                .formRequest()
                .jsonResponse()
                .consoleLogger()
                .build();
    }

    public TusCreateAccountResponse createAccount() {
        URI uri = buildUri(REQUEST_PARAMS);
        return getTusCreateAccountResponse(uri);
    }

    public TusCreateAccountResponse createAccount(String login, String password) {
        Map<String, String> requestParams = new HashMap<>(REQUEST_PARAMS);
        requestParams.put(LOGIN_PARAM, login);
        requestParams.put(PASSWORD_PARAM, password);
        URI uri = buildUri(requestParams);

        return getTusCreateAccountResponse(uri);
    }

    private TusCreateAccountResponse getTusCreateAccountResponse(URI uri) {
        try {
            Map<String, String> headers = Collections.singletonMap("Authorization", "OAuth " + config.getToken());
            return client.get(uri, TusCreateAccountResponse.class, headers);
        } catch (RuntimeException e) {
            throw new RuntimeException("Failed to create account with TUS", e);
        }
    }

    private URI buildUri(Map<String, String> requestParams) {
        URIBuilder uriBuilder = new URIBuilder();
        uriBuilder.setScheme(URL_SCHEME);
        uriBuilder.setHost(config.getHost());
        uriBuilder.setPath(CREATING_ACCOUNT_PATH);
        requestParams.forEach(uriBuilder::addParameter);
        try {
            return uriBuilder.build();
        } catch (URISyntaxException e) {
            throw new RuntimeException("Failed to create URI for TUS", e);
        }
    }
}
