package ru.yandex.autotests.direct.utils.clients.tvm;

import java.lang.reflect.Type;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.time.Duration;
import java.util.Base64;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.Form;
import javax.ws.rs.core.Response;

import com.google.common.base.Joiner;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.glassfish.jersey.apache.connector.ApacheClientProperties;
import org.glassfish.jersey.client.ClientProperties;

import ru.yandex.autotests.direct.utils.config.TvmSecret;

import static javax.ws.rs.client.Entity.form;


/**
 * Провайдер сервисных TVM-тикетов.
 */
@ParametersAreNonnullByDefault
public class TvmTicketsProvider implements ServiceTicketProvider {
    private static final long MAX_CACHE_SIZE = 1_000;
    private static final Duration CACHE_TTL = Duration.ofMinutes(50);
    private static final String DEFAULT_SCOPES = "";
    private static final Joiner JOINER = Joiner.on('|');
    private static final Type TVM_RESPONSE_TYPE = new TypeToken<Map<String, Map<String, String>>>() {
    }.getType();

    private static LoadingCache<TvmSecret, TvmTicketsProvider> INSTANCE_CACHE = CacheBuilder.newBuilder()
            .initialCapacity(3)
            .maximumSize(50)
            .build(new CacheLoader<TvmSecret, TvmTicketsProvider>() {
                @Override
                public TvmTicketsProvider load(TvmSecret tvmSecret) {
                    return new TvmTicketsProvider(tvmSecret);
                }
            });

    private final WebTarget service;
    private final LoadingCache<Long, String> cache;
    private final Base64.Encoder encoder;
    private final Mac hmacSHA256;
    private final Gson gson;

    private TvmTicketsProvider(TvmSecret secrets) {
        gson = new Gson();
        encoder = Base64.getUrlEncoder();

        long srcClientId = Long.parseLong(secrets.getTvmClientId());

        byte[] key = Base64.getUrlDecoder().decode(secrets.getTvmSecret());
        SecretKeySpec secretKey = new SecretKeySpec(key, "HmacSHA256");

        try {
            hmacSHA256 = Mac.getInstance("HmacSHA256");
            hmacSHA256.init(secretKey);
        } catch (NoSuchAlgorithmException | InvalidKeyException e) {
            throw new IllegalStateException(e);
        }

        cache = CacheBuilder.newBuilder()
                .maximumSize(MAX_CACHE_SIZE)
                .expireAfterWrite(CACHE_TTL.getSeconds(), TimeUnit.SECONDS)
                .build(new CacheLoader<Long, String>() {
                    public String load(Long key) {
                        return getTicket(srcClientId, key);
                    }
                });
        service = ClientBuilder.newBuilder()
                .build()
                .property(ApacheClientProperties.CONNECTION_MANAGER, new PoolingHttpClientConnectionManager())
                .property(ClientProperties.CONNECT_TIMEOUT, 3_000)
                .property(ClientProperties.READ_TIMEOUT, 5_000)
                .target("https://tvm-api.yandex.net/2/ticket");
    }

    public static TvmTicketsProvider getInstance(TvmSecret tvmSecret) {
        return INSTANCE_CACHE.getUnchecked(tvmSecret);
    }

    private String getTicket(long srcClientId, long dstClientId) {
        long ts = System.currentTimeMillis() / 1000;

        String sign = getSign(ts, dstClientId);
        String dst = Long.toString(dstClientId);
        Form form = new Form()
                .param("grant_type", "client_credentials")
                .param("src", Long.toString(srcClientId))
                .param("dst", dst)
                .param("ts", Long.toString(ts))
                .param("sign", sign);

        Response response = service.request().post(form(form));

        String body = response.readEntity(String.class);
        if (response.getStatus() == 200) {
            Map<String, Map<String, String>> result = gson.fromJson(body, TVM_RESPONSE_TYPE);
            Map<String, String> info = result.get(dst);
            if (info.containsKey("ticket")) {
                return info.get("ticket");
            } else if (info.containsKey("error")) {
                throw new RuntimeException("failed to obtain ticket: " + info.get("error"));
            }
        }
        throw new RuntimeException("failed to obtain ticket: status " + response.getStatus() + ", body: " + body);
    }

    private String getSign(long ts, long dstClientId) {
        String data = JOINER.join(ts, dstClientId, DEFAULT_SCOPES, "");
        byte[] result = hmacSHA256.doFinal(data.getBytes());
        return encoder.encodeToString(result);
    }

    public String getServiceTicket(long dstClientId) {
        return cache.getUnchecked(dstClientId);
    }
}
