package ru.yandex.autotests.direct.utils.matchers;

import org.hamcrest.Description;
import org.hamcrest.Factory;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeDiagnosingMatcher;

import java.beans.PropertyDescriptor;
import java.util.*;

import static org.apache.commons.lang.StringUtils.capitalize;
import static org.hamcrest.CoreMatchers.equalTo;
import static ru.yandex.autotests.irt.testutils.beans.BeanHelper.*;

/**
 * @author Roman Kuhta (kuhtich@yandex-team.ru)
 */
public class BeanEquals<T> extends TypeSafeDiagnosingMatcher<T> {
    private T expectedBean;
    private Set<String> shouldCheckFields;
    private BeanCompareStrategy beanCompareStrategy;

    public BeanEquals(T expectedBean) {
        this.expectedBean = expectedBean;
        beanCompareStrategy = new BeanCompareStrategy();
        findOutFieldsToCheck();
    }

    public BeanEquals<T> accordingStrategy(BeanCompareStrategy beanCompareStrategy) {
        this.beanCompareStrategy = beanCompareStrategy;
        for (String strategyFields : beanCompareStrategy.getFieldNames()){
            shouldCheckFields.add(strategyFields);
        }
        return this;
    }

    /**
     * Only specified fields will be take into account on beans comparison, all the other should be ignored
     *
     * @param fieldNames
     * @return
     */
    public BeanEquals<T> byFields(String... fieldNames) {
        shouldCheckFields.retainAll(Arrays.asList(fieldNames));
        return this;
    }

    /**
     * Let mather know fields that should be ignored in beans comparison
     * Field names should start with only small letter
     *
     * @param fieldNames
     * @return
     */
    public BeanEquals<T> ignoreFields(String... fieldNames) {
        shouldCheckFields.removeAll(Arrays.asList(fieldNames));
        return this;
    }

    private void findOutFieldsToCheck() {
        shouldCheckFields = new HashSet<>();
        for (PropertyDescriptor field : getPropertyDescriptorsForFields(expectedBean)) {
            if (isSimpleField(field.getPropertyType())) {
                shouldCheckFields.add(field.getName());
            }
        }
    }

    private Matcher getMatcherForField(String fieldName) {
        Matcher matcher = beanCompareStrategy.getFieldMatcher(fieldName);
        if (matcher == null) {
            matcher = equalTo(getProperty(expectedBean, fieldName));
            beanCompareStrategy.putFieldMatcher(fieldName, matcher);
        }
        return matcher;
    }

    @Override
    public void describeTo(Description description) {
        description.appendText("bean with properties {");
        for (String fieldName : shouldCheckFields) {
            Object expectedValue = getProperty(expectedBean, fieldName);
            if (expectedValue == null && beanCompareStrategy.getFieldMatcher(fieldName) == null) {
                continue;
            }
            description.appendText("\n\t").
                    appendText(capitalize(fieldName) + ": ");
            getMatcherForField(fieldName).describeTo(description);
        }
        description.appendText("\n}");
    }

    @Override
    protected boolean matchesSafely(T actualBean, Description mismatchDescription) {
        boolean matchesResult = true;
        for (String fieldName : shouldCheckFields) {
            Object expectedValue = getProperty(expectedBean, fieldName);
            if (expectedValue == null && beanCompareStrategy.getFieldMatcher(fieldName) == null) {
                continue;
            }
            Object actualValue = getProperty(actualBean, fieldName);
            if (!getMatcherForField(fieldName).matches(actualValue)) {
                matchesResult = false;
                mismatchDescription.appendText("[" + capitalize(fieldName) + "] ");
                getMatcherForField(fieldName).describeMismatch(actualValue, mismatchDescription);
                mismatchDescription.appendText("\n\t\t  ");
            }
        }
        return matchesResult;
    }

    /**
     * Creates a matcher that matches when the examined object has values for all of
     * its JavaBean properties (except arrays) that are equal to the corresponding values of the
     * specified bean
     * <p/>
     * For example:
     * <pre>assertThat(myBean, beanEquals(myExpectedBean))</pre>
     *
     * @param expectedBean the bean against which examined beans are compared
     */
    @Factory
    public static <T> BeanEquals<T> beanEquals(T expectedBean) {
        return new BeanEquals<>(expectedBean);
    }

    public T getExpectedBean() {
        return expectedBean;
    }
}
