package ru.yandex.autotests.direct.utils.matchers;

import org.hamcrest.Description;
import org.hamcrest.Factory;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeDiagnosingMatcher;
import ru.yandex.autotests.irt.testutils.json.JsonUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import static org.apache.commons.lang.StringUtils.capitalize;
import static org.hamcrest.core.AllOf.allOf;
import static ru.yandex.autotests.irt.testutils.beans.BeanHelper.getProperty;

/**
 * Created by mariabye on 15.09.14.
 */
public class IsApiBeanCollectionContaining<T> extends TypeSafeDiagnosingMatcher<T[]> {
    private Matcher<T>[] expectedBeanMatchers;
    private boolean exactMatch;
    private boolean sortedOrder;

    public IsApiBeanCollectionContaining(
            boolean exactMatch, boolean sortedOrder, Matcher<T>... expectedBeanMatchers) {
        this.expectedBeanMatchers = expectedBeanMatchers;
        this.exactMatch = exactMatch;
        this.sortedOrder = sortedOrder;
    }

    private T checkMatchPresent(List<T> actualItems, Matcher<T> matcher) {
        for (T actualBean : actualItems) {
            if (matcher.matches(actualBean)) {
                return actualBean;
            }
        }
        return null;
    }

    private T checkMatchFirst(List<T> actualItems, Matcher<T> matcher, Description mismatchDescription) {
        if (actualItems.size() > 0 && matcher.matches(actualItems.get(0))) {
            return actualItems.get(0);
        }
        return null;
    }

    @Override
    protected boolean matchesSafely(T[] actualBeansArray, Description mismatchDescription) {
        boolean matchesResult = true;
        List<T> actualBeans = Arrays.asList(actualBeansArray);
        List<T> cloneActual = new ArrayList<>();
        cloneActual.addAll(actualBeans);
        for (Matcher<T> expectedBeanMatcher : expectedBeanMatchers) {
            T matchItem = null;
            if (sortedOrder) {
                matchItem = checkMatchFirst(cloneActual, expectedBeanMatcher, mismatchDescription);
                if (cloneActual.size() > 0) {
                    cloneActual.remove(0);
                }
            } else {
                matchItem = checkMatchPresent(cloneActual, expectedBeanMatcher);
                if (matchItem != null) {
                    cloneActual.remove(matchItem);
                }
            }
            if (matchItem == null) {
                mismatchDescription.appendText("\nbean matches for \n");
                expectedBeanMatcher.describeTo(mismatchDescription);
                mismatchDescription.appendText("\n\twas not found\n\n");
                matchesResult = false;
            }
        }
        if (exactMatch && matchesResult && cloneActual.size() > 0) {
            mismatchDescription.appendText("collection contains extra items: \n[");
            for (T extra : cloneActual) {
                mismatchDescription.appendText("\n");
                mismatchDescription.appendText(JsonUtils.toString(extra));
                mismatchDescription.appendText("\n");

            }
            mismatchDescription.appendText("]\n");
        }
        if (exactMatch) {
            matchesResult = matchesResult && (cloneActual.size() == 0);

        }

        if (matchesResult == false) {
            mismatchDescription.appendText("the original collection was: \n[");
            for (T actualBean : actualBeans) {
                mismatchDescription.appendText("\n");
                mismatchDescription.appendText(JsonUtils.toString(actualBean));
                mismatchDescription.appendText("\n");

            }
            mismatchDescription.appendText("]");
        }
        return matchesResult;
    }

    @Override
    public void describeTo(Description description) {
        description.appendText("collection of beans contains:\n");
        for (Matcher<T> expectedBeanMatcher : expectedBeanMatchers) {
            expectedBeanMatcher.describeTo(description);
            description.appendText("\n");
        }
    }

    @Factory
    public static <T> IsApiBeanCollectionContaining<T> hasItems(Matcher<T>... expectedBeanMatchers) {
        return new IsApiBeanCollectionContaining<T>(true, false, expectedBeanMatchers);
    }

    @Factory
    public static <T> IsApiBeanCollectionContaining<T> containItems(Matcher<T>... expectedBeanMatchers) {
        return new IsApiBeanCollectionContaining<T>(false, false, expectedBeanMatchers);
    }

    @Factory
    public static <T> IsApiBeanCollectionContaining<T> hasSameOrderItems(Matcher<T>... expectedBeanMatchers) {
        return new IsApiBeanCollectionContaining<T>(false, true, expectedBeanMatchers);
    }
}
