package ru.yandex.autotests.direct.utils.money;

import ru.yandex.autotests.direct.utils.DirectUtilsException;

import java.math.BigDecimal;
import java.math.MathContext;
import java.math.RoundingMode;
import java.text.DecimalFormat;

import com.google.common.base.Preconditions;

/**
 * Позволяет связать сумму и валюту
 *
 * @author Roman Kuhta (kuhtich@yandex-team.ru)
 */
public class Money {
    private Money(BigDecimal value) {
        this(value, Currency.YND_FIXED);
    }

    private Money(BigDecimal value, Currency currency) {
        this.value = value;
        if (currency != null) {
            this.currency = currency;
        } else {
            this.currency = Currency.YND_FIXED;
        }
    }

    private final BigDecimal value;
    private final Currency currency;

    public static Money valueOf(Float value) {
        return new Money(toBigDecimal(value));
    }

    public static Money valueOf(Double value) {
        return new Money(toBigDecimal(value));
    }

    public static Money valueOf(Long value) {
        return new Money(toBigDecimal(value));
    }

    public static Money valueOf(Integer value) {
        return valueOf((long) value);
    }

    public static Money valueOf(BigDecimal value) {
        return new Money(value);
    }

    public static Money valueOf(Float value, Currency currency) {
        return new Money(toBigDecimal(value), currency);
    }

    public static Money valueOf(Double value, Currency currency) {
        return new Money(toBigDecimal(value), currency);
    }

    public static Money valueOf(BigDecimal value, Currency currency) {
        return new Money(value, currency);
    }

    public static Money valueOf(Long value, Currency currency) {
        return new Money(toBigDecimal(value), currency);
    }

    public static Money valueOf(Integer value, Currency currency) {
        return new Money(toBigDecimal(value), currency);
    }

    public static Money valueOf(Float value, String currency) {
        if (currency == null)
            return valueOf(value);
        return valueOf(value, Currency.valueOf(currency.toUpperCase()));
    }

    public static Money valueOf(Double value, String currency) {
        if (currency == null)
            return valueOf(value);
        return valueOf(value, Currency.valueOf(currency.toUpperCase()));
    }

    public static Money valueOf(BigDecimal value, String currency) {
        if (currency == null)
            return valueOf(value);
        return valueOf(value, Currency.valueOf(currency.toUpperCase()));
    }

    public static Money valueOf(Long value, String currency) {
        if (currency == null)
            return valueOf(value);
        return valueOf(value, Currency.valueOf(currency.toUpperCase()));
    }

    public static Money valueOf(Integer value, String currency) {
        if (currency == null)
            return valueOf(value);
        return valueOf(value, Currency.valueOf(currency.toUpperCase()));
    }

    public Float floatValue() {
        return value.floatValue();
    }

    public Double doubleValue() {
        return value.doubleValue();
    }

    public Long longValue() {
        return value.longValue();
    }

    public Long longValueForSum() {
        //По мотивам DIRECT-50363
        Long longValue = value.longValue();
        if (longValue < 0) {
            longValue -= 1;
        }
        return longValue;
    }


    public BigDecimal bigDecimalValue() {
        return value;
    }

    public String stringValue(DecimalFormat format) {
        return format.format(value);
    }

    public String stringValue(MoneyFormat format) {
        return stringValue(format.get());
    }

    public Money negate() {
        return new Money(this.value.negate(), currency);
    }

    public Money add(Money money) {
        Preconditions.checkArgument(this.currency == money.currency, "Должна быть одинаковая валюта");
        return add(money.value);
    }

    public Money add(Float value) {
        return add(toBigDecimal(value));
    }

    public Money add(BigDecimal value) {
        return new Money(this.value.add(value, MathContext.DECIMAL64), currency);
    }

    public Money add(Long value) {
        return add(toBigDecimal(value));
    }

    public Money add(Integer value) {
        return add(toBigDecimal(value));
    }

    public Money subtract(Money money) {
        Preconditions.checkArgument(this.currency == money.currency, "Должна быть одинаковая валюта");
        return subtract(money.value);
    }

    public Money subtract(Float value) {
        return subtract(toBigDecimal(value));
    }

    public Money subtract(BigDecimal value) {
        return new Money(this.value.subtract(value, MathContext.DECIMAL64), currency);
    }

    public Money subtract(Long value) {
        return subtract(toBigDecimal(value));
    }

    public Money subtract(Integer value) {
        return subtract(toBigDecimal(value));
    }

    public Money divide(Money money) {
        Preconditions.checkArgument(this.currency == money.currency, "Должна быть одинаковая валюта");
        return divide(money.value);
    }

    public Money divide(Float value) {
        return divide(toBigDecimal(value));
    }

    public Money divide(Double value) {
        return divide(toBigDecimal(value));
    }

    public Money divide(BigDecimal value) {
        return new Money(this.value.divide(value, MathContext.DECIMAL64), currency);
    }

    public Money divide(Long value) {
        return divide(toBigDecimal(value));
    }

    public Money divide(Integer value) {
        return divide(toBigDecimal(value));
    }

    public Money multiply(Money money) {
        Preconditions.checkArgument(this.currency == money.currency, "Должна быть одинаковая валюта");
        return multiply(money.value);
    }

    public Money multiply(Double value) {
        return multiply(toBigDecimal(value));
    }

    public Money multiply(Float value) {
        return multiply(toBigDecimal(value));
    }

    public Money multiply(BigDecimal value) {
        return new Money(this.value.multiply(value, MathContext.DECIMAL64), currency);
    }

    public Money multiply(Long value) {
        return multiply(toBigDecimal(value));
    }

    public Money multiply(Integer value) {
        return multiply(toBigDecimal(value));
    }

    /**
     * Для версии API5 используются ставки/цены в формате long - старые ставки float, умноженные на миллион (новый подход Директа работы с округлением)
     *
     * @return значение ставки/цены умноженное на миллион
     */
    public Money bidLong() {
        return multiply(1000000L);
    }

    /**
     * Для версии 4Live используются ставки/цены в формате float
     *
     * @return значение ставки/цены в старом формате с плавающей точкой
     */
    public Money bidShort() {
        return divide(new BigDecimal(1000000L));
    }

    /**
     * Окрукление до дефолтного для валюты количества знаков после запятой
     *
     * @param roundingMode Способ округления
     * @return Возвращает округленное значение
     */
    public Money setScale(RoundingMode roundingMode) {
        return setScale(MoneyCurrency.get(currency).getDefaultPrecision(), roundingMode);
    }

    public Money setScale(int precision, RoundingMode roundingMode) {
        return new Money(value.setScale(precision, roundingMode), currency);
    }

    public Money getNext() {
        return this.add(MoneyCurrency.get(currency).getStepPrice().bigDecimalValue());
    }

    public Money getPrevious() {
        return this.subtract(MoneyCurrency.get(currency).getStepPrice().bigDecimalValue());
    }

    public Currency getCurrency() {
        return currency;
    }

    /**
     * Конвертирует деньги в указанную валюту
     *
     * @param currency Желаемая валюта
     * @return Возвращает сконвертированное значение
     */
    public Money convert(Currency currency) {
        return convert(currency, null);
    }

    public Money convert(Currency currency, Double oldCurrencyRate) {
        //Если валюта для конвертации совпадает с вылютой денег, тогда просто возвращаем копию
        if (this.currency == currency) {
            return new Money(this.value, currency);
        }

        Money cashInYE = new Money(this.value, Currency.YND_FIXED);

        //Если текущая валюта не у.е., тогда конвертируем в у.е.
        //Сумма в у. е. = Сумма в валюте × (1 + Ставка НДС) / Курс валюты
        if (this.currency != Currency.YND_FIXED) {
            //если курс валюты учитывает НДС
            if (MoneyCurrency.get(this.currency).isVATIncludedInRate()) {
                cashInYE = cashInYE.multiply(1 + MoneyCurrency.get(this.currency).getVatRate());
            }
            if (oldCurrencyRate == null) {
                cashInYE = cashInYE.divide(MoneyCurrency.get(this.currency).getRate());
            } else {
                cashInYE = cashInYE.divide(oldCurrencyRate);
            }
        }

        if (currency == Currency.YND_FIXED) {
            return cashInYE;
        }

        MoneyCurrency toCurrency = MoneyCurrency.get(currency);

        Money cashInNewCurrency = new Money(cashInYE.bigDecimalValue(), currency);

        //Конвертируем у.е. в валюту
        //Сумма в валюте = Сумма в у. е. × Курс валюты / (1 + Ставка НДС)
        cashInNewCurrency = cashInNewCurrency.multiply(toCurrency.getRate());
        //если курс валюты учитывает НДС
        if (toCurrency.isVATIncludedInRate()) {
            cashInNewCurrency = cashInNewCurrency.divide(1 + toCurrency.getVatRate());
        }

        return cashInNewCurrency;
    }

    public Money withDiscount(Double discount) {
        return valueOf(this.divide(1 - discount).bigDecimalValue(), currency);
    }

    public Money withoutDiscount(Double discount) {
        return valueOf(this.multiply(1 - discount).bigDecimalValue(), currency);
    }

    /**
     * Уменьшает сумму на % НДС для заданной валюты
     *
     * @return результирующее значение
     */
    public Money addVAT() {
        return addVAT(currency);
    }

    /**
     * Уменьшает сумму на % НДС со значением для другой валюты(т.е. страны)
     *
     * @param currency валюта страны, для которой требуется посчитать НДС
     * @return результирующее значение
     */
    public Money addVAT(Currency currency) {
        return this.multiply(1 + MoneyCurrency.get(currency).getVatRate());
    }

    /**
     * Увеличивает сумму на % НДС для заданной валюты
     *
     * @return результирующее значение
     */
    public Money subtractVAT() {
        return subtractVAT(currency);
    }

    /**
     * Увеличивает сумму на % НДС со значением для другой валюты(т.е. страны)
     *
     * @param currency валюта страны, для которой требуется посчитать НДС
     * @return результирующее значение
     */
    public Money subtractVAT(Currency currency) {
        return this.divide(1 + MoneyCurrency.get(currency).getVatRate());
    }

    /**
     * Деление суммы в соответствии с переданным кастомным значением НДС на 1+НДС
     */
    public Money divideValueUsingCustomVatRate(double customVatRate) {
        return this.divide(1 + customVatRate);
    }

    @Override
    public String toString() {
        return value.toString() + " " + currency.value();
    }

    public static BigDecimal toBigDecimal(Float value) {
        if (value == null) {
            throw new DirectUtilsException("Value cannot be null");
        }
        return new BigDecimal(Float.toString(value));
    }

    public static BigDecimal toBigDecimal(Double value) {
        if (value == null) {
            throw new DirectUtilsException("Value cannot be null");
        }
        return new BigDecimal(Double.toString(value));
    }

    public static BigDecimal toBigDecimal(Long value) {
        if (value == null) {
            throw new DirectUtilsException("Value cannot be null");
        }
        return BigDecimal.valueOf(value);
    }

    public static BigDecimal toBigDecimal(Integer value) {
        if (value == null) {
            throw new DirectUtilsException("Value cannot be null");
        }
        return BigDecimal.valueOf(value);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof Money) {
            Money money = (Money) obj;
            return this.value.compareTo(money.value) == 0 && currency == money.currency;
        } else {
            return false;
        }
    }
}
