package ru.yandex.autotests.direct.utils.rules;

import org.apache.commons.lang3.exception.ExceptionUtils;
import org.hamcrest.Matcher;
import org.hamcrest.StringDescription;
import org.junit.internal.AssumptionViolatedException;
import org.junit.rules.TestRule;
import org.junit.runners.model.Statement;
import ru.yandex.autotests.irt.testutils.allure.AllureUtils;
import ru.yandex.autotests.irt.testutils.allure.AssumptionException;
import ru.yandex.qatools.allure.annotations.Step;

import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.CoreMatchers.instanceOf;
import static org.junit.Assert.fail;
import static org.junit.internal.matchers.ThrowableCauseMatcher.hasCause;
import static org.junit.internal.matchers.ThrowableMessageMatcher.hasMessage;

/**
 * Добавлен из-за невозможности расширить стандартный класс {@link org.junit.rules.ExpectedException}
 */
public class ExpectedExceptionRule implements TestRule {

    /**
     * @return a Rule that expects no exception to be thrown (identical to
     * behavior without this Rule)
     */
    public static ExpectedExceptionRule none() {
        return new ExpectedExceptionRule();
    }

    private final ExpectedExceptionMatcherBuilder fMatcherBuilder = new ExpectedExceptionMatcherBuilder();

    private boolean handleAssumptionViolatedExceptions = false;

    private boolean handleAssertionErrors = false;

    private ExpectedExceptionRule() {
    }

    public ExpectedExceptionRule handleAssertionErrors() {
        handleAssertionErrors = true;
        return this;
    }

    public ExpectedExceptionRule handleAssumptionViolatedExceptions() {
        handleAssumptionViolatedExceptions = true;
        return this;
    }

    public Statement apply(Statement base,
                           org.junit.runner.Description description) {
        return new ExpectedExceptionStatement(base);
    }

    /**
     * Adds {@code matcher} to the list of requirements for any thrown
     * exception.
     */
    public ExpectedExceptionRule expect(Matcher<?> matcher) {
        fMatcherBuilder.add(matcher);
        return this;
    }

    /**
     * Adds to the list of requirements for any thrown exception that it should
     * be an instance of {@code type}
     */
    public ExpectedExceptionRule expect(Class<? extends Throwable> type) {
        return expect(instanceOf(type));
    }

    /**
     * Adds to the list of requirements for any thrown exception that it should
     * <em>contain</em> string {@code substring}
     */
    public ExpectedExceptionRule expectMessage(String substring) {
        return expectMessage(containsString(substring));
    }

    /**
     * Adds {@code matcher} to the list of requirements for the message returned
     * from any thrown exception.
     */
    public ExpectedExceptionRule expectMessage(Matcher<String> matcher) {
        return expect(hasMessage(matcher));
    }

    /**
     * Adds {@code matcher} to the list of requirements for the cause of
     * any thrown exception.
     */
    public ExpectedExceptionRule expectCause(Matcher<? extends Throwable> expectedCause) {
        return expect(hasCause(expectedCause));
    }

    private class ExpectedExceptionStatement extends Statement {
        private final Statement fNext;

        public ExpectedExceptionStatement(Statement base) {
            fNext = base;
        }

        @Override
        public void evaluate() throws Throwable {
            try {
                fNext.evaluate();
                if (fMatcherBuilder.expectsThrowable()) {
                    failDueToMissingException();
                }
            } catch (AssumptionViolatedException | AssumptionException e) {
                optionallyHandleException(e, handleAssumptionViolatedExceptions);
            } catch (AssertionError e) {
                optionallyHandleException(e, handleAssertionErrors);
            } catch (Exception e) {
                handleException(e);
            }
        }
    }

    @Step("Сервер должен был вернуть ошибку")
    private void failDueToMissingException() throws AssertionError {
        String expectation = StringDescription.toString(fMatcherBuilder.build());
        fail("Сервер должен был ответить ошибкой " + expectation);
    }

    private void optionallyHandleException(Throwable e, boolean handleException)
            throws Throwable {
        if (handleException) {
            handleException(e);
        } else {
            throw e;
        }
    }

    @Step("Проверяем, что параметры ошибки от сервера верные")
    private void handleException(Throwable e) throws Throwable {
        AllureUtils.addTextAttachment("Ожидаемая ошибка", fMatcherBuilder.build().toString());
        AllureUtils.addTextAttachment("Полученная ошибка", e.toString());
        AllureUtils.addTextAttachment("Stacktrace", ExceptionUtils.getStackTrace(e));

        if (fMatcherBuilder.expectsThrowable()) {
            if (!fMatcherBuilder.build().matches(e)) {
                throw new AssertionError("Неверная ошибка от сервера, ожидалась " +
                        StringDescription.toString(fMatcherBuilder.build()), e);
            }
        } else {
            throw e;
        }
    }
}