package ru.yandex.autotests.direct.utils.strategy.data;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.collections.CollectionUtils;

import ru.yandex.autotests.direct.utils.beans.MongoBeanLoader;
import ru.yandex.autotests.direct.utils.campaigns.CampaignTypeEnum;
import ru.yandex.autotests.direct.utils.strategy.objects.StrategyInfoWeb;

import static freemarker.template.utility.Collections12.singletonList;
import static java.util.Arrays.asList;
import static ru.yandex.autotests.direct.utils.campaigns.CampaignTypeEnum.CPM_BANNER;
import static ru.yandex.autotests.direct.utils.campaigns.CampaignTypeEnum.DMO;
import static ru.yandex.autotests.direct.utils.campaigns.CampaignTypeEnum.DTO;
import static ru.yandex.autotests.direct.utils.campaigns.CampaignTypeEnum.MOBILE;
import static ru.yandex.autotests.direct.utils.campaigns.CampaignTypeEnum.TEXT;

/**
 * Перечисление всех доступных стратегий в директе
 */
public enum Strategies {
    /**
     * Cредняя цена конверсии
     */
    AVERAGE_CPA_OPTIMIZATION_DEFAULT(true, asList(TEXT), StrategyGroup.CPA),

    /**
     * Независимое управление:
     * Cредняя цена конверсии
     * Отключены показы
     */
    AVERAGE_CPA_OPTIMIZATION_SHOWS_DISABLED(true, asList(TEXT, DTO), StrategyGroup.CPA),

    /**
     * Cредняя цена установки приложения
     */
    AVERAGE_CPI_OPTIMIZATION_DEFAULT(true, singletonList(MOBILE), StrategyGroup.CPI),

    /**
     * Cредняя цена клика
     */
    AVERAGE_PRICE_DEFAULT(true, asList(TEXT, MOBILE), StrategyGroup.DEFAULT),


    /**
     * Независимое управление:
     * Cредняя цена клика
     * Отключены показы
     */
    AVERAGE_PRICE_SHOWS_DISABLED(true, asList(TEXT, MOBILE, DTO), StrategyGroup.SEARCH),

    /**
     * ДМО:
     * Оптимизация количества конверсий на всю кампанию
     */
    CPA_OPTIMIZATION_CAMP(true, singletonList(DMO), StrategyGroup.CPA, StrategyGroup.CONTEXT),

    /**
     * ДМО:
     * Оптимизация количества конверсий на фильтр
     */
    CPA_OPTIMIZATION_FILTER(true, singletonList(DMO), StrategyGroup.CPA, StrategyGroup.CONTEXT),

    /**
     * ДМО:
     * Оптимизация количества кликов на всю кампанию
     */
    CPC_OPTIMIZATION_CAMP(true, singletonList(DMO), StrategyGroup.CPC, StrategyGroup.CONTEXT),

    /**
     * ДМО:
     * Оптимизация количества кликов на фильтр
     */
    CPC_OPTIMIZATION_FILTER(true, singletonList(DMO), StrategyGroup.CPC, StrategyGroup.CONTEXT),

    /**
     * Наивысшая доступная позиция
     */
    HIGHEST_POSITION_DEFAULT(false, asList(TEXT, MOBILE), StrategyGroup.DEFAULT),

    /**
     * Независимое управление:
     * Наивысшая доступная позиция
     * Максимально доступный охват
     */
    HIGHEST_POSITION_MAX_COVERAGE(false, asList(TEXT, MOBILE), StrategyGroup.CONTEXT),

    /**
     * Независимое управление:
     * Наивысшая доступная позиция
     * Отключены показы
     */
    HIGHEST_POSITION_SHOWS_DISABLED(false, asList(TEXT, MOBILE, DTO), StrategyGroup.SEARCH),

    /**
     * Средняя рентабельность инвестиций
     */
    ROI_OPTIMIZATION_DEFAULT(true, asList(TEXT), StrategyGroup.ROI),

    /**
     * Независимое управление:
     * Средняя рентабельность инвестиций
     * Отключены показы
     */
    ROI_OPTIMIZATION_SHOWS_DISABLED(true, asList(TEXT, DTO), StrategyGroup.ROI),

    /**
     * Средняя рентабельность инвестиций для ДМО. Название стратегии autobudget_roi, на поиске показы остановлены
     */
    ROI_OPTIMIZATION_DMO(true, asList(DMO), StrategyGroup.ROI),

    /**
     * Независимое управление:
     * Отключены показы
     * Cредняя цена конверсии
     */
    SHOWS_DISABLED_AVERAGE_CPA_OPTIMIZATION(true, singletonList(TEXT), StrategyGroup.CPA),

    /**
     * Независимое управление:
     * Отключены показы
     * Средняя цена установки приложения
     */
    SHOWS_DISABLED_AVERAGE_CPI_OPTIMIZATION(true, singletonList(MOBILE), StrategyGroup.CPI),

    /**
     * Независимое управление:
     * Отключены показы
     * Средняя цена клика
     */
    SHOWS_DISABLED_AVERAGE_PRICE(true, asList(TEXT, MOBILE), StrategyGroup.CONTEXT),

    /**
     * Независимое управление:
     * Отключены показы
     * Максимально доступный охват
     */
    SHOWS_DISABLED_MAX_COVERADGE(false, asList(TEXT, MOBILE), StrategyGroup.CONTEXT),

    /**
     * Независимое управление:
     * Отключены показы
     * Средняя цена инвестиций {@link Strategies#ROI_OPTIMIZATION_DEFAULT}
     */
    SHOWS_DISABLED_ROI_OPTIMIZATION(true, asList(TEXT), StrategyGroup.ROI),

    /**
     * Независимое управление:
     * Отключены показы
     * Недельный пакет кликов с получаение максимального количества кликов
     */
    SHOWS_DISABLED_WEEKLY_BUDGET_MAX_CLICKS(true, asList(TEXT, MOBILE), StrategyGroup.CONTEXT),

    /**
     * Независимое управление:
     * Отключены показы
     * Недельный пакет кликов с получаение максимальной конверсии по целям {@link Strategies#WEEKLY_BUDGET_MAX_CONVERSION_DEFAULT}
     */
    SHOWS_DISABLED_WEEKLY_BUDGET_MAX_CONVERSION(true, singletonList(TEXT), StrategyGroup.CPA),

    /**
     * Независимое управление:
     * Отключены показы
     * Недельный пакет кликов с получаение максимального числа установок приложения
     */
    SHOWS_DISABLED_WEEKLY_BUDGET_MAX_INSTALLATIONS(true, asList(MOBILE), StrategyGroup.CPI),

    /**
     * Независимое управление:
     * Отключены показы
     * Недельный пакет кликов с ограничением средней цены клика
     */
    SHOWS_DISABLED_WEEKLY_PACKET_OF_CLICKS_AVERAGE_CLICK_PRICE(true, asList(TEXT, MOBILE),
            StrategyGroup.CONTEXT),

    /**
     * Независимое управление:
     * Отключены показы
     * Недельный пакет кликов с ограничением максимальной цены клика
     */
    SHOWS_DISABLED_WEEKLY_PACKET_OF_CLICKS_MAX_CLICK_PRICE(true, asList(TEXT, MOBILE), StrategyGroup.CONTEXT),

    /**
     * Недельный бюджет с получением максимального числа кликов
     */
    WEEKLY_BUDGET_MAX_CLICKS_DEFAULT(true, asList(TEXT, MOBILE), StrategyGroup.DEFAULT),

    /**
     * Независимое управление:
     * Недельный бюджет с получением максимального числа кликов
     * Отключены показы
     */
    WEEKLY_BUDGET_MAX_CLICKS_SHOWS_DISABLED(true, asList(TEXT, MOBILE, DTO), StrategyGroup.DEFAULT),

    /**
     * Недельный бюджет с получением максимальной конверсии по цели
     */
    WEEKLY_BUDGET_MAX_CONVERSION_DEFAULT(true, asList(TEXT), StrategyGroup.CPA),

    /**
     * Независимое управление:
     * Недельный бюджет с получением максимальной конверсии по цели
     * Отключены показы
     */
    WEEKLY_BUDGET_MAX_CONVERSION_SHOWS_DISABLED(true, asList(TEXT, DTO), StrategyGroup.CPA),

    /**
     * Недельный бюджет с получением максимального количества установок приложения
     */
    WEEKLY_BUDGET_MAX_INSTALLATIONS_DEFAULT(true, asList(MOBILE), StrategyGroup.CPI),

    /**
     * Недельный пакет кликов с ограничением средней цены клика
     */
    WEEKLY_PACKET_OF_CLICKS_AVERAGE_CLICK_PRICE_DEFAULT(true, asList(TEXT, MOBILE), StrategyGroup.SEARCH),

    /**
     * Независимое управление:
     * Недельный пакет кликов с ограничением средней цены клика
     * Отключены показы
     */
    WEEKLY_PACKET_OF_CLICKS_AVERAGE_CLICK_PRICE_SHOWS_DISABLED(true, asList(TEXT, MOBILE, DTO),
            StrategyGroup.SEARCH),

    /**
     * Независимое управление:
     * Недельный пакет кликов с ограничением максимального числа кликов
     * Отключены показы
     */
    WEEKLY_PACKET_OF_CLICKS_MAX_CLICK_PRICE_SHOWS_DISABLED(true, asList(TEXT, MOBILE, DTO),
            StrategyGroup.SEARCH),


    /**
     * Недельный пакет кликов с ограничением максимальной цены клика
     */
    WEEKLY_PACKET_OF_CLICKS_MAX_CLICK_PRICE_DEFAULT(true, asList(TEXT, MOBILE), StrategyGroup.SEARCH),

    /**
     * Ручное управление ставками CPM кампания
     */
    CPM_MANUAL_CONTROL(false, asList(CPM_BANNER), StrategyGroup.SEARCH, StrategyGroup.CONTEXT);


    private static final String TEMPLATE_PREFIX = "strategy";
    private static final String TEMPLATE_COLLECTION_NAME = "DirectWebTemplates45";
    private String templateName;
    private boolean autoBudget;
    private List<CampaignTypeEnum> campaignTypes;
    private List<StrategyGroup> strategyGroups;

    Strategies(boolean autoBudget, List<CampaignTypeEnum> campaignTypes, StrategyGroup... strategyGroups) {
        this.strategyGroups = asList(strategyGroups);
        this.autoBudget = autoBudget;
        this.campaignTypes = campaignTypes;
    }

    public static class StrategiesFilters {
        public static Predicate<Strategies> byIsAutoOption(boolean auto) {
            return strategy -> strategy.isAutoBudget() == auto
                    && !strategy.strategyGroups.contains(StrategyGroup.CPA)
                    && !strategy.strategyGroups.contains(StrategyGroup.ROI);
        }

        public static Predicate<Strategies> byCampaignTypes(CampaignTypeEnum... campaignTypes) {
            return strategy -> strategy.getCampaignTypes().containsAll(asList(campaignTypes));
        }
    }

    public static List<Object> getStrategyListByPredicate(Predicate<Strategies> filter) {
        return Stream.of(Strategies.values())
                .filter(filter)
                .collect(Collectors.toList());
    }

    public static List<Strategies> getStrategyListExcludeGroup(CampaignTypeEnum campaignType,
            Boolean auto,
            StrategyGroup group,
            StrategyGroup... excludeGroup)
    {
        return getStrategyList(campaignType, auto, false, group).stream()
                .filter(t -> excludeGroup == null ||
                        CollectionUtils.intersection(t.strategyGroups, asList(excludeGroup)).isEmpty())
                .collect(Collectors.toList());
    }

    public static List<Strategies> getStrategyList(CampaignTypeEnum campaignType, Boolean auto,
            StrategyGroup... groups)
    {
        return getStrategyList(campaignType, auto, false, groups);
    }

    public static List<Strategies> getStrategyList(CampaignTypeEnum campaignType, Boolean auto, boolean isOnlyGroups,
            StrategyGroup... groups)
    {
        return Stream.of(Strategies.values())
                .filter(t -> campaignType == null || t.campaignTypes.contains(campaignType))
                .filter(t -> auto == null || t.autoBudget == auto)
                .filter(t -> groups == null || groups.length == 0 || groups[0] == null || (!isOnlyGroups ?
                        !CollectionUtils.intersection(t.getStrategyGroups(), asList(groups)).isEmpty() :
                        CollectionUtils.isEqualCollection(t.getStrategyGroups(), asList(groups))))
                .collect(Collectors.toList());
    }

    public static List<Strategies> getStrategyList(CampaignTypeEnum campaignType) {
        return getStrategyList(campaignType, null);
    }

    public static List<Strategies> getStrategyListExcludeGroup(CampaignTypeEnum campaignType,
            StrategyGroup... excludeGroup)
    {
        return getStrategyListExcludeGroup(campaignType, null, null, excludeGroup);
    }

    public static Strategies getRandomStrategyByIsAutoOption(boolean auto) {
        return getAnotherRandomStrategyByIsAutoOption(null, auto);
    }

    public static Strategies getAnotherRandomStrategyByIsAutoOption(Strategies oldStrategy, boolean auto) {
        List<Object> list = getStrategyListByPredicate(StrategiesFilters.byIsAutoOption(auto));
        Collections.shuffle(list);
        while (list.get(0).equals(oldStrategy)) {
            Collections.shuffle(list);
        }
        return (Strategies) list.get(0);
    }

    public static List<Strategies> getStrategyListByStrategyGroup(StrategyGroup group) {
        List<Strategies> strategies = new ArrayList<Strategies>();
        for (Strategies strategy : Strategies.values()) {
            if (strategy.strategyGroups.contains(group)) {
                strategies.add(strategy);
            }
        }
        return strategies;
    }

    public boolean isAutoBudget() {
        return autoBudget;
    }

    public List<StrategyGroup> getStrategyGroups() {
        return strategyGroups;
    }

    public StrategyInfoWeb getStrategyBean(String tmplSuffix) {
        MongoBeanLoader<StrategyInfoWeb> loader =
                new MongoBeanLoader<>(StrategyInfoWeb.class, TEMPLATE_COLLECTION_NAME);
        StrategyInfoWeb strategyInfoWeb = loader.getBean(this.getTemplateName() + tmplSuffix);
        strategyInfoWeb.setAutoBudget(this.isAutoBudget());
        return strategyInfoWeb;
    }

    public void setStrategyBean(StrategyInfoWeb bean, String tmplSuffix) {
        MongoBeanLoader<StrategyInfoWeb> loader =
                new MongoBeanLoader<>(StrategyInfoWeb.class, TEMPLATE_COLLECTION_NAME);
        loader.saveBean(bean, this.getTemplateName() + tmplSuffix);
    }

    public StrategyInfoWeb getStrategyBean() {
        return getStrategyBean("");
    }


    public String getTemplateName() {
        String tmplName = "";
        //convert TEMPLATE_NAME to templateName;
        for (String word : this.name().split("_")) {
            tmplName += word.substring(0, 1);
            tmplName += word.substring(1).toLowerCase();
        }
        tmplName = TEMPLATE_PREFIX + tmplName;
        return tmplName;
    }

    public List<CampaignTypeEnum> getCampaignTypes() {
        return campaignTypes;
    }
}
