package ru.yandex.autotests.direct.web.api.steps;

import java.io.IOException;

import com.google.gson.annotations.SerializedName;
import okhttp3.Cookie;
import org.apache.http.util.Asserts;
import retrofit2.Call;
import retrofit2.http.Body;
import retrofit2.http.Field;
import retrofit2.http.FormUrlEncoded;
import retrofit2.http.POST;
import retrofit2.http.Query;

import ru.yandex.autotests.direct.utils.BaseSteps;
import ru.yandex.autotests.direct.utils.model.MongoUser;
import ru.yandex.autotests.direct.web.api.DirectWebApiProperties;
import ru.yandex.autotests.direct.web.api.SimpleCookieJar;
import ru.yandex.autotests.direct.web.api.client.ApiClient;
import ru.yandex.autotests.direct.web.api.core.DirectWebApiError;
import ru.yandex.autotests.direct.web.api.core.DirectWebApiStepsContext;
import ru.yandex.qatools.allure.annotations.Step;

public class AuthSteps extends BaseSteps<DirectWebApiStepsContext> {
    private static final String COOKIE_YANDEX_LOGIN = "yandex_login";
    private static final String COOKIE_SESSION_ID = "Session_id";

    /**
     * Метод меняет информацию в сквозном на все степы контексте {@link this#getContext()}
     */
    @Step("Авторизация в паспорте и получение csrf токена")
    void authorise(MongoUser user) {
        authWithCookiesJar(getContext().getCookieJar(), user);
        obtainCsrfToken(extractPassportUid());
    }

    @Step("Получение csrf-токена в intapi (passport uid = {0})")
    public void obtainCsrfToken(String passportUid) {
        String token = getCsrfToken(passportUid);
        Asserts.check(token != null, "Cannot obtain csrf token from direct intapi");
        setupContextCsrfToken(token);
    }


    private void authWithCookiesJar(SimpleCookieJar cookieJar, MongoUser user) {
        ApiClient client = ApiClient.Builder.create()
                .baseUrl("https://passport.yandex.ru")
                .verbose(DirectWebApiProperties.getInstance().isVerbose())
                .cookieJar(cookieJar)
                .build();
        PassportApi api = client.createService(PassportApi.class);
        try {
            api.login("auth", user.getLogin(), user.getPassword()).execute();

        } catch (IOException e) {
            throw new DirectWebApiError("Не удалось авторизоваться в паспорте", e);
        }
        assertAuthenticationComplete(cookieJar, user.getLogin());
        setupContextAuthUser(user);
    }

    private void setupContextAuthUser(MongoUser user) {
        getContext().withCurrentAuthUser(user);
    }

    private String extractPassportUid() {
        Cookie sessionIdCookie = getContext().getCookieJar().getCookie(COOKIE_SESSION_ID);
        if (sessionIdCookie == null) {
            throw new DirectWebApiError(
                    "Can not extract passport UID from session-id cookie, cookie does not exists");
        }

        try {
            return extractPassportUidFromSessionId(sessionIdCookie.value());
        } catch (Exception e) {
            throw new DirectWebApiError(
                    "Error in extracting passport UID from session-id cookie: " + sessionIdCookie, e);
        }
    }

    private String extractPassportUidFromSessionId(String sessionId) {
        String[] parts = sessionId.split("\\|");
        String partWithUid = parts[1];
        return partWithUid.substring(0, partWithUid.indexOf("."));
    }

    private void assertAuthenticationComplete(SimpleCookieJar cookieJar, String login) {
        Cookie cookie = cookieJar.getCookie(COOKIE_YANDEX_LOGIN);
        Asserts.check(cookie != null,
                "Can not log in to passport by \"%s\" (cookie \"%s\" is absent). Check credentials!",
                login, COOKIE_YANDEX_LOGIN);
        Asserts.check(login.equals(cookie.value()),
                "Can not log in to passport (exepcted: cookie \"%s\" is set to \"%s\"; but: it is set to \"%s\").",
                COOKIE_YANDEX_LOGIN,
                login,
                cookie.value());
    }

    private String getCsrfToken(String passportUid) {
        ApiClient client = ApiClient.Builder.create()
                .baseUrl(getContext().getIntapiBaseUrl())
                .verbose(DirectWebApiProperties.getInstance().isVerbose())
                .build();
        CSRFApi api = client.createService(CSRFApi.class);
        try {
            CSRFTokenRequest request = new CSRFTokenRequest();
            request.params.uid = passportUid;
            CSRFTokenResponse response = api.getToken(request).execute().body();
            return response.result.csrfToken;
        } catch (IOException e) {
            throw new DirectWebApiError("Не удалось авторизоваться в паспорте", e);
        }
    }

    private void setupContextCsrfToken(String token) {
        getContext().withCsrfToken(token);
    }

    private interface PassportApi {
        @POST("passport")
        @FormUrlEncoded
        Call<Void> login(@Query("mode") String mode, @Field("login") String login, @Field("passwd") String password);
    }

    private interface CSRFApi {
        @POST("jsonrpc/CSRF")
        Call<CSRFTokenResponse> getToken(@Body CSRFTokenRequest request);
    }

    private class CSRFTokenRequest {
        private String jsonrpc = "2.0";
        private String id = "1";
        private String method = "get_csrf_token";
        private Params params = new Params();

        private class Params {
            @SerializedName("UID")
            private String uid;
        }
    }

    private class CSRFTokenResponse {
        private Result result;

        private class Result {
            @SerializedName("csrf_token")
            private String csrfToken;
        }
    }
}
