package ru.yandex.autotests.direct.web.api.steps;

import java.math.BigDecimal;
import java.util.List;
import java.util.Optional;

import com.apollographql.apollo.ApolloCall;
import com.apollographql.apollo.ApolloClient;
import com.apollographql.apollo.ApolloQueryCall;
import com.apollographql.apollo.CustomTypeAdapter;
import com.apollographql.apollo.api.Response;
import com.apollographql.apollo.exception.ApolloException;
import org.jetbrains.annotations.NotNull;

import ru.yandex.autotests.direct.utils.BaseSteps;
import ru.yandex.autotests.direct.web.api.core.DirectWebApiError;
import ru.yandex.autotests.direct.web.api.core.DirectWebApiStepsContext;
import ru.yandex.autotests.direct.web.api.models.queries.SetAutoOverdraftLimitMutation;
import ru.yandex.autotests.direct.web.api.models.queries.SwitchOffLimitPaymentDataQuery;
import ru.yandex.autotests.direct.web.api.models.type.CustomType;
import ru.yandex.autotests.direct.web.api.models.type.GdSetAutoOverdraftParamsInput;
import ru.yandex.autotests.irt.testutils.allure.AllureUtils;
import ru.yandex.autotests.irt.testutils.json.JsonUtils;

public class AutoOverdraftSteps extends BaseSteps<DirectWebApiStepsContext> {

    public Response<Optional<SetAutoOverdraftLimitMutation.Data>> setAutoOverdraftLimit(String login, BigDecimal autoOverdraftLimit, long personId,
            String paymentMethodCode) {

        ApolloClient client = createApolloClient(login);

        GdSetAutoOverdraftParamsInput input = GdSetAutoOverdraftParamsInput.builder()
                .autoOverdraftLimit(autoOverdraftLimit)
                .personId(personId)
                .paymentMethodCode(paymentMethodCode)
                .build();

        SetAutoOverdraftLimitMutation mutation = SetAutoOverdraftLimitMutation.builder().input(input).build();
        ApolloCall<Optional<SetAutoOverdraftLimitMutation.Data>> call = client.mutate(mutation);
        return executeAndGetResponse(call);
    }

    @NotNull
    private <T> Response<T> executeAndGetResponse(ApolloCall<T> call)
    {
        Response<T> response;
        try {
            response = call.execute();
        } catch (ApolloException e) {
            throw new DirectWebApiError("Exception while processing web api request", e);
        }
        if (response.hasErrors()) {
            throw new DirectWebApiError("Exception while processing web api request, server responded with " + JsonUtils.toString(response));
        } else {
            AllureUtils.addJsonAttachment("Response", JsonUtils.toString(response));
        }
        return response;
    }

    public List<SwitchOffLimitPaymentDataQuery.AutoOverdraftPaymentOption> getPaymentOptions(String login) {

        ApolloClient client = createApolloClient(login);

        ApolloQueryCall<Optional<SwitchOffLimitPaymentDataQuery.Data>> query =
                client.query(SwitchOffLimitPaymentDataQuery.builder().login(login).build());
        Response<Optional<SwitchOffLimitPaymentDataQuery.Data>> response = executeAndGetResponse(query);

        return response.data()
                .map(SwitchOffLimitPaymentDataQuery.Data::client)
                .map(SwitchOffLimitPaymentDataQuery.Client::autoOverdraftPaymentOptions)
                .get();
    }

    private ApolloClient createApolloClient(String login) {
        return ApolloClient.builder()
                .serverUrl(getContext().getBaseUrl() + "/web-api/grid/api?ulogin=" + login)
                .okHttpClient(getContext().getApiClient().getOkClient())
                .addCustomTypeAdapter(CustomType.BIGDECIMAL, new CustomTypeAdapter<BigDecimal>() {
                    @Override public BigDecimal decode(final String value) {
                        return new BigDecimal(value);
                    }
                    @Override public String encode(final BigDecimal value) {
                        return value.toString();
                    }
                })
                .addCustomTypeAdapter(CustomType.LONG, new CustomTypeAdapter<Long>() {
                    @Override public Long decode(final String value) {
                        return new Long(value);
                    }
                    @Override public String encode(final Long value) {
                        return value.toString();
                    }
                })
                .build();
    }
}
