package ru.yandex.autotests.direct.web.pages.feeds.popup;

import org.openqa.selenium.WebElement;
import org.openqa.selenium.support.FindBy;

import ru.yandex.autotests.direct.utils.config.DirectTestRunProperties;
import ru.yandex.autotests.direct.web.data.feeds.FeedSourceEnum;
import ru.yandex.autotests.direct.web.objects.feeds.FeedWeb;
import ru.yandex.autotests.direct.web.util.WebElementsActions;
import ru.yandex.autotests.direct.web.util.testinterfaces.IWebFrom;
import ru.yandex.autotests.direct.web.webelements.bem.FileInputBEM;
import ru.yandex.autotests.direct.web.webelements.bem.RadioBEM;
import ru.yandex.qatools.htmlelements.annotations.Name;
import ru.yandex.qatools.htmlelements.element.Button;
import ru.yandex.qatools.htmlelements.element.CheckBox;
import ru.yandex.qatools.htmlelements.element.HtmlElement;
import ru.yandex.qatools.htmlelements.element.TextInput;
import ru.yandex.qatools.htmlelements.matchers.WebElementMatchers;

import static ch.lambdaj.Lambda.on;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.not;
import static ru.yandex.autotests.direct.utils.matchers.BeanEquals.beanEquals;
import static ru.yandex.autotests.direct.web.steps.CommonSteps.waitForElement;
import static ru.yandex.autotests.direct.web.util.beanutils.BeanFieldsSetter.inAccordanceWith;

/**
 * Created by aleran on 08.09.2015.
 */
@Name("Создание и изменение фида")
@FindBy(css = "form.b-feed-edit-popup")
public class EditFeedPopupBlock extends HtmlElement implements IWebFrom<FeedWeb> {

    @Name("Название фида")
    @FindBy(xpath = ".//span[contains(@data-bem, '\"name\":\"name\"')]//input")
    private TextInput feedNameTextInput;

    @Name("Тип источника")
    @FindBy(xpath = ".//span[contains(@data-bem, '\"name\":\"source\"')]//input")
    private RadioBEM sourceRadio;

    @Name("Ссылка на файл")
    @FindBy(xpath = ".//input[@name = 'feed_file']")
    private FileInputBEM feedFileInput;

    @Name("Текущий файл")
    @FindBy(css = ".b-feed-edit-popup__current-file")
    private WebElement currentFileName;

    @Name("Чекбокс необходимость авторизации")
    @FindBy(xpath = ".//input[@type = 'checkbox']")
    private CheckBox authorisationNeededCheckBox;

    @Name("Ссылка на файл")
    @FindBy(xpath = ".//span[contains(@data-bem, '\"name\":\"url\"')]//input")
    private TextInput urlTextInput;

    @Name("Логин для авторизации")
    @FindBy(xpath = ".//span[contains(@data-bem, '\"name\":\"login\"')]//input")
    private TextInput loginTextInput;

    @Name("Пароль для авторизации")
    @FindBy(xpath = ".//span[contains(@data-bem, '\"name\":\"password\"')]//input")
    private TextInput passwordTextInput;

    @Name("Пароль для авторизации")
    @FindBy(xpath = ".//button[contains(@data-bem, '\"name\":\"lockedSaveButton\"')]")
    private Button saveFeedButton;

    @Name("Лоадер сохранения фида")
    @FindBy(css = ".b-feed-edit-popup__spin")
    private WebElement saveFeedSpinLoader;

    public boolean isSaveButtonEnabled() {
        return saveFeedButton.isEnabled();
    }

    public void waitForSaveFeedSpinLoader() {
        waitForElement(saveFeedSpinLoader,
                (int) DirectTestRunProperties.getInstance().getDirectWebWebdriverImplicityWaitTimeoutSec(),
                not(WebElementMatchers.isDisplayed()));
    }

    public void clickSaveFeedButton() {
        saveFeedButton.click();
        waitForSaveFeedSpinLoader();
    }

    private void fillFileSource(FeedWeb feed) {
        feedFileInput.setFileToUpload(feed.getFeedFilePath());
    }

    private void fillUrlSource(FeedWeb feed) {
        urlTextInput.clear();
        urlTextInput.sendKeys(feed.getUrl());

        if (feed.getLogin() != null) {
            authorisationNeededCheckBox.select();
            loginTextInput.clear();
            loginTextInput.sendKeys(feed.getLogin());
            passwordTextInput.clear();
            passwordTextInput.sendKeys(feed.getPassword());
        }
    }

    @Override
    public void fillParameters(FeedWeb feed) {
        feedNameTextInput.clear();
        feedNameTextInput.sendKeys(feed.getName());

        boolean isExists = false;
        try {
            isExists = WebElementsActions.isExists(sourceRadio);
        } catch (Exception e) {
            //ignore
        }

        switch (feed.getSource()) {
            case FILE:
                if (isExists) {
                    sourceRadio.selectByValue(FeedSourceEnum.FILE.getValue());
                }

                fillFileSource(feed);
                break;
            case URL:
                if (isExists) {
                    sourceRadio.selectByValue(FeedSourceEnum.URL.getValue());
                }
                fillUrlSource(feed);
                break;
        }
    }

    @Override
    public void checkParameters(FeedWeb feedWeb) {
        assertThat("неверные параметры фида", getFormFieldsAccording(feedWeb), beanEquals(feedWeb));
    }

    @Override
    public FeedWeb getFormFieldsAccording(FeedWeb expectedBean) {
        FeedWeb retBeen = inAccordanceWith(expectedBean).forNew(FeedWeb.class)
                .set("name", feedNameTextInput, on(TextInput.class).getText())
                .set("url", urlTextInput, on(TextInput.class).getText())
                .set("login", loginTextInput, on(TextInput.class).getText())
                .set("password", passwordTextInput, on(TextInput.class).getText())
                .set("feedFileName", currentFileName, on(WebElement.class).getText())
                .getActualBean();
        if (retBeen.getUrl() != null && !retBeen.getUrl().isEmpty()) {
            retBeen.setSource(FeedSourceEnum.URL);
        } else {
            retBeen.setSource(FeedSourceEnum.FILE);
        }
        return retBeen;
    }
}
