package ru.yandex.autotests.direct.web.pages.group.dynamictext;

import java.util.List;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import org.openqa.selenium.By;
import org.openqa.selenium.WebElement;
import org.openqa.selenium.support.FindBy;

import ru.yandex.autotests.direct.utils.textresource.TextResourceFormatter;
import ru.yandex.autotests.direct.web.data.groupenums.TargetConditionsEnum;
import ru.yandex.autotests.direct.web.data.textresources.DynamicDataSource;
import ru.yandex.autotests.direct.web.objects.banners.DynamicTextBannerWeb;
import ru.yandex.autotests.direct.web.objects.groups.DynamicTextGroupWeb;
import ru.yandex.autotests.direct.web.objects.groups.TargetingCondition;
import ru.yandex.autotests.direct.web.objects.groups.feedfilter.DynamicFeedFilter;
import ru.yandex.autotests.direct.web.pages.group.GroupBaseBlock;
import ru.yandex.autotests.direct.web.pages.group.blocks.EditFilterBlockBase;
import ru.yandex.autotests.direct.web.util.DirectWebError;
import ru.yandex.autotests.direct.web.util.testinterfaces.Checkable;
import ru.yandex.autotests.direct.web.webelements.bem.RadioBEM;
import ru.yandex.autotests.direct.web.webelements.bem.TextAreaBEM;
import ru.yandex.autotests.irt.testutils.allure.AssumptionException;
import ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher;
import ru.yandex.qatools.htmlelements.annotations.Name;
import ru.yandex.qatools.htmlelements.element.Button;
import ru.yandex.qatools.htmlelements.matchers.WebElementMatchers;

import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.core.IsNot.not;
import static ru.yandex.autotests.direct.httpclient.util.beanmapper.BeanMapper.map;
import static ru.yandex.autotests.direct.utils.matchers.BeanEqualsAssert.assertThat;
import static ru.yandex.autotests.direct.web.steps.CommonSteps.waitForElement;
import static ru.yandex.autotests.direct.web.util.WebElementsActions.isExists;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;
import static ru.yandex.autotests.irt.testutils.beans.BeanFieldsSetter.inAccordanceWith;
import static ru.yandex.qatools.htmlelements.matchers.MatcherDecorators.should;
import static ru.yandex.qatools.htmlelements.matchers.MatcherDecorators.timeoutHasExpired;

/**
 * Created by ssdmitriev on 25.11.15.
 */
@FindBy(xpath = ".//div[contains(@class, 'b-edit-group ')]")
public class DynamicTextGroupBlock extends GroupBaseBlock implements Checkable<DynamicTextGroupWeb> {
    @Name("Радио 'Условия нацелевания'")
    @FindBy(xpath = ".//div[contains(@class, 'b-dynamic-conditions') and contains(@class, 'type-chooser')]//input")
    private RadioBEM targetingConditions;

    @Name("Радио 'Источник данных'")
    @FindBy(css = ".b-edit-group__switch-source input")
    private RadioBEM dataSource;

    @Name("Input 'Цена клика'")
    @FindBy(xpath = ".//span[contains(@data-bem, 'price')]//input")
    private TextAreaBEM clickPrice;

    @Name("Input 'Домен'")
    @FindBy(css = ".b-dynamic-group-domain__model-field input")
    private TextAreaBEM domain;

    @Name("Input 'Параметры в ссылке'")
    @FindBy(xpath = ".//div[contains(@class, 'b-dynamic-group-domain__url-params-input')]")
    private TextAreaBEM domainParams;

    @Name("Переключатель 'Параметры в ссылке'")
    @FindBy(xpath = ".//div[contains(@class, 'b-dynamic-group-domain__url-params-switcher')]/button")
    private Button domainParamsSwitcher;

    @Name("Список баннеров")
    private List<DynamicTextBannerBlock> dynamicTextBannerBlocks;

    @Name("Сообщение о корректности домена")
    @FindBy(xpath = ".//div[contains(@class, 'b-dynamic-group-domain__domain-message')]")
    private WebElement domainMessage;

    @Name("Кнопка 'задать' минус-слова на группу")
    @FindBy(xpath = ".//div[contains(@class, 'b-minus-words-control__switcher ')]/button")
    private Button minusWordsButton;

    @Name("Кнопка 'уточнить' регионы показа")
    @FindBy(xpath = ".//div[contains(@class, 'b-regions-selector')]//button")
    private Button regionsButton;

    @Name("Кнопка 'изменить' корректировки ставок")
    @FindBy(css = "div.b-adjustment-rates-popup button")
    private Button adjustmentRatesButton;

    @Name("Селект файлов фида")
    @FindBy(xpath = ".//button[contains(@class, 'b-group-feed__model-field')]")
    private Button feedsSelectButton;

    @Name("Попап выбора фида")
    @FindBy(xpath = "//div[@class='b-group-feed__popup']")
    private WebElement feedsPopup;

    @Name("'Источник данных' сохраненный")
    @FindBy(css = ".b-edit-group__switch-source")
    private WebElement staticDataSource;

    @Name("Название фида")
    @FindBy(xpath = "//div[contains(@class, 'b-group-feed__text')]")
    private WebElement feedFileName;

    private EditDynamicFilterBlock editDynamicFilterBlock;

    private List<DynamicTextBannerBlock> getDynamicTextBannerBlocks() {
        return dynamicTextBannerBlocks;
    }

    private void fillTargetCondition(TargetingCondition value) {
        if (value == null) {
            return;
        }
        targetingConditions.selectByValue(value.getName());
    }

    private void selectDataSource(DynamicDataSource value) {
        if (value == null) {
            return;
        }
        dataSource.selectByValue(value.getValue());
    }

    private void fillDomain(String value) {
        if (value == null) {
            return;
        }
        domain.clear();
        domain.sendKeys(value);
        should(not(WebElementMatchers.hasClass(containsString("hidden"))))
                .whileWaitingUntil(timeoutHasExpired(TimeUnit.SECONDS.toMillis(4))).matches(domainMessage);
        should(WebElementMatchers.hasClass(containsString("hidden")))
                .whileWaitingUntil(timeoutHasExpired(TimeUnit.SECONDS.toMillis(4))).matches(domainMessage);
    }

    private void fillDomainParams(String value) {
        if (value == null) {
            return;
        }
        domainParamsSwitcher.click();
        domainParams.clear();
        domainParams.sendKeys(value);
    }

    @Override
    public void fillParameters(DynamicTextGroupWeb bean) {
        if (bean == null) {
            return;
        }
        super.fillParameters(bean);
        selectDataSource(bean.getDataSource());
        if (bean.getDataSource() != null) {
            switch (bean.getDataSource()) {
                case FEED:
                    selectSourceFeedFile(bean.getFeedFileName());
                    break;
                case DOMAIN:
                    fillDomain(bean.getDomain());
            }
        }
        fillDomainParams(bean.getDomainParams());
        fillTargetCondition(bean.getTargetingCondition());
        fillFilter(bean.getFeedFilters());
    }

    public DynamicDataSource getDataSource() {
        return DynamicDataSource.valueOf(dataSource.getSelectedValue().toUpperCase());
    }

    public String getDomain() {
        return domain.getText();
    }

    private String getDomainParams() {
        return domainParams.getText();
    }

    private TargetingCondition getTargetCondition() {
        TargetingCondition targetingCondition = new TargetingCondition();
        targetingCondition.setTargetConditionsEnum(TargetConditionsEnum
                .getEnumByValue(targetingConditions.getSelectedValue()));
        targetingCondition.setClickPrise(getClickPrise());
        return targetingCondition;
    }

    private Double getClickPrise() {
        if (isExists(clickPrice)) {
            return Double.valueOf(clickPrice.getText());
        }
        else {
            return null;
        }
    }

    @Override
    public void checkParameters(BeanDifferMatcher<DynamicTextGroupWeb> matcher) {
        assertThat("динамическая текстовая группа не удовлетворяет ожиданиям", getBean(matcher.getExpectedBean()),
                matcher);
    }

    @Override
    public DynamicTextGroupWeb getBean(DynamicTextGroupWeb expectedBean) {
        DynamicTextGroupWeb actualBean = inAccordanceWith(expectedBean)
                .forExisting(map(super.getBean(expectedBean), DynamicTextGroupWeb.class))
                .set("domainParams", this, DynamicTextGroupBlock::getDomainParams)
                .getActualBean();
        assumeThat("источник данных имеет ожидаемое значение", staticDataSource.getText(),
                equalTo(TextResourceFormatter.resource(expectedBean.getDataSource()).toString()));

        actualBean.setDataSource(expectedBean.getDataSource());
        if (expectedBean.getDataSource() == DynamicDataSource.DOMAIN) {
            actualBean.setDomain(getDomain());
            actualBean.setTargetingCondition(getTargetCondition());
        } else {
            actualBean.setFeedFileName(getFeedFileName());
            actualBean.setFeedFilters((List<DynamicFeedFilter>) getFeedFilters(expectedBean.getFeedFilters()));
        }
        return actualBean;
    }

    public void fillBanner(DynamicTextBannerWeb dynamicTextBannerWeb) {
        Long bannerId = Optional.ofNullable(dynamicTextBannerWeb.getBannerId()).orElse(0L);
        dynamicTextBannerBlocks.stream().filter(x -> x.getBannerId().equals(bannerId))
                .findFirst()
                .orElseThrow(() -> new AssumptionException("Ожидалось что в группе есть баннер"))
                .fillParameters(dynamicTextBannerWeb);
    }


    public void checkBannerParameters(BeanDifferMatcher<DynamicTextBannerWeb> matcher, Long bannerId) {
        dynamicTextBannerBlocks.stream().filter(x -> x.getBannerId().equals(bannerId))
                .findFirst()
                .orElseThrow(() -> new AssumptionException("Ожидалось что в группе есть баннер"))
                .checkParameters(matcher);
    }

    public DynamicTextBannerBlock getBannerBlockById(Long adId) {
        DynamicTextBannerBlock block = getDynamicTextBannerBlocks().stream()
                .filter(t -> adId.equals(t.getBannerId()))
                .findFirst().orElseThrow(() -> new DirectWebError("Не найден баннер " + adId));
        block.setConfig(config);
        return block;
    }

    private void selectSourceFeedFile(String feedName) {
        if (feedName == null) {
            return;
        }
        feedsSelectButton.click();
        waitForElement(feedsPopup, 3, WebElementMatchers.isDisplayed());
        feedsPopup.findElements(By.xpath(".//div[contains(@class, 'b-chooser__item')]"))
                .stream()
                .filter(x -> x.getAttribute("data-bem").contains(feedName))
                .findFirst()
                .orElseThrow(NoSuchElementException::new)
                .click();
    }

    public void openMinusWordsPopup() {
        minusWordsButton.click();
    }

    public void openRegionsPopup() {
        regionsButton.click();
    }

    public void openAdjustmentRatesPopup() {
        adjustmentRatesButton.click();
    }

    @Override
    protected EditFilterBlockBase getFilterBlock() {
        return editDynamicFilterBlock;
    }

    public String getFeedFileName() {
        return feedFileName.getText();
    }
}
