package ru.yandex.autotests.direct.web.pages.mediaplans.blocks;

import org.openqa.selenium.WebElement;
import org.openqa.selenium.support.FindBy;
import ru.yandex.autotests.direct.web.data.priceconstructor.MediaplanTrafficVolume;
import ru.yandex.autotests.direct.web.objects.banners.ForecastInfoWeb;
import ru.yandex.autotests.direct.web.objects.banners.OptimizedPhraseInfoWeb;
import ru.yandex.autotests.direct.web.pages.banners.blocks.phrases.BannerPhrasesBlock;
import ru.yandex.autotests.direct.web.pages.banners.blocks.phrases.BannerPreviewBlock;
import ru.yandex.autotests.direct.web.util.DirectWebError;
import ru.yandex.autotests.direct.web.util.testinterfaces.IWebFrom;
import ru.yandex.qatools.htmlelements.annotations.Name;

import java.util.LinkedList;
import java.util.List;

import static ch.lambdaj.Lambda.on;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.direct.utils.matchers.BeanEquals.beanEquals;
import static ru.yandex.autotests.direct.web.data.priceconstructor.MediaplanTrafficVolume.FIVE;
import static ru.yandex.autotests.direct.web.util.beanutils.BeanFieldsSetter.inAccordanceWith;

@FindBy(xpath = ".//ul[contains(@class, 'b-banner-list')] | .//div[@id = 'content']")
public class MediaplanPhrasesBlock extends BannerPhrasesBlock implements IWebFrom<OptimizedPhraseInfoWeb> {

    protected List<BannerPreviewBlock> bannerPreviewBlockList;
    protected List<MediaplanPhraseRowBlock> phrasesList;

    @Name("Прогноз на неделю")
    @FindBy(xpath = ".//span[contains(@class, 'b-forecast-switcher__week')]")
    private WebElement weekForecast;

    @Name("Прогноз на месяц")
    @FindBy(xpath = ".//span[contains(@class, 'b-forecast-switcher__month')]")
    private WebElement monthForecast;

    public Long getOptimizedBannerID(Long bannerId) {
        return bannerPreviewBlockList.stream().filter(x -> x.getBannerIdHeader().contains(bannerId.toString()))
                .findFirst()
                .orElseThrow(() -> new DirectWebError("Не найден баннер " + bannerId))
                .getBannerId();
    }

    /**
     * Поиск фраз, выполняется по префиксу,
     * т.к. для супера на странице медиоплана - к фразе рубрики добавляется ее номер,
     * а на этой же странице для клиента номера рубрики нет.
     *
     * @param bannerId   id баннера ожидаемой фразы
     * @param phraseText Текст ожидаемой фразы
     * @return Возвращается блок строки для найденной фразы или null
     */
    private MediaplanPhraseRowBlock getPhraseRow(Long bannerId, String phraseText) {
        for (MediaplanPhraseRowBlock mediaplanPhraseRowBlock : phrasesList) {
            if (!bannerId.equals(mediaplanPhraseRowBlock.getBannerId()))
                continue;
            String phrase = mediaplanPhraseRowBlock.getPhrase();
            if (phrase != null && phrase.startsWith(phraseText)) {
                return mediaplanPhraseRowBlock;
            }
        }
        return null;
    }

    public List<String> getPhrasesForBanner(Long bannerId) {
        List<String> result = new LinkedList<>();
        for (MediaplanPhraseRowBlock mediaplanPhraseRowBlock : phrasesList) {
            if (!bannerId.equals(mediaplanPhraseRowBlock.getBannerId()))
                continue;
            result.add(mediaplanPhraseRowBlock.getPhrase());
        }
        return result;
    }

    private String getPhraseTextFor(OptimizedPhraseInfoWeb phraseInfo) {
        return phraseInfo.getPhrase();
    }

    public OptimizedPhraseInfoWeb getFormFieldsAccording(OptimizedPhraseInfoWeb expectedPhrase) {

        MediaplanPhraseRowBlock rowPhrase = getPhraseRow(
                expectedPhrase.getBannerId(), getPhraseTextFor(expectedPhrase));
        if (rowPhrase != null) {
            OptimizedPhraseInfoWeb result = new OptimizedPhraseInfoWeb();
            result.setPhrase(expectedPhrase.getPhrase());
            result.setBannerId(expectedPhrase.getBannerId());
            result = inAccordanceWith(expectedPhrase).forExisting(result)
                    .set("trafficVolume", rowPhrase, on(MediaplanPhraseRowBlock.class).getPriceType())
                    .getActualBean();

            if (expectedPhrase.getForecastInfoWeb() != null) {
                ForecastInfoWeb forecastInfoWeb = new ForecastInfoWeb();
                forecastInfoWeb = inAccordanceWith(expectedPhrase.getForecastInfoWeb()).forExisting(forecastInfoWeb)
                        .set("CTRForecast", rowPhrase,
                                on(MediaplanPhraseRowBlock.class).getCTRForecast(result.getTrafficVolume()))
                        .set("showsForecast", rowPhrase,
                                on(MediaplanPhraseRowBlock.class).getShowsForecast())
                        .set("clicksForecast", rowPhrase,
                                on(MediaplanPhraseRowBlock.class).getClicksForecast(result.getTrafficVolume()))
                        .set("budgetForecast", rowPhrase,
                                on(MediaplanPhraseRowBlock.class).getBudgetForecast(result.getTrafficVolume()))
                        .getActualBean();
                result.setForecastInfoWeb(forecastInfoWeb);
            }
            return result;
        }
        return null;
    }

    public Double getGuaranteeCTRForPhrase(OptimizedPhraseInfoWeb expectedPhrase) {
        MediaplanPhraseRowBlock rowPhrase = getPhraseRow(
                expectedPhrase.getBannerId(), getPhraseTextFor(expectedPhrase));
        if (rowPhrase != null) {
            return Double.valueOf(rowPhrase.getCTRForecast(FIVE));
        }
        return null;
    }

    public void fillParameters(OptimizedPhraseInfoWeb phraseInfo) {
        MediaplanPhraseRowBlock row =
                getPhraseRow(phraseInfo.getBannerId(), getPhraseTextFor(phraseInfo));
        if (row != null) {
            String val = phraseInfo.getTrafficVolume().getId();
            row.priceRadioButton.selectByValue(val);
        }
    }

    public void checkParameters(OptimizedPhraseInfoWeb expectedPhrase) {
        OptimizedPhraseInfoWeb actualPhrase = getFormFieldsAccording(expectedPhrase);
        assertThat("неверные параметры фразы " + expectedPhrase.getPhrase(), actualPhrase, beanEquals(expectedPhrase));
    }

    public void declinePhrase(OptimizedPhraseInfoWeb phraseInfoWeb) {
        MediaplanPhraseRowBlock row =
                getPhraseRow(phraseInfoWeb.getBannerId(), getPhraseTextFor(phraseInfoWeb));
        if (row != null) {
            row.deselectPhraseSwitcher();
        }
    }

    public void setPhraseCheckBox(OptimizedPhraseInfoWeb phraseInfoWeb, boolean checkBoxState) {
        MediaplanPhraseRowBlock row =
                getPhraseRow(phraseInfoWeb.getBannerId(), getPhraseTextFor(phraseInfoWeb));
        if (row != null) {
            row.setPhraseCheckBox(checkBoxState);
        }
    }

    public void checkPhraseNotVisible(OptimizedPhraseInfoWeb phraseInfoWeb) {
        MediaplanPhraseRowBlock row =
                getPhraseRow(phraseInfoWeb.getBannerId(), getPhraseTextFor(phraseInfoWeb));
        assertThat("фраза не должна отображаться " + phraseInfoWeb.getPhrase(), row, equalTo(null));
    }


    public void clickOnWeekForecast() {
        weekForecast.click();
    }

    public void clickOnMonthForecast() {
        monthForecast.click();
    }
}
