package ru.yandex.autotests.direct.web.pages.priceconstructor;

import java.util.List;

import org.hamcrest.Matcher;
import org.openqa.selenium.By;
import org.openqa.selenium.NoSuchElementException;
import org.openqa.selenium.WebElement;
import org.openqa.selenium.support.FindBy;

import ru.yandex.autotests.direct.web.data.SelectInputMethod;
import ru.yandex.autotests.direct.web.data.priceconstructor.ConstructorPlatformSelect;
import ru.yandex.autotests.direct.web.data.priceconstructor.ContextConstructor;
import ru.yandex.autotests.direct.web.data.priceconstructor.OfflineConstructorForm;
import ru.yandex.autotests.direct.web.data.priceconstructor.SearchConstructor;
import ru.yandex.autotests.direct.web.data.priceconstructor.TrafficVolume;
import ru.yandex.autotests.direct.web.data.textresources.PriceConstructorResource;
import ru.yandex.autotests.direct.web.pages.ExtendedHtmlElement;
import ru.yandex.autotests.direct.web.util.JavaScriptActions;
import ru.yandex.autotests.direct.web.util.WebElementsActions;
import ru.yandex.autotests.direct.web.util.matchers.IsDisplayedMatcher;
import ru.yandex.autotests.direct.web.webelements.NumericInput;
import ru.yandex.autotests.direct.web.webelements.bem.CheckBoxBEM;
import ru.yandex.autotests.direct.web.webelements.bem.SelectBEM;
import ru.yandex.qatools.htmlelements.annotations.Name;
import ru.yandex.qatools.htmlelements.element.Button;
import ru.yandex.qatools.htmlelements.element.HtmlElement;

import static ch.lambdaj.Lambda.on;
import static org.hamcrest.MatcherAssert.assertThat;
import static ru.yandex.autotests.direct.web.util.WebElementsActions.fillSelect;
import static ru.yandex.autotests.direct.web.util.WebElementsActions.getFirstElementMatches;
import static ru.yandex.autotests.direct.web.util.WebElementsActions.getVisibleElement;
import static ru.yandex.autotests.direct.web.util.WebElementsActions.setCheckboxState;
import static ru.yandex.autotests.direct.web.util.beanutils.BeanFieldsSetter.inAccordanceWith;

@Name("Форма инлайн/оффлайн конструктора цен")
public class PriceConstructorBem extends ExtendedHtmlElement {
    //region Fields
    @Name("Кнопка Назначить")
    @FindBy(xpath = ".//button[contains(@class, 'save')]")
    private List<Button> setPricesButton;

    @Name("Кнопка отмена или закрыть")
    @FindBy(xpath = ".//button[contains(@class, 'close')]")
    private List<Button> closeButton;

    @Name("Для активных на поиске/тематических площадках")
    @FindBy(css = ".b-prices-constructor-platform select")
    private List<SelectBEM> platform;

    @Name("Чекбокс выключения конструктора на поиске")
    @FindBy(xpath = ".//span[contains(@class, 'b-prices-constructor__platform-checkbox')]" +
            "[contains(@data-bem, 'search')]//input")
    private List<CheckBoxBEM> chooseSearch;

    @Name("Чекбокс выключения конструктора на площадках")
    @FindBy(xpath = ".//span[contains(@class, 'b-prices-constructor__platform-checkbox')]" +
            "[contains(@data-bem, 'context')]//input")
    private List<CheckBoxBEM> chooseContext;

    @Name("Назначить ставку для получения объёма трафика")
    @FindBy(xpath = ".//span[contains(@class, 'b-prices-constructor__price-base-select')]//select")
    private List<SelectBEM> trafficVolume;

    @Name("Поле ввода цены для поиска")
    @FindBy(xpath = ".//span[contains(@data-bem, 'search') " +
            "and contains(@class, 'b-prices-constructor__price-input')]//input")
    private List<NumericInput> searchPrice;

    @Name("Поле ввода процентов для поиска")
    @FindBy(xpath = ".//span[contains(@data-bem, 'search') and " +
            "contains(@class, 'b-prices-constructor__proc-input')]//input")
    private List<NumericInput> searchPercent;

    @Name("Поле ввода цены для площадок")
    @FindBy(xpath = ".//span[contains(@data-bem, 'context') " +
            "and contains(@class, 'b-prices-constructor__price-input')]//input")
    private List<NumericInput> contextPrice;

    @Name("Поле ввода процентов для площадок")
    @FindBy(xpath = ".//span[contains(@data-bem, 'context') and " +
            "contains(@class, 'b-prices-constructor__proc-input')]//input")
    private List<NumericInput> contextPercent;

    @Name("Процент охвата")
    @FindBy(xpath = ".//span[contains(@class, 'context-scope-select')]//select")
    private List<SelectBEM> contextScope;

    @Name("Табы выбора конструктора")
    @FindBy(xpath = ".//ul[contains(@class, 'tabs-menu')]")
    private TabbedPaneBEM tabbedPanel;

    @Name("Панель конструктора (нужно для скролла)")
    @FindBy(xpath = ".//div[contains(@class, 'set-phrases-prices__title')]")
    private List<WebElement> constructorPanel;
    //endregion

    //region Field getters
    private Button getSetPriceButton() {
        return getFirstElementMatches(setPricesButton, IsDisplayedMatcher.isDisplayed());
    }

    private Button getCloseButton() {
        return getFirstElementMatches(closeButton, IsDisplayedMatcher.isDisplayed());
    }

    private SelectBEM getPlatform() {
        return getFirstElementMatches(platform, IsDisplayedMatcher.isDisplayed());
    }

    private CheckBoxBEM getChooseSearch() {
        return getFirstElementMatches(chooseSearch, IsDisplayedMatcher.isDisplayed());
    }

    private CheckBoxBEM getChooseContext() {
        return getFirstElementMatches(chooseContext, IsDisplayedMatcher.isDisplayed());
    }

    private SelectBEM getTrafficVolume() {
        return getFirstElementMatches(trafficVolume, IsDisplayedMatcher.isDisplayed());
    }

    private NumericInput getSearchPrice() {
        return getFirstElementMatches(searchPrice, IsDisplayedMatcher.isDisplayed());
    }

    public String getSearchPriceText() {
        return getSearchPrice().getValue().replace(" ", "");
    }

    private NumericInput getSearchPercent() {
        return getFirstElementMatches(searchPercent, IsDisplayedMatcher.isDisplayed());
    }

    private NumericInput getContextPrice() {
        return getFirstElementMatches(contextPrice, IsDisplayedMatcher.isDisplayed());
    }

    public String getContextPriceText() {
        return getContextPrice().getValue().replace(" ", "");
    }

    private NumericInput getContextPercent() {
        return getFirstElementMatches(contextPercent, IsDisplayedMatcher.isDisplayed());
    }

    private SelectBEM getContextScope() {
        return getFirstElementMatches(contextScope, IsDisplayedMatcher.isDisplayed());
    }
    //endregion

    //region Actions
    public void clickPriceCommonTab() {
        tabbedPanel.clickTabByText(PriceConstructorResource.COMMON_PRICE_TAB.toString());
    }

    public void clickPriceWizardTab() {
        tabbedPanel.clickTabByText(PriceConstructorResource.WIZARD_BIDS_TAB.toString());
    }

    public void selectPlatform(ConstructorPlatformSelect platform) {
        getPlatform().selectByValue(platform.getValue());
    }

    public void setConstructorForSearchCheckBox(Boolean value) {
        setCheckboxState(getChooseSearch(), value);
    }

    public void setConstructorForContextCheckBox(Boolean value) {
        setCheckboxState(getChooseContext(), value);
    }

    private void fillTrafficVolume(TrafficVolume volume) {
        if (volume == null)
            return;
        fillSelect(getTrafficVolume(), volume.getVolume(), SelectInputMethod.BY_VALUE);
    }

    private void fillSearchPercent(String value) {
        if (value == null)
            return;
        new WebElementsActions(config).setNumericValue(getSearchPercent(), value);
    }

    private void fillSearchPrice(String value) {
        if (value == null)
            return;
        new WebElementsActions(config).setNumericValue(getSearchPrice(), value);
    }

    private void fillContextScope(String scope) {
        if (scope == null)
            return;
        getContextScope().selectByVisibleText(scope);
    }

    private void fillContextPercent(String percent) {
        if (percent == null)
            return;
        new WebElementsActions(config).setNumericValue(getContextPercent(), percent);
    }

    private void fillContextPrice(String price) {
        if (price == null)
            return;
        new WebElementsActions(config).setNumericValue(getContextPrice(), price);
    }

    public void fillSearchParameters(SearchConstructor constructor) {
        new JavaScriptActions(config).scrollToElement(getVisibleElement(constructorPanel));
        fillTrafficVolume(constructor.getTrafficVolume());
        fillSearchPercent(constructor.getPercent());
        fillSearchPrice(constructor.getPrice());
    }

    public void fillContextParameters(ContextConstructor constructor) {
        new JavaScriptActions(config).scrollToElement(getVisibleElement(constructorPanel));
        fillContextScope(constructor.getContextScope());
        fillContextPercent(constructor.getPercent());
        fillContextPrice(constructor.getPrice());
    }

    public void clickOk() {
        getSetPriceButton().click();
    }

    public void clickCancel() {
        getCloseButton().click();
    }

    public void openConstructorOfflineByForm(OfflineConstructorForm form) {
        switch (form) {
            case COMMON_PRICE:
                clickPriceCommonTab();
                break;
            case BID_WIZZARD:
                clickPriceWizardTab();
                break;
        }
    }
    //endregion

    //region Assertions
    public void checkSearchParameters(SearchConstructor constructor, Matcher<SearchConstructor> matcher) {
        assertThat("неверные параметры конструктора на поиске", getSearchParameters(constructor), matcher);
    }

    public void checkContextParameters(ContextConstructor constructor, Matcher<ContextConstructor> matcher) {
        assertThat("неверные параметры конструктора на площадках", getContextParameters(constructor), matcher);
    }
    //endregion

    public TrafficVolume getTrafficVolumeValue() {
        SelectBEM trafficVolume = getTrafficVolume();
        if (trafficVolume == null) {
            return null;
        } else {
            return TrafficVolume.valueOfSelect(trafficVolume.getFirstSelectedOption().getAttribute("text"));
        }
    }

    public String getContextScopeValue() {
        SelectBEM contextScope = getContextScope();
        if (contextScope == null) {
            return null;
        } else {
            return contextScope.getFirstSelectedOption().getAttribute("textContent");
        }
    }

    public SearchConstructor getSearchParameters(SearchConstructor expected) {
        return inAccordanceWith(expected)
                .forNew(SearchConstructor.class)
                .set("trafficVolume", this, on(PriceConstructorBem.class).getTrafficVolumeValue())
                .set("price", this, on(PriceConstructorBem.class).getSearchPriceText())
                .set("percent", getSearchPercent(), on(NumericInput.class).getValue())
                .getActualBean();
    }

    public ContextConstructor getContextParameters(ContextConstructor expected) {
        return inAccordanceWith(expected)
                .forNew(ContextConstructor.class)
                .set("price", this, on(PriceConstructorBem.class).getContextPriceText())
                .set("percent", getContextPercent(), on(NumericInput.class).getValue())
                .set("contextScope", this, on(PriceConstructorBem.class).getContextScopeValue())
                .getActualBean();
    }

    public static class TabbedPaneBEM extends HtmlElement {
        @Name("Табы")
        @FindBy(css = ".tabs-menu__tab")
        private List<WebElement> tabs;

        private List<WebElement> getTabs() {
            return tabs;
        }

        public void clickTabByText(String text) {
            getTabByText(text).click();
        }

        public WebElement getTabByText(String text) {
            for (WebElement tab : getTabs()) {
                    if (tab.findElement(By.xpath(".//span[contains(@class,'link')]")).getText().equals(text)) {
                        return tab;
                    }
            }
            throw new NoSuchElementException("No tab with text " + text + " found");
        }
    }

    public void checkChooseSearchCheckBox(Matcher matcher) {
        assertThat("чекбокс 'На поиске назначить цену' не соответсвует ожиданиям", getChooseSearch(), matcher);
    }
}
