package ru.yandex.autotests.direct.web.pages.retargeting;

import org.hamcrest.Matcher;
import org.openqa.selenium.WebElement;
import org.openqa.selenium.support.FindBy;
import ru.yandex.autotests.direct.utils.textresource.TextResourceFormatter;
import ru.yandex.autotests.direct.web.data.textresources.RetargetingResource;
import ru.yandex.autotests.direct.web.data.textresources.Strings;
import ru.yandex.autotests.direct.web.objects.retargeting.RetargetingConditionWeb;
import ru.yandex.autotests.direct.web.pages.retargeting.blocks.bem.EditConditionPopupBlockBem;
import ru.yandex.autotests.direct.web.pages.retargeting.blocks.bem.SavedConditionPageRowBem;
import ru.yandex.autotests.direct.web.pages.retargeting.elements.RetargetingPageCheckBox;
import ru.yandex.autotests.direct.web.steps.AlertSteps;
import ru.yandex.autotests.direct.web.util.DirectWebError;
import ru.yandex.autotests.direct.web.util.helpers.TextResourceHelper;
import ru.yandex.qatools.allure.webdriver.annotations.NamedUrl;
import ru.yandex.qatools.allure.webdriver.annotations.NamedUrls;
import ru.yandex.qatools.allure.webdriver.pages.BasePageObject;
import ru.yandex.qatools.htmlelements.annotations.Name;
import ru.yandex.qatools.htmlelements.element.Select;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;

import static java.util.stream.Collectors.toList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.*;
import static ru.yandex.autotests.direct.utils.matchers.TextResourceMatcher.textResourceMatcher;
import static ru.yandex.autotests.direct.web.util.WebElementsActions.getVisibleElement;
import static ru.yandex.autotests.direct.web.util.matchers.IsDisplayedMatcher.isDisplayed;
import static ru.yandex.qatools.htmlelements.matchers.MatcherDecorators.should;
import static ru.yandex.qatools.htmlelements.matchers.MatcherDecorators.timeoutHasExpired;

/**
 * @author Roman Kuhta (kuhtich@yandex-team.ru)
 */
@NamedUrls({
        @NamedUrl(name = "open.show.retargeting.cond", url = "/registered/main.pl?cmd=showRetargetingCond"),
        @NamedUrl(name = "open.show.retargeting.cond.for.login", url = "/registered/main.pl?cmd=showRetargetingCond&ulogin={1}"),
})
public class ShowRetargetingCondPage extends BasePageObject {

    private List<SavedConditionPageRowBem> savedConditionRows;

    private EditConditionPopupBlockBem editConditionPopup;

    @Name("Кнопка добавления условия")
    @FindBy(xpath = ".//span[contains(@class, 'p-retargeting-conditions__new-condition')]")
    private List<WebElement> conditionAddButtons;

    @Name("Кнопка ОК")
    @FindBy(xpath = ".//span[contains(@class, 'actions__ok')]")
    private WebElement okButton;

    @Name("Выбор действия")
    @FindBy(xpath = ".//select[contains(@class, 'actions__select')]")
    private Select actionsSelect;

    @Name("Ссылка получения доступа к целям Метрики")
    @FindBy(linkText = Strings.LINK_TEXT_GET_METRIC_GOALS_ACCESS)
    private WebElement linkMetricGoalsAccess;

    @Name("Картинка загрузки")
    @FindBy(xpath = ".//div[@class='b-loader__img']")
    private WebElement conditionsLoaderImage;

    @Name("Чекбокс выбора всех строк")
    @FindBy(xpath = ".//thead[@class='b-table__header-top']//td[contains(@class, 'b-table__cb')]//input")
    private RetargetingPageCheckBox selectAllRowsCheckbox;

    @Name("Таблица с условиями")
    @FindBy(xpath = ".//div[contains(@class, 'p-retargeting-conditions__table-wrap')]")
    private WebElement conditionsTable;

    private List<SavedConditionPageRowBem> getSavedConditionRows() {
        return savedConditionRows;
    }

    private SavedConditionPageRowBem getSavedRetargetingRowBlock(String conditionName) {
        for (SavedConditionPageRowBem savedRetargetingRowBlock : getSavedConditionRows()) {
            if (savedRetargetingRowBlock.getConditionNameText().equals(conditionName)) {
                return savedRetargetingRowBlock;
            }
        }
        return null;
    }

    private EditConditionPopupBlockBem onEditRetargetingConditionPopup() {
        return editConditionPopup;
    }

    public void selectAllConditions() {
        selectAllRowsCheckbox.select();
    }


    public void selectAction(String actionText) {
        actionsSelect.selectByVisibleText(actionText);
    }

    public void clickOk() {
        okButton.click();
    }

    public void openWrongWorkingCondition(String conditionName) {
        SavedConditionPageRowBem retargetingRow = getSavedRetargetingRowBlock(conditionName);
        if (retargetingRow == null) {
            throw new DirectWebError("Не найдено условие " + conditionName);
        } else {
            retargetingRow.openWrongWorkingCondition();
        }
    }

    public void clickAddConditionButton() {
        WebElement addButton = getVisibleElement(conditionAddButtons);
        if (addButton == null)
            return;
        addButton.click();
        checkEditPopup(isDisplayed());
    }

    public void fillConditionParameters(RetargetingConditionWeb condition) {
        onEditRetargetingConditionPopup().fillParameters(condition);
    }

    public void openCondition(String conditionName) {
        SavedConditionPageRowBem retargetingRow = getSavedRetargetingRowBlock(conditionName);
        if (retargetingRow == null) {
            throw new DirectWebError("Не найдено условие " + conditionName);
        } else {
            retargetingRow.clickOnNameLink();
            checkEditPopup(isDisplayed());
            takeScreenshot();
        }
    }

    public void checkRetargetingCondition(RetargetingConditionWeb condition) {
        openCondition(condition.getRetargetingConditionName());
        onEditRetargetingConditionPopup().checkParameters(condition);
        onEditRetargetingConditionPopup().clickCancel();
    }

    public void clickDeleteCondition(String conditionName) {
        int rowsSize = getRetargetingConditionsList().size();
        SavedConditionPageRowBem retargetingRow = getSavedRetargetingRowBlock(conditionName);
        if (retargetingRow == null) {
            throw new DirectWebError("Не найдено условие " + conditionName);
        } else {
            retargetingRow.clickDelete();
            TextResourceFormatter retargetingMessageDeleteCondition =
                    TextResourceHelper.resource(RetargetingResource.RETARGETING_MESSAGE_DELETE_CONDITION);

            AlertSteps.getInstance(AlertSteps.class, config).accept(
                    textResourceMatcher(retargetingMessageDeleteCondition,
                            equalTo(retargetingMessageDeleteCondition.toString()))
            );
            assertThat("не удалилось условие " + conditionName, getSavedConditionRows(),
                    should(hasSize(lessThan(rowsSize)))
                            .whileWaitingUntil(timeoutHasExpired(TimeUnit.SECONDS.toMillis(30)))
            );
        }
    }

    public void clickSaveCondition() {
        onEditRetargetingConditionPopup().clickSave();
    }

    public void checkError(String errorText) {
        onEditRetargetingConditionPopup().checkErrorInPopup(errorText);
    }

    public void checkEditPopup(Matcher matcher) {
        assertThat("окно ретаргетинга не прошло проверку", getEditPopup(),
                should(matcher).whileWaitingUntil(timeoutHasExpired(TimeUnit.SECONDS.toMillis(30))));
    }

    public void checkConditionsList(Matcher matcher) {
        assertThat("список условий ретаргетинга не прошел проверку", getSavedConditionRows(),
                should(matcher).whileWaitingUntil(timeoutHasExpired(TimeUnit.SECONDS.toMillis(30))));
    }

    public List<String> getRetargetingConditionsList() {
        return getSavedConditionRows().stream().map(x -> x.getConditionNameText()).collect(toList());
    }

    public List<String> getAllUsedCampaigns() {
        List<String> res = new ArrayList<>();
        Pattern pattern = Pattern.compile("cid=\\d+");
        for (SavedConditionPageRowBem row : getSavedConditionRows()) {
            for (String link : row.getCampaignLinksList()) {
                java.util.regex.Matcher matcher = pattern.matcher(link);
                if (matcher.find()) {
                    String campId = matcher.group().replace("cid=", "");
                    if (!res.contains(campId)) {
                        res.add(campId);
                    }
                }
            }
        }
        return res;
    }

    public void clickSaveConditionAsNew() {
        onEditRetargetingConditionPopup().clickSaveAsNew();
    }

    public EditConditionPopupBlockBem getEditPopup() {
        return editConditionPopup;
    }

    public WebElement getTabel() {
        return conditionsTable;
    }

    public void showAnotherCamps(String conditionName) {
        SavedConditionPageRowBem retargetingRow = getSavedRetargetingRowBlock(conditionName);
        if (retargetingRow == null) {
            throw new DirectWebError("Не найдено условие " + conditionName);
        } else {
            retargetingRow.showAnotherCamps();
        }
    }
}
