package ru.yandex.autotests.direct.web.pages.retargeting.blocks;

import org.hamcrest.Matcher;
import org.openqa.selenium.WebElement;
import org.openqa.selenium.support.FindBy;
import ru.yandex.autotests.direct.web.objects.retargeting.RetargetingConditionWeb;
import ru.yandex.autotests.direct.web.util.DirectWebError;
import ru.yandex.autotests.direct.web.util.testinterfaces.IWebFrom;
import ru.yandex.autotests.direct.web.webelements.DirectButton;
import ru.yandex.qatools.htmlelements.annotations.Name;
import ru.yandex.qatools.htmlelements.element.HtmlElement;
import ru.yandex.qatools.htmlelements.element.TextInput;

import java.util.List;

import static ch.lambdaj.Lambda.on;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.direct.utils.matchers.BeanEquals.beanEquals;
import static ru.yandex.autotests.direct.web.util.WebElementsActions.*;
import static ru.yandex.autotests.direct.web.util.beanutils.BeanFieldsSetter.inAccordanceWith;

/**
 * @author Roman Kuhta (kuhtich@yandex-team.ru)
 */
@Name("Попап условия ретаргетинга")
@FindBy(xpath = "//div[contains(@class, 'i-popup_visibility_visible')]//div[@class='b-popupa__content']")
public class RetargetingPopupBlock extends HtmlElement implements IWebFrom<RetargetingConditionWeb> {
    @Name("Список групп")
    private List<RetargetingGroupBlock> groups;

    private List<SavedRetargetingRowBlock> savedRetargetingRowList;

    @Name("Имя")
    @FindBy(css = "span.b-retargeting-condition__name input")
    private List<TextInput> nameList;

    @Name("Добавить примечание")
    @FindBy(css = "span.b-retargeting-condition__add-comment")
    private List<WebElement> addDetailsButton;

    @Name("Удалить примечание")
    @FindBy(css = "div.b-retargeting-condition__del-comment")
    private List<WebElement> removeDetailsButton;

    @Name("Примечание")
    @FindBy(css = "span.b-retargeting-condition__comment input")
    private List<TextInput> details;

    @Name("Кнопка добавления группы")
    @FindBy(xpath = ".//span[contains(@class, 'b-retargeting-condition__add-group')]")
    private WebElement addGroupButton;

    @Name("Кнопка сохранить, добавить или применить")
    @FindBy(xpath = ".//span[contains(@class, '__save')]//input")
    private List<DirectButton> saveButton;

    @Name("Кнопка сохранить как")
    @FindBy(xpath = ".//span[contains(@class, '__new')]//input")
    private List<DirectButton> saveAsButton;

    @Name("Кнопка отмена или назад")
    @FindBy(xpath = ".//span[contains(@class, '__cancel')]//input")
    private List<DirectButton> cancelButton;

    @Name("Первый таб 'Новое условие'")
    @FindBy(xpath = ".//li[contains(@class, 'b-tabbed-pane__tab')][1]")
    private WebElement newConditionTab;

    @Name("Второй таб 'Сохраненные условия'")
    @FindBy(xpath = ".//li[contains(@class, 'b-tabbed-pane__tab')][2]")
    private WebElement savedConditionsTab;

    private TextInput getNameElement() {
        return getVisibleElement(nameList);
    }

    private WebElement getAddDetailsButton() {
        return getVisibleElement(addDetailsButton);
    }

    private WebElement getRemoveDetailsButton() {
        return getVisibleElement(removeDetailsButton);
    }

    private TextInput getDetailsElement() {
        return getVisibleElement(details);
    }

    private List<RetargetingGroupBlock> getGroups() {
        return getVisibleElements(groups);
    }

    private DirectButton getSaveButton() {
        return getVisibleElement(saveButton);
    }

    private DirectButton getSaveAsButton() {
        return getVisibleElement(saveAsButton);
    }

    private DirectButton getCancelButton() {
        return getVisibleElement(cancelButton);
    }

    private RetargetingGroupBlock getRetargetingGroupBlock(String groupNumber) {
        for (RetargetingGroupBlock block : getGroups()) {
            if (groupNumber.equals(block.getGroupNumber())) {
                return block;
            }
        }
        return null;
    }

    private List<SavedRetargetingRowBlock> getSavedRetargetingRowList() {
        return savedRetargetingRowList;
    }

    private SavedRetargetingRowBlock getSavedRetargetingRowBlock(String conditionName) {
        for (SavedRetargetingRowBlock savedRetargetingRowBlock : getSavedRetargetingRowList()) {
            if (savedRetargetingRowBlock.getConditionNameText().equals(conditionName)) {
                return savedRetargetingRowBlock;
            }
        }
        return null;
    }

    public void clickSave() {
        getSaveButton().click();
    }

    public void clickSaveAsNew() {
        getSaveAsButton().click();
    }

    public void clickCancel() {
        getCancelButton().click();
    }

    public void clickNewConditionTab() {
        newConditionTab.click();
    }

    public void clickSavedConditionTab() {
        savedConditionsTab.click();
    }

    public String getName() {
        return getNameElement().getText();
    }

    public String getDetails() {
        return getDetailsElement().getText();
    }

    public void checkSavedRetargetingCondition(String conditionName) {
        SavedRetargetingRowBlock savedRetargetingRowBlock = getSavedRetargetingRowBlock(conditionName);
        if (savedRetargetingRowBlock == null) {
            throw new DirectWebError("Не найдено условие " + conditionName);
        } else {
            savedRetargetingRowBlock.selectRow();
        }
    }

    public void clickEditSavedRetargetingCondition(String conditionName) {
        SavedRetargetingRowBlock savedRetargetingRowBlock = getSavedRetargetingRowBlock(conditionName);
        if (savedRetargetingRowBlock == null) {
            throw new DirectWebError("Не найдено условие " + conditionName);
        } else {
            savedRetargetingRowBlock.clickEdit();
        }
    }

    @Override
    public void fillParameters(RetargetingConditionWeb bean) {
        fillTextInput(getNameElement(), bean.getRetargetingConditionName());
        if (bean.getRetargetingConditionDescription() != null) {
            if (getVisibleElements(addDetailsButton).size() > 0) {
                getAddDetailsButton().click();
            }
            fillTextInput(getDetailsElement(), bean.getRetargetingConditionDescription());
        } else {
            if (getVisibleElements(removeDetailsButton).size() > 0) {
                getRemoveDetailsButton().click();
            }
        }

        //удаляем или добавляем группы
        while (getGroups().size() > bean.getRetargetingConditionItemsWeb().length) {
            getGroups().get(0).clickDelete();
        }
        while (getGroups().size() < bean.getRetargetingConditionItemsWeb().length) {
            addGroupButton.click();
        }

        //заполняем группы
        for (int i = 1; i <= bean.getRetargetingConditionItemsWeb().length; i++) {
            RetargetingGroupBlock groupBlock = getRetargetingGroupBlock(String.valueOf(i));
            groupBlock.fillParameters(bean.getRetargetingConditionItemsWeb()[i - 1]);
        }
    }

    @Override
    public void checkParameters(RetargetingConditionWeb expectedBean) {
        assertThat("неверные параметры условия ретаргетинга", getFormFieldsAccording(expectedBean),
                beanEquals(expectedBean));
        assertThat("неверное количество групп условия", getGroups().size(),
                equalTo(expectedBean.getRetargetingConditionItemsWeb().length));
        if (getGroups().size() != expectedBean.getRetargetingConditionItemsWeb().length)
            return;
        for (int i = 1; i <= expectedBean.getRetargetingConditionItemsWeb().length; i++) {
            RetargetingGroupBlock groupBlock = getRetargetingGroupBlock(String.valueOf(i));
            groupBlock.checkParameters(expectedBean.getRetargetingConditionItemsWeb()[i - 1]);
        }
    }

    @Override
    public RetargetingConditionWeb getFormFieldsAccording(RetargetingConditionWeb expectedBean) {
        return inAccordanceWith(expectedBean).forNew(RetargetingConditionWeb.class)
                .set("bannerID", expectedBean, on(RetargetingConditionWeb.class).getBannerId())
                .set("retargetingConditionName", this,
                        on(RetargetingPopupBlock.class).getName())
                .set("retargetingConditionDescription", this,
                        on(RetargetingPopupBlock.class).getDetails())
                .getActualBean();
    }

    public void checkRetargetingPopupFields(RetargetingConditionWeb expectedBean, Matcher matcher) {
        assertThat("поле 'Имя' не соответствуют ожиданиям", getNameElement(), matcher);
        assertThat("поле 'Детали' не соответствуют ожиданиям", getDetailsElement(), matcher);
        if (expectedBean.getRetargetingConditionItemsWeb() != null) {
            for (int i = 1; i <= expectedBean.getRetargetingConditionItemsWeb().length; i++) {
                RetargetingGroupBlock groupBlock = getRetargetingGroupBlock(String.valueOf(i));
                groupBlock.checkGoalFields(matcher);
            }
        }
    }
}
