package ru.yandex.autotests.direct.web.pages.retargeting.blocks.bem;

/**
 * @author Roman Kuhta (kuhtich@yandex-team.ru)
 */

import java.util.Objects;

import org.hamcrest.Matcher;
import org.openqa.selenium.WebElement;
import org.openqa.selenium.support.FindBy;

import ru.yandex.autotests.direct.web.data.SelectInputMethod;
import ru.yandex.autotests.direct.web.data.retargeting.RetargetingConditionGoalItemWeb;
import ru.yandex.autotests.direct.web.objects.retargeting.RetargetingConditionGoalType;
import ru.yandex.autotests.direct.web.util.StringHelper;
import ru.yandex.autotests.direct.web.util.testinterfaces.IWebFrom;
import ru.yandex.autotests.direct.web.webelements.bem.RetargetingSelectBem;
import ru.yandex.autotests.direct.web.webelements.bem.SelectBEM;
import ru.yandex.qatools.htmlelements.annotations.Name;
import ru.yandex.qatools.htmlelements.element.HtmlElement;

import static ch.lambdaj.Lambda.on;
import static org.hamcrest.MatcherAssert.assertThat;
import static ru.yandex.autotests.direct.utils.matchers.BeanEquals.beanEquals;
import static ru.yandex.autotests.direct.web.util.WebElementsActions.fillSelect;
import static ru.yandex.autotests.direct.web.util.beanutils.BeanFieldsSetter.inAccordanceWith;

@Name("Цель в группе условия ретаргетинга")
@FindBy(xpath = ".//div[contains(@class, 'b-retargeting-condition-edit-rule__type')]/..")
public class RetargetingGoalBlockBEM extends HtmlElement implements IWebFrom<RetargetingConditionGoalItemWeb> {

    @Name("Кнопка удаления цели")
    @FindBy(xpath = ".//span[contains(@class , 'remove-rule')]")
    private WebElement removeGoalButton;

    @Name("Выбор типа цели")
    @FindBy(xpath = ".//div[contains(@class, 'b-retargeting-condition-edit-rule__type')]//select")
    private SelectBEM goalTypeSelect;

    @Name("Выбор цели")
    @FindBy(xpath = ".//div[contains(@class, 'b-retargeting-condition-edit-rule__list')]/div[contains(@class, 'dropdown')]//button")
    private RetargetingSelectBem goalSelect;

    @Name("Время цели")
    @FindBy(xpath = ".//div[contains(@class, 'b-retargeting-condition-edit-rule__field-day')]//input")
    private WebElement goalTimeValue;

    @Name("Порядковый номер цели")
    @FindBy(xpath = ".//div[@class = 'b-retargeting-condition-edit-rule__index']")
    private WebElement goalNumber;

    public void setGoalTime(Integer goalTime) {
        if (goalTime == null || getGoalTime().equals(goalTime.toString())) {
            return;
        }
        goalTimeValue.clear();
        goalTimeValue.sendKeys(goalTime.toString());
    }

    public String getGoalTime() {
        return goalTimeValue.getAttribute("value");
    }

    public RetargetingConditionGoalType getGoalType() {
        return RetargetingConditionGoalType.getEnumByValue(goalTypeSelect.getSelectedValue().toLowerCase());
    }

    public String getGoalText() {
        return StringHelper.withSimpleWhitespaces(goalSelect.getFirstSelectedOption().getText().split("\n")[0]);
    }

    public void clickDelete() {
        removeGoalButton.click();
    }

    public void clickOnGoalSelect() {
        goalSelect.getWrappedElement().click();
    }

    @Override
    public void fillParameters(RetargetingConditionGoalItemWeb bean) {
        fillSelect(goalTypeSelect, Objects.toString(bean.getGoalType(), null), SelectInputMethod.BY_VALUE);
        fillSelect(goalSelect, bean.getGoalText().replaceAll(":: ", ":: "),
                SelectInputMethod.BY_TEXT); //replace SPACE to NO-BREAK SPACE
        setGoalTime(bean.getTime());
    }

    @Override
    public void checkParameters(RetargetingConditionGoalItemWeb expectedBean) {
        assertThat("неверные параметры цели", getFormFieldsAccording(expectedBean),
                beanEquals(expectedBean));
    }

    @Override
    public RetargetingConditionGoalItemWeb getFormFieldsAccording(RetargetingConditionGoalItemWeb expectedBean) {
        return inAccordanceWith(expectedBean).forNew(RetargetingConditionGoalItemWeb.class)
                .set("time", this,
                        on(RetargetingGoalBlockBEM.class).getGoalTime())
                .set("goalText", this,
                        on(RetargetingGoalBlockBEM.class).getGoalText())
                .set("goalType", this,
                        on(RetargetingGoalBlockBEM.class).getGoalType())
                .getActualBean();
    }

    public void checkGoalSelect(Matcher matcher) {
        assertThat("поле не соответствуют ожиданиям", goalSelect, matcher);
    }

    public RetargetingConditionGoalItemWeb getRetargetingConditionGoalItemWeb() {
        RetargetingConditionGoalItemWeb result = new RetargetingConditionGoalItemWeb();
        result.setGoalText(getGoalText());
        result.setGoalType(getGoalType());
        result.setTime(Integer.parseInt(getGoalTime()));
        return result;
    }
}
