package ru.yandex.autotests.direct.web.pages.transfer;

import org.hamcrest.Matcher;
import org.openqa.selenium.WebElement;
import org.openqa.selenium.support.FindBy;

import ru.yandex.autotests.direct.utils.config.DirectTestRunProperties;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.money.MoneyFormat;
import ru.yandex.autotests.direct.web.objects.transfer.PayCampElement;
import ru.yandex.autotests.direct.web.objects.transfer.TransferMoney;
import ru.yandex.autotests.direct.web.objects.transfer.TransferType;
import ru.yandex.autotests.direct.web.pages.ExtendedHtmlElement;
import ru.yandex.qatools.htmlelements.annotations.Name;
import ru.yandex.qatools.htmlelements.element.Button;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.is;

/**
 * User: buhter
 * Date: 10.12.12
 * Time: 20:10
 */
@Name("Форма для переноса денежных средств между кампаниями")
@FindBy(css = "div.p-transfer-between-campaigns")
public class TransferForm extends ExtendedHtmlElement {

    @Name("Кнопка 'Перенести средства'")
    @FindBy(css = "button.p-transfer-between-campaigns__submit-button")
    public Button transferButton;

    @Name("Список кампаний с которых переносятся средства")
    @FindBy(xpath = ".//td[contains(@class, 'list_type_source')]/div[contains(@class, 'list')]")
    public WebElement campaignsFromList;

    @Name("Счетчик кампаний с которых переносятся средства")
    @FindBy(xpath = ".//td[contains(@class,'source')]//div[contains(@class,'count')]")
    public WebElement campaignsFromListCount;

    @Name("Список кампаний на которые переносятся средства")
    @FindBy(xpath = ".//td[contains(@class, 'list_type_destination')]/div[contains(@class, 'list')]")
    public WebElement campaignsToList;

    @Name("Счетчик кампаний на которые переносятся средства")
    @FindBy(xpath = ".//td[contains(@class,'destination')]//div[contains(@class,'count')]")
    public WebElement campaignsToListCount;

    @Name("Общая сумма переноса")
    @FindBy(css = "td.p-transfer-between-campaigns__summary-sum")
    public WebElement totalTransferValue;

    @Name("Таблица с кампаниями с которых переносятся средства")
    @FindBy(xpath = ".//div[contains(@class, 'direction_from')]/div[contains(@class, 'table-list')]")
    private TransferFormTable transferFormTableFrom;

    @Name("Таблица с кампаниями на которые переносятся средства")
    @FindBy(xpath = ".//div[contains(@class, 'direction_to')]/div[contains(@class, 'table-list')]")
    private TransferFormTable transferFormTableTo;

    private TransferFormTable getTransferFormTableFrom() {
        transferFormTableFrom.setConfig(config);
        return transferFormTableFrom;
    }

    private TransferFormTable getTransferFormTableTo() {
        transferFormTableTo.setConfig(config);
        return transferFormTableTo;
    }

    //Тип переноса выбранный на форме
    public TransferType transferType;

    public void selectTransferType(TransferType transferType) {
        this.transferType = transferType;
    }

    /**
     * Метод заполнения формы переноса средств по бину TransferMoney
     * Выбирает кампании для переноса и вводит суммы для переноса.
     *
     * @param transferMoney - бин для заполнения формы
     */
    public void fillTransferForm(TransferMoney transferMoney) {
        for (PayCampElement payCampElement : transferMoney.getFromCampaigns()) {
            if (payCampElement.getSum() == (null)) {
                if (transferMoney.getFromCampaigns().size() == 1) {
                    selectCampaignFrom(payCampElement.getCampaignId());
                }
            } else {
                setTransferFromValue(payCampElement.getCampaignId(),
                        payCampElement.getSum());
            }
        }
        for (PayCampElement payCampElement : transferMoney.getToCampaigns()) {
            if (payCampElement.getSum() == (null)) {
                if (transferMoney.getToCampaigns().size() == 1) {
                    selectCampaignTo(payCampElement.getCampaignId());
                }
            } else {
                setTransferToValue(payCampElement.getCampaignId(),
                        payCampElement.getSum());
            }
        }

    }

    /**
     * Метод проверки формы переноса средств по бину TransferMoney
     * Проверяет суммы для переноса, наличие и количество кампаний в списках переноса и общую сумму переноса средств
     *
     * @param transferMoney - бин для проверки формы
     * @param moneyCurrency -
     */
    public void checkTransferForm(TransferMoney transferMoney, MoneyCurrency moneyCurrency) {
        for (PayCampElement payCampElement : transferMoney.getFromCampaigns()) {
            if (payCampElement.getSum() != (null)) {
                shouldSeeTransferFromValue(payCampElement.getCampaignId(),
                        is(payCampElement.getSum()));
                shouldSeeCampaignsFromList(containsString(String.valueOf(payCampElement.getCampaignId())));
            }
        }
        for (PayCampElement payCampElement : transferMoney.getToCampaigns()) {
            if (payCampElement.getSum() != (null)) {
                shouldSeeTransferToValue(payCampElement.getCampaignId(),
                        is(payCampElement.getSum()));
                shouldSeeCampaignsToList(containsString(payCampElement.getCampaignId().toString()));
            }
        }
        if (transferType == TransferType.ALL_TO_ONE) {
            shouldSeeCampaignsFromListCounterValue(is(transferMoney.getFromCampaignsAmount()));
        }
        if (transferType == TransferType.ONE_TO_ALL) {
            shouldSeeCampaignsToListCounterValue(is(transferMoney.getToCampaignsAmount()));
        }
        if (transferMoney.getTotalTransferValue().equals(Money.valueOf(0f))) {
            shouldSeeTotalTransferValue(is("—"));
        } else {
            shouldSeeTotalTransferValue(is(transferMoney.getTotalTransferValue()
                    .stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED)
                    .concat(" ")
                    .concat(moneyCurrency.getAbbreviation(DirectTestRunProperties.getInstance().getDirectCmdLocale()))));
        }
    }

    public void shouldSeeEmptyFromTable(MoneyCurrency moneyCurrency) {
        shouldSeeTotalTransferValue(is("—"));
        shouldSeeCampaignsFromList(is("—"));
    }

    public void shouldSeeEmptyToTable(MoneyCurrency moneyCurrency) {
        shouldSeeTotalTransferValue(is("—"));
        shouldSeeCampaignsToList(is("—"));
    }

    public void clickClearFromCampaignsOption() {
        getTransferFormTableFrom().clearAll();
    }

    public void clickClearToCampaignsOption() {
        getTransferFormTableTo().clearAll();
    }

    public void clickTransferFromAllCampaignsOption() {
        getTransferFormTableFrom().transferAll();
    }

    public void clickSelectAllToCampaignsOption() {
        getTransferFormTableTo().transferAll();
    }

    public void clickDistributeToCampaignsOption() {
        getTransferFormTableTo().transferAll();
    }

    public void submitTransferForm() {
        transferButton.click();
    }

    private void selectCampaignFrom(Long campaignToId) {
        getTransferFormTableFrom().getTransferFormTableRow(campaignToId).selectCampaign();
    }

    private void selectCampaignTo(Long campaignToId) {
        getTransferFormTableTo().getTransferFormTableRow(campaignToId).selectCampaign();
    }

    private void setTransferToValue(Long campaignToId, String value) {
        getTransferFormTableTo().getTransferFormTableRow(campaignToId).fillSumAmountInput(value);
    }

    private void setTransferFromValue(Long campaignToId, String value) {
        getTransferFormTableFrom().getTransferFormTableRow(campaignToId).fillSumAmountInput(value);
    }

    private void shouldSeeCampaignsFromList(Matcher matcher) {
        assertThat("кампания отсутствует в списке кампаний с которых осуществяется перенос средств",
                campaignsFromList.getText(), matcher);
    }

    private void shouldSeeCampaignsToList(Matcher matcher) {
        assertThat("кампания отсутствует в списке кампаний на которые осуществяется перенос средств",
                campaignsToList.getText(), matcher);
    }

    //
    private void shouldSeeCampaignsFromListCounterValue(Matcher matcher) {
        assertThat("неверное значения счетчика кампаний с которых осуществяется перенос средств",
                Integer.parseInt(campaignsFromListCount.getText().replace("(", "").replace(")", "")), matcher);
    }

    public void shouldSeeCampaignsToListCounterValue(Matcher matcher) {
        assertThat("неверное значения счетчика кампаний на которые осуществяется перенос средств",
                Integer.parseInt(campaignsToListCount.getText().replace("(", "").replace(")", "")), matcher);
    }

    public void shouldSeeTotalTransferValue(Matcher matcher) {
        String transferValue = totalTransferValue.getText();
        //TODO убрать когда починят DIRECT-33481
        if (transferValue.matches("\\d* \\d.*")) {
            transferValue = transferValue.replaceFirst(" ", String.valueOf((char) 160));
        }
        assertThat("неверная общая сумма для переноса", transferValue, matcher);
    }

    private void shouldSeeTransferToValue(Long campaignToId, Matcher matcher) {
        assertThat(getTransferFormTableTo().
                        getTransferFormTableRow(campaignToId).getSumAmountValue(),
                matcher);
    }

    private void shouldSeeTransferFromValue(Long campaignToId, Matcher matcher) {
        assertThat(getTransferFormTableFrom().
                        getTransferFormTableRow(campaignToId).getSumAmountValue(),
                matcher);
    }

    public void checkAvailableTransferMoney(Long campaignToId, Matcher<String> matcher) {
        getTransferFormTableFrom().checkAvailableTransferMoney(campaignToId, matcher);
    }

    public String getAvailableTransferMoney(Long campaignToId) {
        return getTransferFormTableFrom().getAvailableTransferMoney(campaignToId);

    }
}