package ru.yandex.autotests.direct.web.steps;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import com.google.common.base.Preconditions;
import com.yandex.direct.api.v5.bidmodifiers.BidModifierGetItem;
import com.yandex.direct.api.v5.campaigns.CampaignGetItem;
import com.yandex.direct.api.v5.campaigns.TextCampaignSettingsEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.hamcrest.Matcher;

import ru.yandex.autotests.direct.cmd.DirectCmdSteps;
import ru.yandex.autotests.direct.cmd.data.banners.GroupsParameters;
import ru.yandex.autotests.direct.cmd.data.campaigns.ShowCampsRequest;
import ru.yandex.autotests.direct.cmd.data.campaigns.ShowCampsTabsEnum;
import ru.yandex.autotests.direct.cmd.data.commons.CampaignStrategy;
import ru.yandex.autotests.direct.cmd.data.commons.banner.Banner;
import ru.yandex.autotests.direct.cmd.data.commons.group.Group;
import ru.yandex.autotests.direct.cmd.data.commons.group.RetargetingCondition;
import ru.yandex.autotests.direct.cmd.data.strategy.AjaxSaveAutoBudgetRequest;
import ru.yandex.autotests.direct.cmd.data.strategy.AjaxSaveAutobudgetResponse;
import ru.yandex.autotests.direct.cmd.util.CmdStrategyBeans;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.FeedsUpdateStatus;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BannerImagesRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BannersRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.FeedsRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.MinusWordsRecord;
import ru.yandex.autotests.direct.utils.campaigns.CampaignTypeEnum;
import ru.yandex.autotests.direct.utils.matchers.BeanEqualsAssert;
import ru.yandex.autotests.direct.utils.model.MongoUser;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.strategy.data.Strategies;
import ru.yandex.autotests.direct.web.TestEnvironment;
import ru.yandex.autotests.direct.web.api.steps.DirectWebApiSteps;
import ru.yandex.autotests.direct.web.data.contactinfo.Regions;
import ru.yandex.autotests.direct.web.data.converters.BannerInfoWebConverter;
import ru.yandex.autotests.direct.web.data.converters.MobileBannerInfoWebConverter;
import ru.yandex.autotests.direct.web.data.factory.BeanFactories;
import ru.yandex.autotests.direct.web.data.factory.templates.TemplateEnum;
import ru.yandex.autotests.direct.web.objects.banners.BannerInfoWeb;
import ru.yandex.autotests.direct.web.objects.banners.BannerPhraseInfoWeb;
import ru.yandex.autotests.direct.web.objects.banners.MobileAppBannerInfoWeb;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.CampaignFakeInfo;
import ru.yandex.autotests.directapi.enums.AdGroupType;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupAddItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignAddItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignAddItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignSettingMap;
import ru.yandex.autotests.directapi.model.common.Value;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.steps.UserSteps;
import ru.yandex.autotests.irt.testutils.allure.AllureUtils;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.allure.webdriver.annotations.WithoutScreenshot;

import static com.google.common.collect.ImmutableList.of;
import static java.lang.Long.parseLong;
import static java.util.stream.Collectors.toList;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsNot.not;
import static ru.yandex.autotests.direct.utils.beans.BeanWrapper.wrap;
import static ru.yandex.autotests.direct.utils.matchers.BeanEquals.beanEquals;
import static ru.yandex.autotests.direct.web.TestEnvironment.getNewHttpClientSteps;
import static ru.yandex.autotests.direct.web.TestEnvironment.newDbSteps;
import static ru.yandex.autotests.direct.web.data.TestLogins.AT_DIRECT_SUPER;
import static ru.yandex.autotests.direct.web.data.converters.BannerInfoWebConverter.cmdBannerToBannerInfoWeb;
import static ru.yandex.autotests.direct.web.data.converters.BannerInfoWebConverter.cmdGroupToBannerInfoWeb;
import static ru.yandex.autotests.direct.web.data.converters.BannerInfoWebConverter.toCmdBanner;
import static ru.yandex.autotests.direct.web.data.converters.BannerInfoWebConverter.toCmdGroup;
import static ru.yandex.autotests.direct.web.data.converters.SiteLinkInfoWebConverter.expandSiteLinks;
import static ru.yandex.autotests.direct.web.util.beanutils.BeanHelper.copyProperties;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

public class BackendSteps {
    private static final CidLoginCache loginsCache = new CidLoginCache();

    private String login;
    private String fakeLogin;
    private DirectWebApiSteps directWebApiSteps;

    public BackendSteps(String userLogin) {
        this.login = userLogin;
    }

    public BackendSteps() {
        this.login = TestEnvironment.getDirectWebProperties().getSuperLogin();
    }

    public static Boolean toBoolean(String directApiBooleanString) {
        if (directApiBooleanString == null) {
            return null;
        }
        switch (directApiBooleanString) {
            case Value.YES:
                return true;
            case Value.NO:
                return false;
            default:
                return null;
        }
    }

    public static String toDirectApiBooleanString(Boolean value) {
        return value ? Value.YES : Value.NO;
    }

    protected ApiSteps getApiSteps() {
        return TestEnvironment.getDirectApi();
    }

    protected UserSteps getApiUserSteps() {
        return getApiSteps().as(login, fakeLogin).userSteps;
    }

    protected UserSteps getApiUserSteps(String login) {
        return getApiSteps().as(login).userSteps;
    }

    protected DirectWebApiSteps directWebApiSteps() {
        if (directWebApiSteps == null) {
            directWebApiSteps = new DirectWebApiSteps();
            directWebApiSteps.authoriseAs(MongoUser.get(AT_DIRECT_SUPER.getLogin()));
        }
        return directWebApiSteps;
    }

    protected DirectCmdSteps getCmdSteps() {
        return getNewHttpClientSteps();
    }

    public BackendSteps as(String login) {
        this.login = login;
        return this;
    }

    public BackendSteps withFakeLogin(String fakeLogin) {
        this.fakeLogin = fakeLogin;
        return this;
    }

    // Создание кампании

    @Step("Создание через api архивной кампании под супером для логина {0}")
    public Long createArchivedCampaign(String login) {
        Long campaignId = createStoppedCampaign(login);
        makeCampaignsReadyForArchivation(campaignId);
        makeCampaignArchived(campaignId);
        loginsCache.mapCidToLogin(campaignId, login);
        return campaignId;
    }

    @Step("Создание через api остановленной кампании под супером для логина {0}")
    public Long createStoppedCampaign(String login) {
        Long campaignId = createClientActiveCampaign(login);
        makeCampaignStopped(campaignId);
        loginsCache.mapCidToLogin(campaignId, login);
        return campaignId;
    }

    @Step("Создание через api черновика кампании под логином {0} для логина {1}")
    public Long createManagerDraftCampaign(String managerLogin, String clientLogin) {
        UserSteps managerSteps = getApiUserSteps(managerLogin);
        Long campaignId = managerSteps.campaignSteps().addDefaultTextCampaign(clientLogin);
        Long adGroupId = managerSteps.adGroupsSteps().addGroup(
                new AdGroupAddItemMap()
                        .withCampaignId(campaignId)
                        .withDefaultGroup(AdGroupType.BASE)
                        .withName("Default group")
                        .withRegionIds(Long.parseLong(Regions.RUSSIA.getRegionID())),
                clientLogin);
        managerSteps.adsSteps().addDefaultTextAd(adGroupId, clientLogin);
        managerSteps.keywordsSteps().addDefaultKeyword(clientLogin, adGroupId);
        loginsCache.mapCidToLogin(campaignId, clientLogin);
        return campaignId;
    }

    @Step("Создание через api активной кампании под логином {0} для логина {1}")
    public Long createManagerActiveCampaign(String managerLogin, String clientLogin) {
        Long campaignId = createManagerDraftCampaign(managerLogin, clientLogin);
        makeCampaignActive(campaignId);
        loginsCache.mapCidToLogin(campaignId, clientLogin);
        return campaignId;
    }

    @Step("Создание через api черновика кампании под супером для логина {0}")
    public Long createClientDraftCampaign(String login) {
        String defaultKeyword = "утопить смартфон";
        Long campaignId = getApiUserSteps().campaignSteps().addCampaign(new CampaignAddItemMap()
                .defaultCampaignAddItem()
                .withTextCampaign(new TextCampaignAddItemMap()
                        .defaultTextCampaign()
                        .withSettings(new TextCampaignSettingMap()
                                .withOption(TextCampaignSettingsEnum.REQUIRE_SERVICING)
                                .withValue(YesNoEnum.NO))), login);
        Long adGroupId = getApiUserSteps().adGroupsSteps().addGroup(
                new AdGroupAddItemMap()
                        .withCampaignId(campaignId)
                        .withDefaultGroup(AdGroupType.BASE)
                        .withName("Default group")
                        .withRegionIds(Long.parseLong(Regions.RUSSIA.getRegionID())),
                login);
        getApiUserSteps().adsSteps().addDefaultTextAd(adGroupId, login);
        getApiUserSteps().keywordsSteps().addKeyword(login, adGroupId, defaultKeyword);
        loginsCache.mapCidToLogin(campaignId, login);
        return campaignId;
    }

    @Step("Создание через api активной кампании под супером для логина {0}")
    public Long createClientActiveCampaign(String login) {
        Long campaignId = createClientDraftCampaign(login);
        makeCampaignActive(campaignId);
        loginsCache.mapCidToLogin(campaignId, login);
        return campaignId;
    }

    @Step("Создание черновика кампании через api под супером для логина {0} и выставление стратегии")
    public Long createCampaignWithStrategy(User account, Strategies strategy) {
        AllureUtils.addJsonAttachment("Параметры стратегии", wrap(strategy).toString());
        Long campaignId = createClientDraftCampaign(account.getLogin());
        setStrategy(campaignId, strategy);
        return campaignId;
    }

    @Step("Через api создаем черновик РМП кампании под {0} для {1}")
    public Long createManagerDraftMobileAppCampaign(String managerLogin, String clientLogin) {
        Long campaignId = TestEnvironment.getApiSteps()
                .as(ru.yandex.autotests.directapi.model.Logins.SUPER_LOGIN, managerLogin)
                .userSteps.campaignSteps()
                .addDefaultMobileAppCampaign(clientLogin);
        Long adGroupId = getApiUserSteps().adGroupsSteps().addGroup(
                new AdGroupAddItemMap()
                        .withCampaignId(campaignId)
                        .withDefaultGroup(AdGroupType.MOBILE_CONTENT)
                        .withName("Default group")
                        .withRegionIds(Long.parseLong(Regions.RUSSIA.getRegionID())),
                clientLogin);
        getApiUserSteps().adsSteps().addDefaultMobileAppAd(adGroupId, clientLogin);
        getApiUserSteps().keywordsSteps().addDefaultKeyword(clientLogin, adGroupId);
        loginsCache.mapCidToLogin(campaignId, clientLogin);
        return campaignId;
    }

    @Step("Через api создаем активную РМП кампанию под {0} для {1}")
    public Long createManagerActiveMobileAppCampaign(String managerLogin, String clientLogin) {
        Long campaignId = createManagerDraftMobileAppCampaign(managerLogin, clientLogin);
        makeCampaignActive(campaignId);
        return campaignId;
    }

    @Step("Через api создаем активные РМП кампании в количестве {3} для {1} под {0}")
    @WithoutScreenshot
    public Long[] createManagerActiveMobileAppCampaigns(String managerLogin, String clientLogin, int amount) {
        Long[] campaigns = new Long[amount];
        IntStream.range(0, amount)
                .forEach(i -> campaigns[i] = createManagerActiveMobileAppCampaign(managerLogin, clientLogin));
        AllureUtils.addJsonAttachment("Список кампаний", wrap(campaigns).toString());
        return campaigns;
    }

    @Step("Через api создаем активные текстовые кампании в количестве {3} для {1} под {0}")
    @WithoutScreenshot
    public Long[] createManagerActiveCampaigns(String managerLogin, String clientLogin, int amount) {
        Long[] campaigns = new Long[amount];
        IntStream.range(0, amount).forEach(i -> campaigns[i] = createManagerActiveCampaign(managerLogin, clientLogin));
        AllureUtils.addJsonAttachment("Список кампаний", wrap(campaigns).toString());
        return campaigns;
    }

    @Step("Через api создаем активную РМП-кампанию под {0} для {1}")
    public Long createAgencyActiveMobileAppCampaign(String agencyLogin, String clientLogin) {
        return createManagerActiveMobileAppCampaign(agencyLogin, clientLogin);
    }

    @Step("Через api создаем активную кампанию под {0} для {1}")
    public Long createAgencyActiveCampaign(String agencyLogin, String clientLogin) {
        return createManagerActiveCampaign(agencyLogin, clientLogin);
    }

    @Step("Через api создаем черновик РМП кампаню для {0}")
    public Long createClientDraftMobileAppCampaign(String client) {
        Long campaignId = getApiUserSteps().campaignSteps().addDefaultMobileAppCampaign(client);
        Long adGroupId = getApiUserSteps().adGroupsSteps().addGroup(
                new AdGroupAddItemMap()
                        .withCampaignId(campaignId)
                        .withDefaultGroup(AdGroupType.MOBILE_CONTENT)
                        .withName("Default group")
                        .withRegionIds(Long.parseLong(Regions.RUSSIA.getRegionID())),
                client);
        getApiUserSteps().adsSteps().addDefaultMobileAppAd(adGroupId, client);
        getApiUserSteps().keywordsSteps().addDefaultKeyword(client, adGroupId);
        loginsCache.mapCidToLogin(campaignId, client);
        return campaignId;
    }

    @Step("Через api создаем активную РМП кампаню для {0}")
    public Long createClientActiveMobileAppCampaign(String client) {
        Long campaignId = createClientDraftMobileAppCampaign(client);
        makeCampaignActive(campaignId);
        return campaignId;
    }

    @Step
    public Long[] createClientActiveCampaigns(String client, int campaignsAmount) {
        Long[] campaigns = new Long[campaignsAmount];
        IntStream.range(0, campaignsAmount).forEach(i -> campaigns[i] = createClientActiveCampaign(client));
        return campaigns;
    }

    @Step
    public Long[] createClientActiveMobileAppCampaigns(String client, int campaignsAmount) {
        Long[] campaigns = new Long[campaignsAmount];
        IntStream.range(0, campaignsAmount).forEach(i -> campaigns[i] = createClientActiveMobileAppCampaign(client));
        return campaigns;
    }

    //-------------------

    //Действия над кампаниями

    @Step("Разрархивирование кампании {0}")
    public void unArchiveCampaigns(String login, Long... campaignId) {
        getCmdSteps().campsMultiActionSteps().unarchiveCampaigns(login, campaignId);
    }

    @Step("Получение через cmd отсортированного списка кампаний клиента {0}")
    public List<Long> getCampaignIds(String login) {
        return getCmdSteps().campaignSteps()
                .getShowCamps(new ShowCampsRequest().withTab(ShowCampsTabsEnum.ALL.getValue())
                        .withUlogin(login))
                .getCampaigns().stream()
                .map(x -> {
                    Long cid = Long.valueOf(x.getCid());
                    loginsCache.mapCidToLogin(cid, login);
                    return cid;
                })
                .sorted(Long::compare)
                .collect(toList());
    }

    @Step("Установка через api стратегии для кампании {0}")
    public void setStrategy(Long campaignId, Strategies strategy) {
        AllureUtils.addJsonAttachment("Параметры стратегии", wrap(strategy).toString());
        CampaignStrategy ajaxStrategy = CmdStrategyBeans.getStrategyBean(strategy);
        AjaxSaveAutobudgetResponse response = getCmdSteps().strategySteps().postAjaxSaveAutobudget(
                new AjaxSaveAutoBudgetRequest()
                        .withCid(campaignId.toString())
                        .withuLogin(loginsCache.getLoginByCid(campaignId))
                        .withJsonStrategy(ajaxStrategy));
        assumeThat("отсутствует сообщение об ошибке", response.getErrors(), is(nullValue()));
    }

    @Step("Подготовка к удалению кампании {0} через api")
    public void makeCampaignsReadyForArchivation(Long campaignId) {
        CampaignFakeInfo fakeInfo = TestEnvironment.getApiUserSteps()
                .campaignFakeSteps().fakeGetCampaignParams(campaignId);
        fakeInfo.setOrderID("0");
        getApiUserSteps().campaignFakeSteps().fakeCampaignParams(fakeInfo);
    }

    public CampaignFakeInfo getCampaignFakeInfo(int campaignId) {
        return getApiUserSteps().campaignFakeSteps().fakeGetCampaignParams(campaignId);
    }

    @Step("Удаление через api всех кампаний логина {0}")
    public void deleteAllCampaigns(String login) {
        List<CampaignGetItem> campaigns = getApiUserSteps().campaignSteps()
                .getCampaigns(login, new GetRequestMap()
                        .withAllFieldNames()
                        .withSelectionCriteria(new CampaignsSelectionCriteriaMap()));
        if (!CollectionUtils.isEmpty(campaigns)) {
            campaigns.forEach(x -> loginsCache.mapCidToLogin(x.getId(), login));
            deleteCampaigns(campaigns.stream().map(x -> x.getId()).toArray(Long[]::new));
        }
    }

    public void deleteCampaigns(String... campaignIds) {
        if (!ArrayUtils.isEmpty(campaignIds)) {
            deleteCampaigns(Stream.of(campaignIds).map(x -> x != null ? parseLong(x) : x).toArray(Long[]::new));
        }
    }

    @Step("Удаление кампаний через api")
    public void deleteCampaigns(Long... campaignIds) {
        AllureUtils.addJsonAttachment("Id кампаний", wrap(campaignIds).toString());
        if (!ArrayUtils.isEmpty(campaignIds)) {
            campaignIds = Stream.of(campaignIds).filter(x -> x != null).toArray(Long[]::new);
            getApiUserSteps().campaignFakeSteps().makeCampaignReadyForDelete(campaignIds);
            Stream.of(campaignIds).forEach(
                    id -> {
                        String login = loginsCache.getLoginByCid(id);
                        Long[] bids = getApiUserSteps().bidModifiersSteps()
                                .bidModifiersGetByCampaignId(login, id.intValue())
                                .stream()
                                .map(BidModifierGetItem::getId)
                                .toArray(Long[]::new);
                        if (!ArrayUtils.isEmpty(bids)) {
                            getApiUserSteps().bidModifiersSteps().bidModifiersDelete(login, bids);
                        }
                    });
            applyForIds((login, ids) -> {
                        getCmdSteps().campsMultiActionSteps().deleteCampaigns(login, ids);
                        Stream.of(ids).forEach(id ->
                                newDbSteps().useShardForLogin(login)
                                        .bannersPerformanceSteps()
                                        .deleteBannersPerformanceRecordsByCid(id));
                    },
                    loginsCache::getLoginByCid, campaignIds);
        }
    }

    @Step("Делаем кампанию {0} активной")
    public void makeCampaignActive(Long campaignId) {
        String login = loginsCache.getLoginByCid(campaignId);
        getApiUserSteps().makeCampaignActiveV5(login, campaignId);
    }

    @Step("Делаем кампанию {0} остановленной")
    public void makeCampaignStopped(Long campaignId) {
        String login = loginsCache.getLoginByCid(campaignId);
        getApiUserSteps().campaignFakeSteps().makeCampaignStopped(campaignId);
    }

    @Step("Архивируем кампанию {0}")
    public void makeCampaignArchived(Long campaignId) {
        String login = loginsCache.getLoginByCid(campaignId);
        getApiUserSteps().campaignSteps().campaignsArchive(login, campaignId);
    }

    public void setRandomOrderIdToCampaign(Long campaignId) {
        getApiUserSteps().campaignFakeSteps().setRandomOrderID(campaignId);
    }

    //-------------------

    // Создание группы и баннера

    @Step("Создание баннера через cmd в новую активную кампанию под супером для логина {0} с остатком {1}")
    public BannerInfoWeb createBannerToNewActiveCampaign(String clientLogin, Money sum, BannerInfoWeb bannerInfo) {
        BannerInfoWeb result = new BannerInfoWeb();
        bannerInfo.setBannerId(0L);
        copyProperties(result, bannerInfo);
        Long campaignId = getCmdSteps().campaignSteps().saveNewDefaultTextCampaign(clientLogin);
        Group group = toCmdGroup(bannerInfo);
        group.setBanners(of(toCmdBanner(bannerInfo)));
        getCmdSteps().groupsSteps().postSaveTextAdGroups(
                GroupsParameters
                        .forNewCamp(clientLogin, campaignId, group));
        cmdBannerToBannerInfoWeb(getCmdSteps().groupsSteps()
                .getGroups(clientLogin, campaignId)
                .get(0)
                .getBanners().get(0), result);
        makeCampaignActive(campaignId);
        getApiUserSteps().campaignFakeSteps().setCampaignSum(campaignId.intValue(), sum.floatValue());
        getApiUserSteps().groupFakeSteps().makeGroupFullyModerated(result.getAdGroupID());
        getApiUserSteps().bannersFakeSteps().makeBannerActive(result.getBannerId());
        loginsCache.mapBidToCid(result.getBannerId(), campaignId);
        loginsCache.mapCidToLogin(campaignId, clientLogin);
        return result;
    }

    public BannerInfoWeb createBannerToNewActiveCampaign(String clientLogin, Money sum, String template) {
        return createBannerToNewActiveCampaign(clientLogin, sum, BeanFactories.bannersFactory().getBean(template));
    }

    @Step("Создание через cmd баннера в кампанию {0}")
    public BannerInfoWeb createBanner(String template, Long campaignId) {
        BannerInfoWeb bannerInfo = BeanFactories.bannersFactory().getBean(template);
        return createBanner(bannerInfo, campaignId);
    }

    @Step("Добавление баннера в группу")
    public BannerInfoWeb createBanner(BannerInfoWeb bannerInfo, Long campaignId) {
        return addBannersToGroup(0L, campaignId, bannerInfo).stream()
                .max((x, y) -> x.getBannerId().compareTo(y.getBannerId()))
                .get();
    }

    @Step("Добавление {1} новых текстовых групп в кампанию {0}")
    public List<BannerInfoWeb> addNewDefaultGroupsToCampaign(Long campaignId, int amount) {
        String template = TemplateEnum.TextBanner.FULL_BANNER;
        List<BannerInfoWeb> bannerList = new ArrayList<>();
        IntStream.range(0, amount).forEach(x -> bannerList.add(createBanner(template, campaignId)));
        return bannerList;
    }

    @Step("Добавление {1} новых РМП групп в кампанию {0}")
    public List<Long> addNewDefaultMobileAppGroupsToCampaign(Long campaignId, int amount) {
        List<Long> adGroupIds = new ArrayList<>();
        IntStream.range(0, amount).forEach(x -> adGroupIds.add(addNewDefaultMobileAppGroupToCampaign(campaignId)));
        return adGroupIds;
    }

    @Step("Создание через API новой дефолтной РМП группы в кампании {0}")
    public Long addNewDefaultMobileAppGroupToCampaign(Long campaignId) {
        String clientLogin = loginsCache.getLoginByCid(campaignId);
        Long adGroupId = getApiUserSteps().adGroupsSteps().addGroup(
                new AdGroupAddItemMap()
                        .withCampaignId(campaignId)
                        .withDefaultGroup(AdGroupType.MOBILE_CONTENT)
                        .withName("Default group")
                        .withRegionIds(Long.parseLong(Regions.RUSSIA.getRegionID())),
                clientLogin);
        getApiUserSteps().adsSteps().addDefaultMobileAppAd(adGroupId, clientLogin);
        getApiUserSteps().keywordsSteps().addDefaultKeyword(clientLogin, adGroupId);
        return adGroupId;
    }

    @Step("Создание через cmd баннера в кампанию {0}")
    public List<BannerInfoWeb> addBannersToGroup(Long groupId, Long campaignId, BannerInfoWeb... bannerInfo) {
        Preconditions.checkArgument(!ArrayUtils.isEmpty(bannerInfo), "как минимум один баннер должен быть передан");
        AllureUtils.addJsonAttachment("Параметры баннера", wrap(bannerInfo).toString());
        Group group = toCmdGroup(bannerInfo[0]);
        group.setAdGroupID(groupId.toString());
        String ulogin = loginsCache.getLoginByCid(campaignId);
        group.setBanners(
                Stream.of(bannerInfo)
                        .map(x ->
                                toCmdBanner(x).withBid(0L))
                        .collect(toList()));
        getCmdSteps().groupsSteps().prepareGroupForUpdate(group, CampaignTypeEnum.TEXT);
        getCmdSteps().groupsSteps().postSaveTextAdGroups(
                GroupsParameters
                        .forExistingCamp(ulogin, campaignId, group));
        Group actualGroup = getCmdSteps().groupsSteps().getGroups(ulogin, campaignId)
                .stream()
                .max((x, y) -> Long.valueOf(x.getAdGroupID()).compareTo(Long.valueOf(y.getAdGroupID())))
                .get();

        List<BannerInfoWeb> result = actualGroup.getBanners().stream()
                .map(x -> {
                    loginsCache.mapBidToCid(x.getBid(), campaignId);
                    return cmdGroupToBannerInfoWeb(actualGroup, cmdBannerToBannerInfoWeb(x));
                }).collect(toList());
        return result;
    }
    //----------------------

    // Работа с группами и баннерами

    @Step("Получение через БД id группы баннера с id {0}")
    public Long getAdGroupIdByBannerId(Long bannerId) {
        return newDbSteps().useShardForLogin(loginsCache.getLoginByBid(bannerId))
                .bannersSteps()
                .getBanner(bannerId).getPid();
    }

    @Step("Получение через БД отсортированного списка баннеров кампании {0}")
    public List<BannersRecord> getBannersRecord(Long campaignId) {
        List<BannersRecord> records =
                newDbSteps().useShardForLogin(loginsCache.getLoginByCid(campaignId)).bannersSteps()
                        .getBannersByCid(campaignId).stream()
                        .sorted(Comparator.comparingLong(BannersRecord::getBid))
                        .collect(toList());
        records.forEach(x -> loginsCache.mapBidToCid(x.getBid(), campaignId));
        return records;
    }

    @Step("Получение через БД отсортированного списка ID баннеров кампании {0}")
    public List<Long> getBannerIds(Long campaignId) {
        return getBannersRecord(campaignId).stream().map(x -> x.getBid()).collect(toList());
    }

    @Step("Получение параметров баннера {0} через cmd")
    public BannerInfoWeb getBannerInfo(Long bannerId) {
        return getBannerInfoByBannerId(bannerId,
                BannerInfoWebConverter::cmdBannerToBannerInfoWeb,
                BannerInfoWebConverter::cmdGroupToBannerInfoWeb);
    }

    @Step("Получение параметров мобильного баннера {0} через cmd")
    public MobileAppBannerInfoWeb getMobileBannerInfo(Long bannerId) {
        return getBannerInfoByBannerId(bannerId,
                MobileBannerInfoWebConverter::cmdBannerToMobileAppBannerInfoWeb,
                MobileBannerInfoWebConverter::cmdGroupToMobileAppBannerInfoWeb);
    }

    private <T> T getBannerInfoByBannerId(Long bannerId, Function<Banner, T> bannerMapper,
            BiFunction<Group, T, T> groupMapper)
    {
        String login = loginsCache.getLoginByBid(bannerId);
        BannersRecord bannersRecord = newDbSteps().useShardForLogin(login).bannersSteps().getBanner(bannerId);
        Group group = getCmdSteps().groupsSteps()
                .getGroup(login, bannersRecord.getCid(), bannersRecord.getPid());
        T result =
                bannerMapper.apply(group.getBanners()
                        .stream()
                        .filter(x -> x.getBid().equals(bannerId))
                        .findFirst()
                        .orElseThrow(NoSuchElementException::new));
        return groupMapper.apply(group, result);
    }

    @Step("Перевод баннеров в активное состояние")
    public void makeBannerActive(Long... bannerId) {
        getApiUserSteps().bannersFakeSteps().makeBannerActive(bannerId);
    }

    public void moderateBanners(Long... bannerIds) {
        getApiUserSteps().bannersFakeSteps().makeBannersModerated(bannerIds);
    }

    @Step("Остановка баннеров через API")
    public void stopBanners(Long... bannerIds) {
        AllureUtils.addJsonAttachment("Id баннеров", wrap(bannerIds).toString());
        applyForIds((x, y) -> getApiUserSteps().adsSteps().adsSuspend(x, y), loginsCache::getLoginByBid,
                bannerIds);
    }

    @Step("Обновить баннер")
    public Long updateBanner(BannerInfoWeb banner) {
        AllureUtils.addJsonAttachment("Параметры баннера", wrap(banner).toString());
        Long cid = loginsCache.getCidByBid(banner.getBannerId());
        String ulogin = loginsCache.getLoginByCid(cid);
        Predicate<Group> hasBannerWithExpectedId = x -> x.getBanners().stream()
                .filter(y -> y.getBid().equals(banner.getBannerId()))
                .findFirst()
                .isPresent();
        Group group = getCmdSteps().groupsSteps().getGroups(ulogin, cid).stream()
                .filter(hasBannerWithExpectedId)
                .findFirst()
                .orElseThrow(NoSuchElementException::new);
        List<Banner> banners = group.getBanners();

        for (int i = 0; i < banners.size(); i++) {
            if (banners.get(i).getBid().equals(banner.getBannerId())) {
                banners.set(i, toCmdBanner(banner));
            } else {
                expandSiteLinks(banners.get(i));
            }
        }
        banner.setAdGroupID(parseLong(group.getAdGroupID()));
        banner.setAdGroupName(group.getAdGroupName());
        group = toCmdGroup(banner, group);
        getCmdSteps().groupsSteps().prepareGroupForUpdate(group, CampaignTypeEnum.TEXT);
        GroupsParameters params = GroupsParameters.forExistingCamp(ulogin, cid, group);
        getCmdSteps().groupsSteps().postSaveTextAdGroups(params);
        return banner.getBannerId();
    }

    public BannersRecord getBannerFromDb(Long bannerId) {
        return newDbSteps().useShardForLogin(loginsCache.getLoginByBid(bannerId))
                .bannersSteps().getBanner(bannerId);
    }
    //----------------------

    // Ретаргетинг

    @Step("Добавление условий ретаргетинга через api, пользователь: {0}")
    public RetargetingCondition addConditionForUser(User user) {
        Long newConditionId = getApiUserSteps().retargetingListsSteps().addDefaultRetargetingLists(user.getLogin());
        return getRetargetingCondition(user, newConditionId);
    }

    @Step("Получение условия ретаргетинга через cmd-степы, пользователь: {0}, id условия: {1}")
    public RetargetingCondition getRetargetingCondition(User user, Long conditionId) {
        RetargetingCondition retCondition = getCmdSteps().retargetingSteps().getShowRetargetingCond(user.getLogin())
                .get(conditionId);
        return retCondition;
    }

    @Step("Добавление условия ретаргетинга {2} в группу {1}")
    public void addRetargetingConditionToGroup(String clientLogin, Long groupId, Long conditionId) {
        getApiUserSteps().audienceTargetsSteps()
                .addWithRetargetingList(clientLogin, groupId, conditionId);
    }

    @Step("Удаляем условие ретаргетинга через web-api степы")
    public void deleteRetargetingCondition(String login, Long conditionId) {
        directWebApiSteps().retargetingSteps().deleteRetargetingCondition(of(conditionId), login);
    }
    //----------------------

    // Ключевые фразы
    @Step("Получение через api первой фразы из баннера {0}")
    public BannerPhraseInfoWeb getFirstPhrase(Long bannerId) {
        return getBannerInfo(bannerId).getPhrases()[0];
    }

    @Step("Получение через api всех фраз баннера {0}")
    public List<BannerPhraseInfoWeb> getBannerPhrases(Long bannedId) {
        return Stream.of(getBannerInfo(bannedId).getPhrases()).collect(toList());
    }

    public void suspendKeywordsById(String login, Long pid) {
        getApiUserSteps().keywordsSteps().keywordsSuspend(login, pid);
    }

    public void resumeKeywordsById(String login, Long pid) {
        getApiUserSteps().keywordsSteps().keywordsResume(login, pid);
    }

    @Step("Изменяем через api состояние фразы {1} у пользователя {0}")
    public void setPhraseActive(String login, long pid, Boolean isActive) {
        if (isActive == null) {
            return;
        }
        if (isActive) {
            resumeKeywordsById(login, pid);
        } else {
            suspendKeywordsById(login, pid);
        }
    }

    private BannerPhraseInfoWeb getPhraseInfoByPhrase(BannerPhraseInfoWeb[] phrases, String phrase) {
        for (BannerPhraseInfoWeb phrase1 : phrases) {
            if (phrase1.getPhrase().equals(phrase)) {
                return phrase1;
            }
        }
        return null;
    }
    //----------------------

    // Фиды

    @Step("Обновление статуса фидов: статус: {0}, логин: {1}")
    public void updateFeedsStatus(FeedsUpdateStatus feedsStatus, String clientId, List<Long> feedIds) {
        AllureUtils.addJsonAttachment("Ид фидов", wrap(feedIds).toString());
        int shard = newDbSteps().shardingSteps().getShardByClientID(Long.parseLong(clientId));
        newDbSteps().useShard(shard).feedsSteps().updateFeedsStatus(feedsStatus, clientId, feedIds);
    }

    @Step("Обновление статуса фида: статус: {0}, логин: {1}")
    public void updateFeedStatus(FeedsUpdateStatus feedsStatus, String clientId, Long feedId) {
        updateFeedsStatus(feedsStatus, clientId, of(feedId));
    }
    //-------------------

    // Проверки
    @Step("Проверяем, что в базе у баннера №{0} есть картинка")
    public void shouldSeeBannerWithImageInDB(Long bannerId) {
        BannerImagesRecord bannerImagesRecord = newDbSteps().useShardForLogin(loginsCache.getLoginByBid(bannerId))
                .imagesSteps().getBannerImagesByBid(bannerId);
        assertThat("у баннера №" + bannerId + " нет картинки", bannerImagesRecord, is(not(nullValue())));
    }

    @Step("Проверяем, что кампания {0} управляется агентством {1}")
    public void shouldSeeCampaignAgencyLogin(int campaignId, String agencyLogin) {
        assertThat("поле agencyLogin не сооветствует ожиданиям",
                getCampaignFakeInfo(campaignId).getAgencyLogin(), equalTo(agencyLogin));
    }

    @Step("Проверяем, что баннер {0} в таблице banners в БД удовлетворяет условию {1}")
    public void shouldSeeBannerFieldsInDb(Long bannerId, Matcher<BannersRecord> matcher) {
        assertThat("поля баннера в БД не соответствуют ожиданиям", getBannerFromDb(bannerId), matcher);
    }

    @Step("Проверем, что фразы в баннерах {0} и {1} совпадают по полям {2}")
    public void shouldSeeBannerPhrasesEqualByFields(Long actualBannerId, Long expectedBannerId, String... byFields) {
        BannerPhraseInfoWeb[] originalPhrases = getBannerInfo(actualBannerId).getPhrases();
        BannerPhraseInfoWeb[] copiedPhrases = getBannerInfo(expectedBannerId).getPhrases();
        for (BannerPhraseInfoWeb originalPhrase : originalPhrases) {
            BeanEqualsAssert.assertThat(getPhraseInfoByPhrase(copiedPhrases, originalPhrase.getPhrase())
                    , beanEquals(originalPhrase)
                            .byFields(byFields));
        }
    }
    //--------------------

    // Utils
    private void applyForIds(BiConsumer<String, Long[]> action, Function<Long, String> mapper, Long... bannerIds) {
        if (!ArrayUtils.isEmpty(bannerIds)) {
            Map<String, List<Long>> loginToIdsMap = new HashMap<>();
            Stream.of(bannerIds).forEach(
                    x -> {
                        String login = mapper.apply(x);
                        if (!loginToIdsMap.containsKey(login)) {
                            loginToIdsMap.put(login, new ArrayList<>());
                        }
                        loginToIdsMap.get(login).add(x);
                    });
            loginToIdsMap.keySet().forEach(x -> action.accept(x, loginToIdsMap.get(x).stream().toArray(Long[]::new)));
        }
    }

    //----------------------

    // for DNA

    public void deleteMinusPhraseLibrariesExceptLast(String login, int leaveLibrariesNumber) {
        List<MinusWordsRecord> minusPhraseLibraries = newDbSteps().useShardForLogin(login).minusWordsSteps()
                .getMinusWordsBiClientId(
                        Long.parseLong(User.get(login).getClientID())
                );
        if (minusPhraseLibraries.size() > leaveLibrariesNumber) {
            int numberOfDeletedRecords = minusPhraseLibraries.size() - leaveLibrariesNumber;
            for (int i = 0; i < numberOfDeletedRecords; i++) {
                newDbSteps().useShardForLogin(login).minusWordsSteps().deleteRecord(
                        minusPhraseLibraries.get(i).getMwId()
                );
            }
        }
    }

    public void deleteCampaignsExceptLast(String login, int leaveCampaignsNumber) {
        deleteCampaignsWithDefendedCampaigns(login, 0, leaveCampaignsNumber, new ArrayList<>());
    }

    public void deleteCampaignsWithDefendedCampaigns(String login, int saveFirstCampsNumber, int saveLastCampsNumber,
            List<Long> savedCampaigns) {
        List<Long> clientCampaigns = getApiUserSteps().campaignSteps().getAllCampaignIds(login);
        Collections.sort(clientCampaigns);
        if (clientCampaigns.size() > saveFirstCampsNumber + saveLastCampsNumber) {
            for (int i = saveFirstCampsNumber; i < clientCampaigns.size() - saveLastCampsNumber; i++) {
                if (!savedCampaigns.isEmpty() && savedCampaigns.contains(clientCampaigns.get(i))) {
                    continue;
                }
                deleteCampaigns(clientCampaigns.get(i));
            }
        }
    }

    public void deleteFeedsExceptFirstAndLast(String login, int saveFirstFeedsNumber, int saveLastFeedsNumber) {
        List<FeedsRecord> feeds = newDbSteps().useShardForLogin(login).feedsSteps()
                .getFeeds(Long.parseLong(User.get(login).getClientID()));
        if (feeds.size() > saveFirstFeedsNumber + saveLastFeedsNumber) {
            for (int i = saveFirstFeedsNumber; i < feeds.size() - saveLastFeedsNumber; i++) {
                newDbSteps().useShardForLogin(login).feedsSteps().deleteFeedById(feeds.get(i).getFeedId());
            }
        }
    }

    //----------------------
}
