package ru.yandex.autotests.direct.web.steps;

import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import org.hamcrest.Matcher;
import org.openqa.selenium.By;
import org.openqa.selenium.Keys;
import org.openqa.selenium.TimeoutException;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.WebElement;
import org.openqa.selenium.internal.WrapsElement;

import ru.yandex.autotests.direct.utils.beans.IBeanWrapper;
import ru.yandex.autotests.direct.web.util.DirectWebError;
import ru.yandex.autotests.direct.web.util.WebDriverUtils;
import ru.yandex.autotests.direct.web.util.helpers.URLParameterHelper;
import ru.yandex.autotests.irt.testutils.allure.AllureUtils;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.allure.webdriver.steps.BaseSteps;

import static java.lang.String.format;
import static java.util.stream.Collectors.toList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static ru.yandex.autotests.direct.web.util.OperationSystemUtils.sleep;
import static ru.yandex.autotests.direct.web.util.matchers.webdriver.HasTitle.hasTitle;
import static ru.yandex.autotests.direct.web.util.matchers.webdriver.HasURL.hasURL;
import static ru.yandex.qatools.htmlelements.matchers.MatcherDecorators.should;
import static ru.yandex.qatools.htmlelements.matchers.MatcherDecorators.timeoutHasExpired;

public class CommonSteps extends BaseSteps {

    private static String mainWindowHandle;

    @Step("Обновляем страницу")
    public void refreshPage() {
        config.getDriver().navigate().refresh();
    }

    public void shouldSeeBrowser(Matcher<WebDriver> matcher) {
        shouldSeePage(matcher, 15);
    }

    public void shouldSeePage(Matcher<WebDriver> matcher) {
        shouldSeePage(matcher, 15);
    }

    @Step("Проверяем, что страница удовлетворяет условию {0}")
    public void shouldSeePage(Matcher<WebDriver> matcher, Integer timeoutInSeconds) {
        assertThat("страница не найдена", config.getDriver(), should(matcher)
                .whileWaitingUntil(timeoutHasExpired(TimeUnit.SECONDS.toMillis(timeoutInSeconds))));
    }

    @Step("Открываем страницу {0}")
    public void activatePage(Matcher<WebDriver> matcher) {
        if (!WebDriverUtils.activatePage(config.getDriver(), matcher)) {
            throw new DirectWebError("Страница не найдена");
        }
    }

    @Step
    public void closePage(Matcher<WebDriver> matcher) {
        if (!WebDriverUtils.activatePage(config.getDriver(), matcher)) {
            throw new DirectWebError("Страница не найдена");
        } else {
            config.getDriver().close();
            config.getDriver().switchTo().window((String) config.getDriver().getWindowHandles().toArray()[0]);
        }
    }

    @Deprecated
    public void setDriverToOperateWithPage(String pageTitleOrURL) {
        WebDriver popup;
        Set windowHandles = config.getDriver().getWindowHandles();
        for (int i = 0; i < 10; i++) {
            for (Object windowHandle : windowHandles.toArray()) {
                popup = config.getDriver().switchTo().window((String) windowHandle);
                if (popup.getTitle().contains(pageTitleOrURL) ||
                        popup.getCurrentUrl().contains(pageTitleOrURL))
                {
                    return;
                }
            }
            try {
                Thread.sleep(1000);
            } catch (InterruptedException e) {
                throw new DirectWebError(e);
            }
        }
        config.getDriver().quit();
        throw new DirectWebError("Страница с идентификатором: '" + pageTitleOrURL + "' не найдена");
    }

    @Step("Ожидаем открытия страницы с заголовком {0} в течение {1}")
    public void waitForPageTitle(Matcher<String> matcher, long timeoutInSeconds) {
        assertThat("не открылась страница", config.getDriver(),
                should(hasTitle(matcher))
                        .whileWaitingUntil(timeoutHasExpired(TimeUnit.SECONDS.toMillis(timeoutInSeconds)))
        );
    }

    public void waitForPageTitle(Matcher<String> matcher) {
        waitForPageTitle(matcher, 10);
    }

    //оборачиваем waitForPageTitle для выброса TimeoutException, чтобы сработала retryRule при падении
    public void waitForPageTitleWithTimeOutException(org.hamcrest.Matcher<java.lang.String> matcher,
            long timeoutInSeconds)
    {
        try {
            waitForPageTitle(matcher, timeoutInSeconds);
        } catch (Error e) {
            throw new TimeoutException("Превышено время ожидания открытия страницы" + matcher.toString());
        }
    }

    @Step("Проверяем, что тело страницы удовлетворяет {0}, ожидая 3 миллисекунды")
    public void shouldSeePageBody(Matcher matcher) {
        assertThat(config.getDriver().findElement(By.tagName("body")),
                should(matcher).whileWaitingUntil(timeoutHasExpired(TimeUnit.SECONDS.toMillis(3))));
    }

    @Step("Проверяем, что тело страницы удовлетворяет {0}, ожидая {1} миллисекунд")
    public void shouldSeePageBody(Matcher matcher, long timeoutInSeconds) {
        assertThat(config.getDriver().findElement(By.tagName("body")),
                should(matcher).whileWaitingUntil(timeoutHasExpired(TimeUnit.SECONDS.toMillis(timeoutInSeconds))));
    }

    @Step("Проверяем, что входные данные соответствуют: {1}")
    public void shouldSeeBean(IBeanWrapper testData, Matcher matcher) {
        AllureUtils.addJsonAttachment("Проверяемые данные", testData.toString());
        assertThat("данные не соответствуют ожиданиям", testData.getBean(), matcher);
    }

    @Step
    public Long getCampaignIDFromURL() {
        return Long.valueOf(getParameterFromURL("cid"));
    }

    @Step
    public Long[] getBannerIdFromURL() {
        List<String> ids = Arrays.asList(getParameterFromURL("bids").split(","));
        return ids.stream()
                .map(Long::valueOf)
                .collect(toList())
                .toArray(new Long[ids.size()]);
    }

    @Step
    public void shouldSeeCampaignIDFromURL(Matcher matcher) {
        assertThat(getCampaignIDFromURL(), matcher);
    }

    @Step
    public void removeFocusFromActiveElement() {
        config.getDriver().switchTo().activeElement().sendKeys(Keys.TAB);
    }

    private String getParameterFromURL(String paramName) {
        assertThat("в url не найден параметр " + paramName, config.getDriver(),
                should(hasURL(containsString(paramName + "=")))
                        .whileWaitingUntil(timeoutHasExpired(TimeUnit.SECONDS.toMillis(30)))
        );
        return URLParameterHelper.getUrlParameterByName(config.getDriver().getCurrentUrl(), paramName);
    }

    public static <T extends WebElement> void waitForElement(T element, int seconds, Matcher<T> matcher) {
        assertThat(element, should(matcher).whileWaitingUntil(timeoutHasExpired(TimeUnit.SECONDS.toMillis(seconds))));
    }

    public static <T extends WrapsElement> void waitForElement(T element, int seconds, Matcher<T> matcher) {
        assertThat(element, should(matcher).whileWaitingUntil(timeoutHasExpired(TimeUnit.SECONDS.toMillis(seconds))));
    }

    public static <T> void waitForActionResult(Supplier<? extends T> resultSupplier, int durationInSeconds,
            Matcher<T> matcher)
    {
        long startTime = System.nanoTime();
        long nanoDuration = TimeUnit.SECONDS.toNanos(durationInSeconds);
        T result;
        do {
            result = resultSupplier.get();
            if (matcher.matches(result)) {
                return;
            }
            sleep(1000);
        } while (System.nanoTime() - startTime < nanoDuration);

        assertThat(format("условие выполнено в течении %d секунд", durationInSeconds), result, matcher);
    }
}