package ru.yandex.autotests.direct.web.steps.account;


import java.util.List;

import org.hamcrest.Matcher;
import org.openqa.selenium.WebElement;

import ru.yandex.autotests.direct.utils.beans.IBeanWrapper;
import ru.yandex.autotests.direct.utils.strategy.objects.StrategyInfoWeb;
import ru.yandex.autotests.direct.web.helpers.FinanceApiHelper;
import ru.yandex.autotests.direct.web.objects.transfer.PayCampElement;
import ru.yandex.autotests.direct.web.objects.transfer.PaymentMoney;
import ru.yandex.autotests.direct.web.pages.account.ClientWalletPage;
import ru.yandex.autotests.direct.web.pages.account.PayMethodsEnum;
import ru.yandex.autotests.direct.web.pages.clients.WalletLimitEditInfoWeb;
import ru.yandex.autotests.direct.web.pages.payment.PayWithCashBalancePage;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.allure.webdriver.pages.BasePageObject;
import ru.yandex.qatools.allure.webdriver.steps.BaseSteps;
import ru.yandex.qatools.allure.webdriver.steps.WebDriverSteps;
import ru.yandex.qatools.htmlelements.matchers.WebElementMatchers;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.not;
import static ru.yandex.autotests.direct.web.steps.CommonSteps.waitForElement;
import static ru.yandex.autotests.direct.web.util.matchers.webdriver.HasTitle.hasTitle;

public class AccountSteps extends BaseSteps {
    private PayWithCashBalancePage onPayWithCashBalancePage() {
        return BasePageObject.getInstance(PayWithCashBalancePage.class, config);
    }

    private ClientWalletPage onAccountPage() {
        return BasePageObject.getInstance(ClientWalletPage.class, config);
    }

    @Step("Проверяем предупреждение о повторном подключении общего счета")
    public void shouldSeeAccountFrequencyWarning(Matcher<WebElement> matcher) {
        onAccountPage().checkAccountFrequencyWarning(matcher);
    }

    @Step("Производим оплату на странице общего счета")
    public void pay(float sum) {
        onAccountPage().pay(sum);
    }

    @Step("Производим оплату на странице общего счета через новую форму")
    public void payWithNewForm(float sum, PayMethodsEnum paymentType) {
        onAccountPage().payWithNewForm(sum, paymentType);
    }

    @Step("Получаем код оплаты наличными")
    public String getCashPaymentCode() {
        return onAccountPage().getCashPaymentCode();
    }

    @Step("Производим оплату наличными на странице общего счета")
    public void payWithCash(Long campaignId) {
        payWithCash();
        getInstance(WebDriverSteps.class, config).activatePage(hasTitle(containsString("Яндекс.Баланс")));
        isPaymentCodeDisplayed();
        shouldSeePaymentCodeFor(campaignId);
    }

    @Step("Производим оплату наличными на странице общего счета")
    public void payWithCash() {
        onAccountPage().payWithCash();
    }

    @Step("Проверяем наличие кода оплаты при оплате наличными со страницы общего счета")
    public void isPaymentCodeDisplayed() {
        onPayWithCashBalancePage().isPaymentCodeDisplayed();
    }

    @Step("Проверяем код оплаты при оплате наличными со страницы общего счета")
    public void shouldSeePaymentCodeFor(Long campaignId) {
        onPayWithCashBalancePage().shouldSeePaymentCodeFor(campaignId);
    }

    @Step("Проверяем кнопку включения общего счета")
    public void shouldSeeEnableButton(Matcher<WebElement> matcher) {
        onAccountPage().checkEnableButton(matcher);
    }

    @Step("Проверяем кнопку отключения общего счета")
    public void shouldSeeDisableButton(Matcher<WebElement> matcher) {
        onAccountPage().checkDisableButton(matcher);
    }

    @Step("Включаем общий счет")
    public void enableAccount() {
        onAccountPage().clickEnableAccount();
    }

    @Step("Отключаем общий счет с распределением поровну")
    public void disableAccountShareEqual(List<Long> campaigns) {
        onAccountPage().openDisablePopup();
        onAccountPage().acceptAlertIfPresent();
        onAccountPage().disableAccountShareEqual(campaigns);
    }

    @Step("Отключаем общий счет с распределением поровну")
    public void disableAccountShareEqual() {
        onAccountPage().openDisablePopup();
        onAccountPage().acceptAlertIfPresent();
        onAccountPage().disableAccountShareEqual();
    }

    @Step("Отключаем общий счет с пропорциональным распределением")
    public void disableAccountShareProportion(List<Long> campaigns) {
        onAccountPage().openDisablePopup();
        onAccountPage().acceptAlertIfPresent();
        onAccountPage().disableAccountShareProportion(campaigns);
    }

    @Step("Отключаем общий счет с пропорциональным распределением")
    public void disableAccountShareProportion() {
        onAccountPage().openDisablePopup();
        onAccountPage().acceptAlertIfPresent();
        onAccountPage().disableAccountShareProportion();
    }

    @Step("Отключаем общий счет")
    public void disableAccount(List<PayCampElement> campElementList) {
        onAccountPage().openDisablePopup();
        onAccountPage().acceptAlertIfPresent();
        onAccountPage().disableAccount(campElementList);
    }

    @Step("Отключаем общий счет")
    public void disableAccount() {
        onAccountPage().openDisablePopup();
        onAccountPage().acceptAlertIfPresent();
    }

    @Step("Проверяем сумму на общем счету с номером {0}")
    public void shouldSeeAccountAmount(Long accountID, Matcher matcher) {
        assertThat("не совпала сумма на общем счете"
                , new FinanceApiHelper().getAccountAmount(accountID)
                , matcher);
    }

    @Step("Проверяем сумму на агентском общем счету с номером {0}")
    public void shouldSeeAccountAmount(User agency, User client, Matcher matcher) {
        assertThat("не совпала сумма на общем счете"
                , new FinanceApiHelper().getAccountAmount(client, agency)
                , matcher);
    }

    @Step("Проверяем кнопку отключения общего счета")
    public void disableButtonStateShould(Matcher matcher) {
        onAccountPage().getDisableAccountPopupBlock().checkDisableAccountButton(matcher);
    }

    @Step("Открываем попап отключения общего счета")
    public void openDisableAccountPopup() {
        onAccountPage().openDisablePopup();
    }

    @Step("Заполняем форму отключения общего счета")
    public void fillDisableForm(PaymentMoney paymentMoney) {
        onAccountPage().fillDisableForm(paymentMoney);
    }

    @Step("Проверяем кнопку отключения общего счета на попапе отключения общего счета")
    public void shouldSeeDisableButtonInDisableAccountPopup(Matcher<WebElement> matcher) {
        onAccountPage().getDisableAccountPopupBlock().checkDisableAccountButton(matcher);
    }

    @Step("Проверяем попап отключения общего счета")
    public void shouldSeeDisableAccountPopup(Matcher<WebElement> matcher) {
        assertThat("неправильное состояние попапа отключения общего счета"
                , onAccountPage().getDisableAccountPopupBlock(), matcher);
    }

    @Step("Проверяем текст диалога")
    public void shouldSeeAccountDialogMessage(Matcher<String> matcher) {
        onAccountPage().getDialog().checkMessage(matcher);
    }

    @Step("Открыть попап дневного бюджета")
    public void openWalletDayBudgetPopup() {
        onAccountPage().openWalletDayBudgetPopup();
    }

    @Step("Установливаем дневной бюджет на ОС и проверяем закрытие попапа")
    public void fillWalletDayBudgetAndExpectSuccess(IBeanWrapper<StrategyInfoWeb> strategyInfoWebWrapped) {
        onAccountPage().fillWalletDayBudgetParameters(strategyInfoWebWrapped.getBean());
        waitForElement(onAccountPage().getDayBudgetPopupBlock().getWrappedElement(), 2,
                not(WebElementMatchers.isDisplayed()));
    }

    @Step("Установливаем дневной бюджет на ОС и ожидаем появление алерта")
    public void fillWalletDayBudgetSaveAndWaitAlert(IBeanWrapper<StrategyInfoWeb> strategyInfoWebWrapped) {
        onAccountPage().fillWalletDayBudgetParameters(strategyInfoWebWrapped.getBean());
        waitForElement(onAccountPage().getDayBudgetPopupBlock().getAlertTextElement(), 2,
                WebElementMatchers.isDisplayed());
    }

    @Step("Проверяем значение Дневного бюджета на странице ОС")
    public void shouldSeeWalletDayBudget(Double sumDayBudget) {
        onAccountPage().checkWalletDayBudget(equalTo(sumDayBudget));
    }

    @Step("Открыть попап порога отключения")
    public void openWalletLimitEditPopup() {
        onAccountPage().openWalletLimitEditPopup();
        waitForElement(onAccountPage().onWalletLimitEditPopup().getWrappedElement(), 5,
                WebElementMatchers.isDisplayed());
    }

    @Step("Редактируем порог отключения")
    public void fillWalletLimitEditParameters(WalletLimitEditInfoWeb autooverdraftParam) {
        onAccountPage().fillWalletLimitEditParameters(autooverdraftParam);
    }

    @Step("Подключаем порог")
    public void switchOnWalletLimitEdit() {
        onAccountPage().switchOnWalletLimitEdit();
        waitForElement(onAccountPage().onWalletLimitEditPopup().getWrappedElement(), 5,
                not(WebElementMatchers.isDisplayed()));
    }

    @Step("Отключаем порог")
    public void switchOffWalletLimitEdit() {
        onAccountPage().switchOffWalletLimitEdit();
        waitForElement(onAccountPage().onWalletLimitEditPopup().getWrappedElement(), 5,
                not(WebElementMatchers.isDisplayed()));
    }

    @Step("Проверяем подключенный порог")
    public void shouldSeeSwitchOnWalletLimitEdit(Matcher<WebElement> matcher) {
        onAccountPage().checkTextWalletLimitEdit(matcher);
    }

    @Step("Получаем ожидаемые параметры порога отключения")
    public WalletLimitEditInfoWeb getUncheckedLimitEditParameters() {
        return onAccountPage().getUncheckedLimitEditParameters(false);
    }

    @Step("Получаем актуальные параметры порога отключения")
    public WalletLimitEditInfoWeb getCheckedLimitEditParameters() {
        return onAccountPage().getCheckedLimitEditParameters(true);
    }
}
