package ru.yandex.autotests.direct.web.steps.campaigns;

import java.util.List;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.ArrayUtils;
import org.hamcrest.Matcher;
import org.openqa.selenium.Keys;
import org.openqa.selenium.WebElement;
import org.openqa.selenium.internal.WrapsElement;

import ru.yandex.autotests.direct.cmd.data.sort.SortOrder;
import ru.yandex.autotests.direct.utils.beans.IBeanWrapper;
import ru.yandex.autotests.direct.utils.strategy.objects.StrategyInfoWeb;
import ru.yandex.autotests.direct.utils.textresource.ITextResource;
import ru.yandex.autotests.direct.web.data.showcampaigns.ActionLinks;
import ru.yandex.autotests.direct.web.data.showcampaigns.ShowCampsColumns;
import ru.yandex.autotests.direct.web.data.showcampaigns.SortParameters;
import ru.yandex.autotests.direct.web.data.textresources.ShowCampaignsResource;
import ru.yandex.autotests.direct.web.pages.DirectCommonPage;
import ru.yandex.autotests.direct.web.pages.account.PayMethodsEnum;
import ru.yandex.autotests.direct.web.pages.campaigns.ShowCampsPage;
import ru.yandex.autotests.direct.web.pages.campaigns.ShowCampsPage.TabItem;
import ru.yandex.autotests.direct.web.pages.campaigns.blocks.CampaignRow;
import ru.yandex.autotests.direct.web.steps.AlertSteps;
import ru.yandex.autotests.direct.web.steps.campaigns.priceconstructor.PriceConstructorBemSteps;
import ru.yandex.autotests.direct.web.util.DirectWebError;
import ru.yandex.autotests.direct.web.util.WebElementsActions;
import ru.yandex.autotests.direct.web.util.helpers.URLParameterHelper;
import ru.yandex.autotests.direct.web.util.matchers.IsListSortedByParameterMatcher;
import ru.yandex.autotests.irt.testutils.allure.AllureUtils;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.allure.webdriver.annotations.WithoutScreenshot;
import ru.yandex.qatools.allure.webdriver.steps.BaseSteps;
import ru.yandex.qatools.htmlelements.element.CheckBox;
import ru.yandex.qatools.htmlelements.element.Link;
import ru.yandex.qatools.htmlelements.element.TypifiedElement;
import ru.yandex.qatools.htmlelements.matchers.WebElementMatchers;

import static ch.lambdaj.Lambda.extract;
import static ch.lambdaj.Lambda.on;
import static java.util.Objects.isNull;
import static java.util.stream.Collectors.toList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasItems;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.nullValue;
import static ru.yandex.autotests.direct.utils.beans.BeanWrapper.wrap;
import static ru.yandex.autotests.direct.web.steps.CommonSteps.waitForElement;
import static ru.yandex.autotests.direct.web.util.OperationSystemUtils.sleep;
import static ru.yandex.autotests.direct.web.util.matchers.IsAnyListItemContainsMatcher.anyItemContains;
import static ru.yandex.qatools.allure.webdriver.pages.BasePageObject.withParameters;
import static ru.yandex.qatools.htmlelements.matchers.MatcherDecorators.should;
import static ru.yandex.qatools.htmlelements.matchers.MatcherDecorators.timeoutHasExpired;
import static ru.yandex.qatools.htmlelements.matchers.WebElementMatchers.isDisplayed;

/**
 * @author Roman Kuhta (kuhtich@yandex-team.ru)
 */
public class ShowCampsSteps extends BaseSteps {
    private static final String ACTIVE_TAB_CLASS_VALUE = "tab_current_yes";
    private static final String DASH_SYMBOL = "-";
    private static final String ATTR_CLASS = "class";
    private static final String ATTR_HREF = "href";

    public AlertSteps alert() {
        return BaseSteps.getInstance(AlertSteps.class, config);
    }

    public PriceConstructorBemSteps onPriceConstructor() {
        PriceConstructorBemSteps steps = getInstance(PriceConstructorBemSteps.class, config, this);
        steps.forPage(onShowCampsPage());
        return steps;
    }

    @Step("Ищем кампанию {0}")
    private CampaignRow searchCampaignByCampId(String campId) {
        List<CampaignRow> campaignList = onShowCampsPage().getCampaignsRows();
        for (CampaignRow campaign : campaignList) {
            if (campaign.getCampaignNumberValue().equalsIgnoreCase(campId)) {
                return campaign;
            }
        }
        throw new IllegalStateException("Кампания с ID = " + campId + " не найдена в списке");
    }

    public ShowCampsPage onShowCampsPage() {
        return ShowCampsPage.getInstance(ShowCampsPage.class, config);
    }

    public DirectCommonPage onDirectCommonPage() {
        return DirectCommonPage.getInstance(DirectCommonPage.class, config);
    }

    @Step("Выбираем таб {0}")
    public void selectTab(ITextResource tabName) {
        TabItem tabItem = onShowCampsPage().getTab(tabName.toString());
        if (!tabItem.getAttribute(ATTR_CLASS).contains(ACTIVE_TAB_CLASS_VALUE)) {
            tabItem.getTabLink().click();
        }
        assertThat(onShowCampsPage().getTab(tabName.toString()).getAttribute(ATTR_CLASS),
                should(containsString(ACTIVE_TAB_CLASS_VALUE))
                        .whileWaitingUntil(timeoutHasExpired((5000)))
        );
    }

    @Step("Таб {0} должен {1}")
    public void shouldSeeTab(String tabName, Matcher<WebElement> tabState) {
        if (onShowCampsPage().getTabList().size() > 0) {
            assertThat("таб " + tabName + " неактивен",
                    onShowCampsPage().getTab(tabName),
                    tabState);
        }
    }

    @Step("Включаем общий счет")
    public void clickWalletEnableLink() {
        onShowCampsPage().getWalletEnableLink().click();
    }

    @Step("Выставляем цену {1} кампании {0}")
    public void setCampaignPaymentValue(Long campaignId, String sum) {
        onShowCampsPage().findCampaignRow(campaignId).setCampaignPayInputValue(sum);
    }

    @Step("Проверяем цену {1} кампании {0}")
    public void shouldSeeCampaignPaymentValue(Long campaignId, Matcher matcher) {
        onShowCampsPage().findCampaignRow(campaignId).checkCampaignPayInput(matcher);
    }

    @Step("Нажимаем оплатить")
    public void clickPayButton() {
        onShowCampsPage().clickPayButton();
    }

    @Step("Нажимаем оплатить яндекс деньгами")
    public void clickYandexMoneyPayButton() {
        onShowCampsPage().clickYandexMoneyPayButton();
    }

    @Step("Проверяем, что ошибка оплаты кампаний {0}")
    public void shouldSeeCampaignsPaymentError(Matcher matcher) {
        onShowCampsPage().checkPaymentError(matcher);
    }

    @Step("Оплачиваем кампанию {0}")
    public void payForCampaign(Long campaignId) {
        Link payForCampaignLink =
                onShowCampsPage().findCampaignRow(campaignId).getLink(ActionLinks.PAY);
        assertThat("не найдена ссылка на оплату кампании с номером " + campaignId,
                payForCampaignLink.getWrappedElement(), isDisplayed());
        payForCampaignLink.click();
    }

    @Step("Открываем редактирование кампании {0}")
    public void editCampaignParameters(Long campaignId) {
        Link editCampaignParametersLink =
                onShowCampsPage().findCampaignRow(campaignId).getLink(ActionLinks.PARAMETERS);
        assertThat("не найдена ссылка на параметры кампании с номером " + campaignId,
                editCampaignParametersLink.getWrappedElement(), isDisplayed());
        editCampaignParametersLink.click();
    }

    @Step("Проверяем список кампаний")
    public void shouldSeeCampaignsList(Matcher matcher) {
        AllureUtils.addTextAttachment("Параметры кампаний", matcher.toString());
        onShowCampsPage().shouldSeeCampaignsList(matcher);
    }

    @Step("Удаляем кампанию {0}")
    public void deleteCampaign(Long campaignId) {
        assertThat("iD кампании - null", campaignId, not(nullValue()));
        onShowCampsPage().findCampaignRow(campaignId).clickDeleteCampaignLink();
        alert().accept();
    }

    @Step("Проверяем сумму для колонки {0}")
    public void shouldSeeCorrectTotalFor(ShowCampsColumns column) {
        float valueSum = countValuesSum(column);
        float actualValue = getColumnTotal(column);
        assertThat("неверная итоговая сумма для колонки " + column.getColumnName(), actualValue, equalTo(valueSum));
    }

    @Step("Получаем сумму для колонки {0}")
    private float getColumnTotal(ShowCampsColumns column) {
        switch (column) {
            case TOTAL:
                return Float.valueOf(onShowCampsPage().getTotalRest().getText());
            case CLICKS:
                return Float.valueOf(onShowCampsPage().getTotalClicks().getText());
            default:
                throw new DirectWebError("Для колонки " + column.getColumnName() + " недоступно итоговое значение");
        }
    }

    @Step("Проверяем, что строка кампании {0} {1}")
    public void shouldSeeCampaign(Long campaignID, Matcher matcher) {
        assertThat(onShowCampsPage().getCampaignRow(campaignID), matcher);
    }

    @Step("Проверяем, что остаток на кампании {0} {1}")
    public void shouldSeeCampaignRest(Long campaignId, Matcher matcher) {
        onShowCampsPage().findCampaignRow(campaignId).checkTotalValue(matcher);
    }

    @Step("Проверяем, что остаток на кампании {0} {1} ожидая {2} секунд  {3} раз")
    public void shouldSeeCampaignRestWithTimeoutAndRetries(Long campaignId, Matcher matcher, long timeoutInSeconds,
            int retryTimes)
    {
        for (int retry = 1; retry <= retryTimes; retry++) {
            if (!matcher.matches(Float.valueOf(onShowCampsPage().getCampaignRow(campaignId).getTotalValueText()))) {
                sleep(TimeUnit.SECONDS.toMillis(timeoutInSeconds));
                config.getDriver().navigate().refresh();
            } else {
                return;
            }
        }
        onShowCampsPage().findCampaignRow(campaignId).checkTotalValue(matcher);
    }

    @Step("Проверяем, что таб {0} {1}")
    public void shouldSeeTabLink(String tabName, Matcher<WebElement> tabState) {
        if (onShowCampsPage().getTabList().size() > 0) {
            onShowCampsPage().getTab(tabName).checkTabLink(tabState);
        }
    }

    @Step("Ищем {0}")
    public void inputSearchString(String searchString) {
        onShowCampsPage().setSearchInputValue(searchString);
    }

    @Step
    public void cutStringInSearchFieldBySymbolsCount(int cutLength) {
        int symbolsAmount = onShowCampsPage().getSearchInput().getText().length();
        assertThat("не могу сократить пустую строку", symbolsAmount, not(equalTo(0)));
        for (int i = 0; i < cutLength; i++) {
            onShowCampsPage().getSearchInput().sendKeys(Keys.BACK_SPACE);
        }
    }

    @Step("Очищаем поле поиска")
    public void clearSearchField() {
        onShowCampsPage().getSearchInput().clear();
    }

    @Step("Проверяем, что список кампаний может быть отсортирован по колонке {0} в порядке {1}")
    @WithoutScreenshot
    public void shouldSeeCampaignsSortedByColumn(ShowCampsColumns column, SortOrder sortOrder) {
        sortByColumn(column, sortOrder);
        shouldSeeCampaignListSortedByColumn(column, sortOrder);
    }

    @Step("Проверяем, что список кампаний может быть отсортирован по параметру {0} в порядке {1}")
    @WithoutScreenshot
    public void shouldSeeCampaignsSortedByParameter(SortParameters parameter, SortOrder sortOrder) {
        sortByParameter(parameter, sortOrder);
        shouldSeeCampaignListSortedByParameter(parameter, sortOrder);
    }

    @Step("Проверяем, что тизер первой помощи отображается")
    public void shouldSeeFirstHelpTeaser() {
        onShowCampsPage().checkHelpTeaserBlock();
    }

    @Step("Заказываем первую помощь")
    public void clickOnFirstHelpRequestButton() {
        onShowCampsPage().clickHelpTeaserOrderServiceButton();
    }

    @Step("Кликаем ссылку {1} в кампании {0}")
    public void clickCampaignActionLink(Long campaignId, ActionLinks actionLink) {
        onShowCampsPage().findCampaignRow(campaignId).getLink(actionLink).click();
    }

    // Checkbox
    @Step("Выбираем кампанию {0}")
    public void selectCampaignCheckbox(Long campaignId) {
        onShowCampsPage().findCampaignRow(campaignId).selectCampaignCheckbox();
    }

    @Step("Снимаем чекбокс кампании {0}")
    public void deselectCampaignCheckbox(Long campaignId) {
        onShowCampsPage().findCampaignRow(campaignId).deselectCampaignCheckbox();
    }

    @Step("Проверяем, что чекбокс кампании {0} {1}")
    public void shouldSeeCampaignCheckbox(Long campaignId, Matcher<CheckBox> checkboxState) {
        onShowCampsPage().findCampaignRow(campaignId).checkCampaignCheckbox(checkboxState);
    }

    // ID and name
    @Step("Проверяем, что название кампании {0} {1}")
    public void shouldSeeCampaignName(Long campaignId, Matcher matcher) {
        onShowCampsPage().findCampaignRow(campaignId).checkCampaignName(matcher);
    }

    @Step("Проверяем, что кампания {0} является мобильной")
    public void checkIsCampaignMobile(Long campaignId) {
        onShowCampsPage().findCampaignRow(campaignId).checkIsCampaignMobile();
    }


    @Step("Проверяем, что список кампаний {0}")
    public void shouldSeeCampaignNamesList(Matcher matcher) {
        List<String> campNameList = extract(onShowCampsPage().getCampaignsRows(),
                on(CampaignRow.class).getCampaignName().getText());
        assertThat("список имен кампаниий отличается от ожидаемого", campNameList, matcher);
    }

    @Step("Проверяем ссылки кампании {0}")
    public void shouldSeeCampaignActionLinks(Long campaignId, List<String> actionsList) {
        AllureUtils.addJsonAttachment("Должны быть видны ссылки", wrap(actionsList).toString());

        List<String> actions = onShowCampsPage().findCampaignRow(campaignId).getLinksList().stream()
                .map(Link::getReference)
                .map(t -> URLParameterHelper.getUrlParameterByName(t, "cmd"))
                .collect(toList());
        assertThat("список действий для кампании '" + campaignId + "' отличается от ожидаемого",
                actions,
                anyItemContains(actionsList));
    }


    // Parameters
    @Step
    public void shouldSeeParametersValue(Long campaignId, Matcher<String> isEmptyMatcher) {
        onShowCampsPage().findCampaignRow(campaignId).checkParametersValue(isEmptyMatcher);
    }


    // State
    @Step
    public void shouldSeeCampaignStatus(Long campaignId, Matcher<String> equalToMatcher) {
        onShowCampsPage().findCampaignRow(campaignId).checkStatusValue(equalToMatcher);
    }


    @Step
    public void shouldSeeLockIconsForCampaign(Long campaignId, Matcher<WebElement> lockPresence) {
        onShowCampsPage().findCampaignRow(campaignId).checkCampaignLockIcon(lockPresence);
    }

    @Step
    public void shouldSeeArchivedCampaigns(Matcher<WebElement> presenceMatcher) {
        for (CampaignRow campaign : onShowCampsPage().getCampaignsRows()) {
            campaign.checkCampaignLockIcon(presenceMatcher);
        }
    }

    // Start Date
    @Step
    public void shouldSeeCampaignStartDate(Long campaignId, String startDate) {
        onShowCampsPage().findCampaignRow(campaignId).checkStartValue(startDate);
    }

    // Clicks
    @Step
    public void shouldSeeClicksValueInCorrectFormat(Long campaignId) {
        onShowCampsPage().findCampaignRow(campaignId).checkClicksValue();
    }

    // Money Left
    @Step
    public void shouldSeeMoneyLeftValueInCorrectFormat(Long campaignId) {
        onShowCampsPage().findCampaignRow(campaignId).checkTotalValue();
    }

    // Money Sum
    @Step
    public void shouldSeeMoneySumValueInCorrectFormat(Long campaignId) {
        onShowCampsPage().findCampaignRow(campaignId).checkMoneySumValue();
    }

    // CampList
    @Step
    public void shouldSeeCampaignTable(Matcher matcher) {
        assertThat("список кампаний верный", getCampaignIdsList(), matcher);
    }

    @Step
    public float countValuesSum(ShowCampsColumns column) {
        float totalSum = 0;
        String columnValue;
        List<CampaignRow> campaignRows = onShowCampsPage().getCampaignsRows();
        for (CampaignRow campaignRow : campaignRows) {
            columnValue = campaignRow.getColumnValue(column);
            if (!columnValue.equalsIgnoreCase(DASH_SYMBOL)) {
                totalSum += Float.valueOf(columnValue);
            }
        }
        return totalSum;
    }

    @Step
    public void shouldSeeCampList(Matcher matcher) {
        assertThat("состояние списка кампаний не отличается от ожидаемого", getCampaignIdsList(), matcher);
    }

    @Step
    public void shouldSeeCampaignListSortedByColumn(ShowCampsColumns column, SortOrder sortOrder) {
        SortedMap<Object, String> columnValuesMap = new TreeMap<>();
        List<CampaignRow> campaignRows = onShowCampsPage().getCampaignsRows();
        if (column.equals(ShowCampsColumns.TOTAL)) {
            for (CampaignRow campaignRow : campaignRows) {
                columnValuesMap
                        .put(Float.valueOf(campaignRow.getColumnValue(column)), campaignRow.getCampaignNumberValue());
            }
        } else {
            for (CampaignRow campaignRow : campaignRows) {
                columnValuesMap.put(campaignRow.getColumnValue(column), campaignRow.getCampaignNumberValue());
            }
        }
        Object[] expectedArray = columnValuesMap.values().toArray();
        if (sortOrder.equals(SortOrder.DESCENDING)) {
            ArrayUtils.reverse(expectedArray);
        }
        assertThat("сортировка по колонке '" + column + "' некорректная",
                getCampaignIdsList().stream().map(x -> x.toString()).toArray(String[]::new), equalTo(expectedArray));
    }

    @Step
    public void shouldSeeCampaignListSortedByParameter(SortParameters sortParameter, SortOrder order) {
        List<CampaignRow> campaignRows = onShowCampsPage().getCampaignsRows();
        assertThat("список кампаний отстортирован некорректно",
                campaignRows,
                IsListSortedByParameterMatcher.isSortedByParameter(sortParameter, order));
    }

    @Step
    public void shouldSeeActionLinksForCampaignsList(List<Long> campIdList, List<String> actionsList) {
        for (Long campId : campIdList) {
            shouldSeeCampaignActionLinks(campId, actionsList);
        }
    }

    @Step
    public void clickCampaignActionLinkWithAlert(Long campaignId, ActionLinks actionLink, Matcher<String> matcher) {
        clickCampaignActionLink(campaignId, actionLink);
        alert().accept(matcher);
    }

    // All Checkbox
    @Step
    public void selectAllCheckbox() {
        onShowCampsPage().getCampaignsListHeader().getAllCheckbox().select();
    }

    @Step
    public void deselectAllCheckbox() {
        onShowCampsPage().getCampaignsListHeader().deselectAllCheckbox();
    }

    // Parameters
    @Step
    public void selectParameter(SortParameters sortParameter) {
        onShowCampsPage().getCampaignsListHeader().selectParameter(sortParameter);
    }

    @Step
    public void shouldSeeSelectedParameter(SortParameters sortParameter) {
        onShowCampsPage().getCampaignsListHeader().checkSelectedParameter(sortParameter);
    }

    @Step
    public void clickShowButton() {
        onShowCampsPage().getCampaignsListHeader().clickShowButton();
    }

    @Step
    public void clickCancelButton() {
        onShowCampsPage().getCampaignsListHeader().clickCancelButton();
    }

    @Step
    public void clickParametersLink() {
        onShowCampsPage().getCampaignsListHeader().clickParametersLink();
    }

    @Step
    public void clickOnParametersCloseButton() {
        onShowCampsPage().getCampaignsListHeader().clickOnParametersCloseButton();
    }

    @Step("Сортируем список кампаний по колонке {0} в порядке {1}")
    public void sortByColumn(ShowCampsColumns column, SortOrder sortOrder) {
        column.getColumnLink(onShowCampsPage().getCampaignsListHeader().getColumnsList()).click();
        // необходимо заново получить линку после первого клика, со старым её экземпляром не проходит
        WebElement columnLink = column.getColumnLink(onShowCampsPage().getCampaignsListHeader().getColumnsList());
        if (!columnLink.getAttribute(ATTR_HREF).contains(sortOrder.getSortParameter())) {
            columnLink.click();
        }
    }

    // Groups
    @Step("Сортируем список кампаний по параметру {0} в порядке {1}")
    public void sortByParameter(SortParameters sortParameter, SortOrder sortOrder) {
        clickParametersLink();
        selectParameter(sortParameter);
        clickShowButton();
        WebElement columnLink = onShowCampsPage().getCampaignsListHeader().getSelectedSortParameter();
        if (!columnLink.getAttribute(ATTR_HREF).contains(sortOrder.getSortParameter())) {
            columnLink.click();
        }
    }

    @Step("Выполняем массовое действие '{1}' для кампаний {0}")
    public void performCampaignsListAction(List<Long> campaignList, ITextResource action) {
        for (Long campaignId : campaignList) {
            onShowCampsPage().findCampaignRow(campaignId).selectCampaignCheckbox();
        }
        performMassAction(action);
    }

    @Step
    public void performCampaignsListActionWithAlert(List<Long> campIdList, ITextResource action, Matcher matcher) {
        selectTab(ShowCampaignsResource.ALL_CAMPAIGNS);
        for (Long campaignId : campIdList) {
            onShowCampsPage().findCampaignRow(campaignId).selectCampaignCheckbox();
        }
        performMassActionWithAlert(action, matcher);
    }

    /**
     * @return список ID кампаний со страницы кампаний
     */
    private List<Long> getCampaignIdsList() {
        return onShowCampsPage().getCampaignsRows().stream()
                .map(x -> Long.parseLong(x.getCampaignNumberValue()))
                .collect(toList());
    }

    @Step
    public void shouldSeeAllCampaignsTabContains(ITextResource tabName) {
        selectTab(tabName);
        List<Long> tabList = getCampaignIdsList();
        selectTab(ShowCampaignsResource.ALL_CAMPAIGNS);
        assertThat("не все кампании с таба " + tabName + " присутствуют в списке всех кампаний"
                , getCampaignIdsList(), hasItems(tabList.toArray(new Long[tabList.size()])));
    }

    @Step("Устанавливаем значение дневного бюджета на кампанию")
    public void setDayBudgetForCampaign(Long campaignId, IBeanWrapper<StrategyInfoWeb> strategyInfoWebWrapped) {
        onShowCampsPage().fillDayBudgetParametersAndSave(campaignId, strategyInfoWebWrapped.getBean());
    }

    @Step("Проверяем, что блок дневного бюджета {0}")
    public void shouldSeeDayBudgetPopupBlock(Matcher matcher) {
        assertThat("блок дневного бюджета не соответствует ожиданиям",
                onShowCampsPage().getDayBudgetBlock(), should(matcher)
                        .whileWaitingUntil(timeoutHasExpired(TimeUnit.SECONDS.toMillis(5))));
    }

    @Step("Проверяем, что для компании {0} значение дневного бюджета соответствует {1}, " +
            "режим показа дневного бюджета соответствует {2}")
    public void shouldSeeDayBudgetParametersForCampaign(Long campaignId, Matcher amountMatcher,
            Matcher showModeMatcher)
    {
        onShowCampsPage().openDayBudgetBlockForCampaign(campaignId);
        onShowCampsPage().getDayBudgetBlock().checkAmount(amountMatcher);
        onShowCampsPage().getDayBudgetBlock().checkShowMode(showModeMatcher);
    }

    @Step
    public void shouldSeeDayBudgetBlockAlert(Matcher<WebElement> matcher) {
        onShowCampsPage().checkDayBudgetBlockAlert(matcher);
    }

    @Step("Проверяем, что значение дневного бюджета для компании {0} соответствует {1}")
    public void shouldSeeDayBudgetOnCampaignTable(Long campaignId, Matcher matcher) {
        CampaignRow campRow = onShowCampsPage().getCampaign(campaignId);
        if (campRow == null) {
            return;
        }
        assertThat("текст дневного бюджета в таблице кампаний не соответствует ожиданиям", campRow,
                matcher);
    }

    @Step
    public void openRegisteredRepresentatives() {
        onShowCampsPage().clickOnRegisteredRepresentativesLink();
    }

    @Step("Проверяем, что ссылка 'Управление кампаниями с помощью Exel' соответствует {0}")
    public void shouldSeeManageCampaignsInExcelLink(Matcher matcher) {
        onShowCampsPage().checkManageCampaignsInExcelLink(matcher);
    }


    @Step("Открываем страницу 'Управление кампаниями с помощью API'")
    public void openApiSettingsPage() {
        onShowCampsPage().clickOnManageCampaignsWithApiLink();
    }

    @Step("Проверяем, что ссылка 'Управление кампаниями с помощью API' соответствует {0}")
    public void shouldSeeManageCampaignsWithApiLink(Matcher matcher) {
        onShowCampsPage().checkManageCampaignsWithApiLink(matcher);
    }

    @Step
    public void shouldSeeTransferFundsLink(Matcher matcher) {
        onShowCampsPage().checkTransferFundsLink(matcher);
    }

    @Step
    public void shouldSeeStopCampaignLink(Long campaignId, Matcher matcher) {
        onShowCampsPage().checkStopCampaignLink(campaignId, matcher);
    }

    @Step
    public void shouldSeeResumeCampaignLink(Long campaignId, Matcher matcher) {
        onShowCampsPage().checkResumeCampaignLink(campaignId, matcher);
    }

    @Step
    public void shouldSeeSeniorRepresentativeLogin(Matcher matcher) {
        onShowCampsPage().checkSeniorRepresentative(matcher);
    }

    @Step
    public void shouldSeeRegisteredRepresentative(Matcher matcher) {
        onShowCampsPage().shouldSeeRegisteredRepresentative(matcher);
    }

    @Step
    public void shouldSeeMassActionsSelect(Matcher<TypifiedElement> stateMatcher) {
        onShowCampsPage().getMassActions().checkMassActionSelect(stateMatcher);
    }

    @Step
    public void shouldSeeMassActionsPerformButton(Matcher<WebElement> stateMatcher) {
        onShowCampsPage().getMassActions().checkPerformButton(stateMatcher);
    }

    @Step
    public void selectMassActionFromTheList(ITextResource massAction) {
        onShowCampsPage().getMassActions().selectMassAction(massAction.toString());
    }

    @Step
    public void clickPerformButton() {
        onShowCampsPage().getMassActions().clickPerformButton();
    }

    @Step
    public void clickPerformButtonWithAlert(Matcher<String> matcher) {
        onShowCampsPage().getMassActions().clickPerformButton();
        alert().accept(matcher);
    }

    // Groups
    @Step
    public void performMassAction(ITextResource massAction) {
        closeDnaInventation();
        selectMassActionFromTheList(massAction);
        clickPerformButton();
    }

    @Step
    public void performMassActionWithAlert(ITextResource massAction, Matcher<String> matcher) {
        selectMassActionFromTheList(massAction);
        clickPerformButtonWithAlert(matcher);
    }

    @Step("Проверяем, что блок Массовых действий с кампаниями {0} в списке массовых действий")
    public void shouldSeeMassActionSelect(Matcher<Iterable<String>> matcher) {
        onShowCampsPage().getMassActions().checkMassActionsSelect(matcher);
    }

    @Step
    public void shouldSeeManagerForCampaign(Long campaignId, Matcher matcher) {
        onShowCampsPage().checkManagerForCampaign(campaignId, matcher);
    }

    @Step
    public void clickUserSettingsLink() {
        onShowCampsPage().clickUserSettingsLink();
    }

    @Step
    public void clickPaymentsLink() {
        onShowCampsPage().clickPaymentsLink();
    }

    @Step("Открываем конструктор ставок для кампании {0}")
    public void openOfflineConstructorFor(Long campaignId) {
        onShowCampsPage().clickPriceOnCampaign(campaignId);
    }

    @Step("Проверяем, что итоговое значение остатка {0}")
    public void shouldSeeTotalRest(Matcher<WebElement> matcher) {
        onShowCampsPage().checkCampaignsTotalRest(matcher);
    }

    @Step("Проверяем, что Итоговое значение кликов {0}")
    public void shouldSeeTotalCLicks(Matcher<WebElement> matcher) {
        onShowCampsPage().checkCampaignsTotalClicks(matcher);
    }

    @Step("Проверяем, соответствует ли значение валюты {1}")
    public void shouldSeeCurrency(Matcher<String> matcher) {
        onShowCampsPage().checkCurrency(matcher);
    }


    @Step("Кликаем открытие статистики для всех кампаниий")
    public void openStatisticFoAllCamps() {
        onShowCampsPage().clickShowAllCampStat();
    }

    @Step("Кликаем открытие статистики для кампании {0}")
    public void showCampaignStatistics(Long campaignId) {
        onShowCampsPage().clickOnCampaignStatisticsLink(campaignId);
    }

    @Step("Проверяем, что ссылка 'Цена' кампании {0} удовлетворяет условию {1}")
    public void shouldSeePriceOnCampaign(Long campaignId, Matcher<WrapsElement> matcher) {
        onShowCampsPage().checkPriceOnCampaign(campaignId, matcher);
    }

    @Step("Добавляем счётчики метрики")
    public void addMetrikaCountersAndCheckNotVisible(List<Long> metrikaCounters) {
        onShowCampsPage().addMetrikaCountersAndCheckNotVisible(metrikaCounters);
    }

    @Step("Изменяем счётчики метрики")
    public void changeMetrikaCountersSaveAndCheckNotVisible(List<Long> metrikaCounters) {
        onShowCampsPage().changeMetrikaCountersAndCheckNotVisible(metrikaCounters);
    }

    @Step("Удаляем счётчики метрики")
    public void deleteMetrikaCounters() {
        onShowCampsPage().deleteMetrikaCounters();
    }

    @Step("Заполняем счётчики метрики")
    public void fillMetrikaCounters(String metrikaCounters) {
        onShowCampsPage().fillMetrikaCounters(metrikaCounters);
    }

    public void fillMetrikaCounters(List<Long> metrikaCounters) {
        onShowCampsPage().fillMetrikaCounters(metrikaCounters);
    }

    @Step("Проверяем сообщение валидации в попапе добавления счетчика метрики")
    public void shouldSeeMetrikaCountersValidationMessage(Matcher<WebElement> checkMatcher) {
        onShowCampsPage().checkMetrikaValidationMessage(checkMatcher);
    }

    @Step("Устанавливаем значение дневного бюджета на кампанию и проверяем закрытие попапа")
    public void setDayBudgetForCampaignAndExpectSuccess(Long campaignId,
            IBeanWrapper<StrategyInfoWeb> strategyInfoWebWrapped)
    {
        onShowCampsPage().fillDayBudgetParametersAndSave(campaignId, strategyInfoWebWrapped.getBean());
        waitForElement(onShowCampsPage().getDayBudgetBlock().getWrappedElement(), 2,
                not(WebElementMatchers.isDisplayed()));
    }

    @Step("Открываем список видео в попапе видеодополнений")
    public void clickOnSelectVideo() {
        onShowCampsPage().clickOnSelectVideo();
    }

    @Step("Оплачиваем общий счет")
    public void payWallet(Float sum) {
        onShowCampsPage().payWallet(sum);
    }

    @Step("Оплачиваем общий счет в новой форме оплаты")
    public void payWalletWithNewForm(Float sum, PayMethodsEnum payMethod) {
        onShowCampsPage().payWalletWithNewForm(sum, payMethod);
    }

    @Step("Закрытие блока перехода в новый интерфейс")
    public void closeDnaInventation() {
        WebElement closeDnaInventation = onDirectCommonPage().getDnaInventationClose();
        if (WebElementsActions.isExists(closeDnaInventation)) {
            closeDnaInventation.click();
        }
    }
}
